/*
 * $Id: util.c,v 1.34 2001/05/01 14:33:46 kg4ijb Exp $
 *
 * XASTIR, Amateur Station Tracking and Information Reporting
 * Copyright (C) 1999,2000  Frank Giannandrea
 * Copyright (C) 2000,2001  The Xastir Group
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 *
 * Look at the README for more information on the program.
 */

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include <time.h>
#include <unistd.h>
#include <signal.h>
#include <string.h>
#include <stdio.h>
#include <stdlib.h>
#include <sys/ioctl.h>
#include <sys/stat.h>
#include <ctype.h>
#include <math.h>

#include "xastir.h"
#include "util.h"
#include "db.h"
#include "main.h"

#ifdef HAVE_DMALLOC
#include <dmalloc.h>
#endif


// For mutex debugging with Linux threads only
#ifdef MUTEX_DEBUG
#include <asm/errno.h>
extern int pthread_mutexattr_setkind_np(pthread_mutexattr_t *attr, int kind);
#endif


int position_amb_chars;

#define ACCEPT_0N_0E    /* set this to see stations at 0N/0E on the map */


/*************************************************************************/
/* output_lat - format position with position_amb_chars for transmission */
/*************************************************************************/
char *output_lat(char *in_lat, int comp_pos) {
    int i,j;

    if (!comp_pos)
        in_lat[7]=in_lat[8]; /*Shift N/S down for transmission */
    else if (position_amb_chars>0)
        in_lat[7]='0';

    j=0;
    if (position_amb_chars>0 && position_amb_chars<5) {
        for (i=6;i>(6-position_amb_chars-j);i--) {
            if (i==4) {
                i--;
                j=1;
            }
            if (!comp_pos) {
                in_lat[i]=' ';
            } else
                in_lat[i]='0';
        }
    }

    if (!comp_pos) {
        in_lat[8] = '\0';
    }

    return(in_lat);
}


/**************************************************************************/
/* output_long - format position with position_amb_chars for transmission */
/**************************************************************************/
char *output_long(char *in_long, int comp_pos) {
    int i,j;

    if (!comp_pos)
        in_long[8]=in_long[9]; /*shift e/w down for transmission */
    else if (position_amb_chars>0)
        in_long[8]='0';

    j=0;
    if (position_amb_chars>0 && position_amb_chars<5) {
        for (i=7;i>(7-position_amb_chars-j);i--) {
            if (i==5) {
                i--;
                j=1;
            }
            if (!comp_pos) {
                in_long[i]=' ';
            } else
                in_long[i]='0';
        }
    }

    if (!comp_pos)
        in_long[9] = '\0';

    return(in_long);
}



/*********************************************************************/
/* distance_from_my_station - compute distance from my station and   */
/*       course with a given call                                    */
/* return distance and course                                        */
/*********************************************************************/

double distance_from_my_station(char *call_sign, char *course_deg) {
    DataRow *p_station;
    double distance;
    float value;
    long l_lat, l_lon;

    distance = 0.0;
    l_lat = convert_lat_s2l(my_lat);
    l_lon = convert_lon_s2l(my_long);
    p_station = NULL;
    if (search_station_name(&p_station,call_sign,1)) {
        value = (float)calc_distance_course(l_lat,l_lon,p_station->coord_lat,p_station->coord_lon,course_deg); /* knots */
        if (units_english_metric)
            distance = value * 1.15078;
        else
            distance = value * 1.852;

//    printf("DistFromMy: %s %s -> %f\n",temp_lat,temp_long,distance);
    }
    return(distance);
}


/*********************************************************************/
/* PHG range calculation                                             */
/*                                                                   */
/*********************************************************************/
double phg_range(char p, char h, char g) {
    double power, height, gain, range;

    power = (double)( (p-'0')*(p-'0') );    // lclint said: "Assignment of char to double" here
    height = 10.0 * pow(2.0, (double)(h-'0'));
    gain = pow(10.0, (double)(g-'0') / 10.0);
    range = sqrt(2.0 * height * sqrt((power / 10.0) * (gain / 2.0)));
    return(range);
}


/*********************************************************************/
/* get_line - read a line from a file                                */
/*********************************************************************/
char *get_line(FILE *f, char *linedata, int maxline) {
    int size_line;

    strcpy(linedata,"");
    (void)fgets(linedata, 32767, f);
    size_line = (int)strlen(linedata);
    if (size_line > maxline)
        linedata[maxline] = '\0';
    else
        linedata[size_line-1] = '\0';

    return(linedata);
}


char *new_get_line(FILE *f, char *linedata, int maxline) {
    char *ptr;
    int pos;
    static char Buffer[4097];

    pos=0;
    *linedata = '\0';
    while((ptr = strpbrk(Buffer, "\r\n")) == NULL && !feof(f) && strlen(Buffer) <= 4096)
        (void)fread(&Buffer[strlen(Buffer)], sizeof(char), sizeof(Buffer)-strlen(Buffer)-1, f);

    if (ptr)
        *ptr = '\0';

    strncpy(linedata, Buffer, (size_t)maxline);
    if ((pos = (int)strlen(Buffer)) > maxline)
        fprintf(stderr, "Line length overflow: get_line\n");
    else if (pos < maxline)
        linedata[pos] = '\0';

    if (ptr) {
        for (ptr++; *ptr == '\r' || *ptr == '\n'; ptr++);
            strcpy(Buffer, ptr);
    }
    return(linedata);
}


time_t time_from_aprsstring(char *aprs_time) {
    int day, hour, minute;
    char tz[20];
    struct tm *time_now, alert_time;
    time_t timenw;
    long zone;

    (void)time(&timenw);
    time_now = localtime(&timenw);
    zone = timezone - 3600 * (int)(time_now->tm_isdst > 0);
    tz[0] = tz[1] = '\0';
    switch (sscanf(aprs_time, "%2d%2d%2d%19s", &day, &hour, &minute, tz)) {
        case 0:
            day = 0;

        case 1:
            hour = 0;

        case 2:
            minute = 0;

        default:
            break;
    }
    alert_time = *time_now;
    alert_time.tm_sec = 0;
    if (day) {
        alert_time.tm_mday = day;
        if (day < (time_now->tm_mday - 10)) {
            if (++alert_time.tm_mon > 11) {
                alert_time.tm_mon = 0;
                alert_time.tm_year++;
            }
        }
        if (day > (time_now->tm_mday + 10)) {
            if (--alert_time.tm_mon < 0) {
                alert_time.tm_mon = 11;
                alert_time.tm_year--;
            }
        }
        alert_time.tm_min = minute;
        alert_time.tm_hour = hour;
        if ((char)tolower((int)tz[0]) == 'z') {
            alert_time.tm_hour -= zone/3600;
            zone %= 3600;
            if (zone)
                alert_time.tm_min += zone/60;

            if (alert_time.tm_min > 60) {
                alert_time.tm_hour++;
                alert_time.tm_min -= 60;
            }
            if (alert_time.tm_hour > 23) {
                alert_time.tm_hour -= 24;
                alert_time.tm_mday++;
                if (mktime(&alert_time) == -1) {
                    alert_time.tm_mday = 1;
                    alert_time.tm_mon++;
                    if (mktime(&alert_time) == -1) {
                        alert_time.tm_mon = 0;
                        alert_time.tm_year++;
                    }
                }
            } else if (alert_time.tm_hour < 0) {
                alert_time.tm_hour += 24;
                if (--alert_time.tm_mday <= 0) {
                    if (--alert_time.tm_mon < 0) {
                        alert_time.tm_year--;
                        alert_time.tm_mon = 11;
                        alert_time.tm_mday = 31;
                    } else if (alert_time.tm_mon == 3 || alert_time.tm_mon == 5 ||
                                alert_time.tm_mon == 8 || alert_time.tm_mon == 10)
                        alert_time.tm_mday = 30;

                    else if (alert_time.tm_mon == 1)
                        alert_time.tm_mday = (alert_time.tm_year%4 == 0) ? 29: 28;

                    else
                        alert_time.tm_mday = 31;
                }
            }
        }
    }
    else alert_time.tm_year--;
    return(mktime(&alert_time));
}

char *compress_posit(const char *input_lat, const char group, const char *input_lon, const char symbol,
            const int last_course, const int last_speed, const char *phg) {
    static char pos[100];
    char lat[5], lon[5];
    char c, s, t, ext;
    int temp, deg;
    double minutes;

    (void)sscanf(input_lat, "%2d%lf%c", &deg, &minutes, &ext);
    temp = 380926 * (90 - (deg + minutes / 60.0) * (ext=='N'?1:-1));
    lat[3] = (char)(temp%91 + 33); temp /= 91;
    lat[2] = (char)(temp%91 + 33); temp /= 91;
    lat[1] = (char)(temp%91 + 33); temp /= 91;
    lat[0] = (char)(temp    + 33);
    lat[4] = '\0';
    (void)sscanf(input_lon, "%3d%lf%c", &deg, &minutes, &ext);
    temp = 190463 * (180 + (deg + minutes / 60.0) * (ext=='W'?-1:1));
    lon[3] = (char)(temp%91 + 33); temp /= 91;
    lon[2] = (char)(temp%91 + 33); temp /= 91;
    lon[1] = (char)(temp%91 + 33); temp /= 91;
    lon[0] = (char)(temp    + 33);
    lon[4] = '\0';
    c = s = t = ' ';
    if (last_course > 0 || last_speed > 0) {
        c = (char)(last_course/4 + 33);
        s = pow(1.08, (double)last_speed) + 32;
        t = 'C';
    } else if (strlen(phg) >= 6) {
        double power, height, gain, range;
        c = '{';

        //power = (phg[3]-'0') + '\1';        // What is correct here?
        power = (double)((int)(phg[3]-'0') + 1);

        height= 10.0 * pow(2.0,(double)phg[4] - (double)'0');
        gain = pow(10.0,((double)(phg[5]-'0') / 5.0));
        range = sqrt(2.0 * power * gain * height);
        s = 33 + range;
        t = 'C';
    }
    sprintf(pos, "%c%s%s%c%c%c%c", group, lat, lon, symbol, c, s, t);
    return pos;
}


/*
 * See if position is defined
 * 90N 180W (0,0 in internal coordinates) is our undefined position
 * 0N/0E is excluded from trails, could be excluded from map (#define ACCEPT_0N_0E)
 */

int position_defined(long lat, long lon, int strict) {

    if (lat == 0l && lon == 0l)
        return(0);              // undefined location
#ifndef ACCEPT_0N_0E
    if (strict)
#endif
        if (lat == 90*60*60*100l && lon == 180*60*60*100l)      // 0N/0E
            return(0);          // undefined location
    return(1);
}


/* convert latitude from long to string */
void convert_lat_l2s(long lat, char *str, int type) {
    char ns;
    float deg;
    int ideg;
    float min;

    strcpy(str,"");
    deg = (float)(lat - 32400000l) / 360000.0;
    ns = 'S';
    if (deg <= 0) {
        ns = 'N';
        deg = (float)fabs(deg);
    }
    ideg = (int)deg;
    min = (deg - ideg)*60.0;
    switch(type) {
        case(CONVERT_LP_NOSP): /* do low P w/no space */
            sprintf(str,"%02d%05.2f%c",ideg,min,ns);  // 0 maybe 00-0.00N !!??
            break;
        case(CONVERT_LP_NORMAL): /* do low P normal */
            sprintf(str,"%02d %05.2f%c",ideg,min,ns);
            break;
        case(CONVERT_HP_NOSP): /* do HP w/no space */
            sprintf(str,"%02d%06.3f%c",ideg,min,ns);
            break;
        case(CONVERT_HP_NORMAL):
        default: /* do HP normal */
            sprintf(str,"%02d %06.3f%c",ideg,min,ns);
            break;
    }
}



/* convert longitude from long to string */
void convert_lon_l2s(long lon, char *str,int type) {
    char ew;
    float deg;
    int ideg;
    float min;

    strcpy(str,"");
    deg = (float)(lon - 64800000l) / 360000.0;
    ew = 'E';
    if (deg <= 0) {
        ew = 'W';
        deg = (float)fabs(deg);
    }
    ideg = (int)deg;
    min = (deg - ideg)*60.0;
    switch(type) {
        case(CONVERT_LP_NOSP): /* do low P w/nospacel */
            sprintf(str,"%03d %05.2f%c",ideg,min,ew);
            break;
        case(CONVERT_LP_NORMAL): /* do low P normal */
            sprintf(str,"%03d %05.2f%c",ideg,min,ew);
            break;
        case(CONVERT_HP_NOSP): /* do HP w/nospace */
            sprintf(str,"%03d%06.3f%c",ideg,min,ew);
            break;
        case(CONVERT_HP_NORMAL):
        default: /* do HP normal */
            sprintf(str,"%03d %06.3f%c",ideg,min,ew);
            break;
    }
}



/* convert latitude from string to long with 1/100 sec resolution */
long convert_lat_s2l(char *lat) {      /* N=0, Ctr=90, S=180 */
    long centi_sec;
    char n[4];

    centi_sec=0l;
    if (lat[4]=='.' && ((char)toupper((int)lat[7])=='N' || (char)toupper((int)lat[8])=='N'
                    || (char)toupper((int)lat[7])=='S' || (char)toupper((int)lat[8])=='S')) {
        substr(n,lat,2);       // degrees
        centi_sec=atoi(n)*60*60*100;
        substr(n,lat+2,2);     // minutes
        centi_sec += atoi(n)*60*100;
        substr(n,lat+5,3);     // fractional minutes
        if (!isdigit((int)n[2]))
            n[2] = '0';            /* extend low precision */
        centi_sec += atoi(n)*6;
        if ((char)toupper((int)lat[7])=='N' || (char)toupper((int)lat[8])=='N')
            centi_sec = -centi_sec;
        centi_sec += 90*60*60*100;
    }
    return(centi_sec);
}



/* convert longitude from string to long with 1/100 sec resolution */
long convert_lon_s2l(char *lon) {     /* W=0, Ctr=180, E=360 */
    long centi_sec;
    char n[4];

    centi_sec=0l;
    if (lon[5]=='.' && ((char)toupper((int)lon[8])=='W' || (char)toupper((int)lon[9])=='W' || (char)toupper((int)lon[8])=='E' || (char)toupper((int)lon[9])=='E')) {
        substr(n,lon,3);    // degrees 013
        centi_sec=atoi(n)*60*60*100;

        substr(n,lon+3,2);  // minutes 26
        centi_sec += atoi(n)*60*100;
        // 01326.66E  01326.660E
        substr(n,lon+6,3);  // fractional minutes 66E 660
        if (!isdigit((int)n[2]))
            n[2] = '0';            /* extend low precision */
        centi_sec += atoi(n)*6;
        if ((char)toupper((int)lon[8])=='W' || (char)toupper((int)lon[9])=='W')
            centi_sec = -centi_sec;
        centi_sec +=180*60*60*100;;
    }
    return(centi_sec);
}



double convert_lat_l2r(long lat) {
    double ret;
    double degrees;

    degrees = (double)(lat / (100.0*60.0*60.0)) -90.0;
    ret = (-1)*degrees*(M_PI/180.0);
    return(ret);
}



double convert_lon_l2r(long lon) {
    double ret;
    double degrees;

    degrees = (double)(lon / (100.0*60.0*60.0)) -180.0;
    ret = (-1)*degrees*(M_PI/180.0);
    return(ret);
}



double calc_distance_course(long pos1_lat, long pos1_lon, long pos2_lat, long pos2_lon, char *course_deg) {
    float ret;
    double r_lat1,r_lat2;
    double r_lon1,r_lon2;
    double r_d,r_c;

    r_lat1 = convert_lat_l2r(pos1_lat);
    r_lon1 = convert_lon_l2r(pos1_lon);
    r_lat2 = convert_lat_l2r(pos2_lat);
    r_lon2 = convert_lon_l2r(pos2_lon);
    r_d = acos(sin(r_lat1) * sin(r_lat2) + cos(r_lat1) * cos(r_lat2) * cos(r_lon1-r_lon2));

    if (cos(r_lat1) < 0.0000000001) {
        if (r_lat1>0.0)
            r_c=M_PI;
        else
            r_c=0.0;
    } else {
        if (sin((r_lon2-r_lon1))<0.0)
            r_c = acos((sin(r_lat2)-sin(r_lat1)*cos(r_d))/(sin(r_d)*cos(r_lat1)));
        else
            r_c = (2*M_PI) - acos((sin(r_lat2)-sin(r_lat1)*cos(r_d))/(sin(r_d)*cos(r_lat1)));

    }
    sprintf(course_deg,"%.1f",(180.0/M_PI)*r_c);
    ret = r_d*180*60/M_PI;
    return(ret);
}




int filethere(char *fn) {
    FILE *f;
    int ret;

    ret =0;
    f=fopen(fn,"r");
    if (f != NULL) {
        ret=1;
        (void)fclose(f);
    }
    return(ret);
}


int filecreate(char *fn) {
    FILE *f;
    int ret;

    if (! filethere(fn)) {   // If no file

        ret = 0;
        printf("Making user %s file\n", fn);
        f=fopen(fn,"w+");   // Create it
        if (f != NULL) {
            ret=1;          // We were successful
            (void)fclose(f);
        }
        return(ret);        // Couldn't create file for some reason
    }
    return(1);  // File already exists
}



time_t file_time(char *fn) {
    struct stat nfile;

    if(stat(fn,&nfile)==0)
        return((time_t)nfile.st_ctime);

    return(-1);
}



void log_data(char *file, char *line) {
    FILE *f;

    f=fopen(file,"a");
    if (f!=NULL) {
        fprintf(f,"%s\n",line);
        (void)fclose(f);
    }
    else
        printf("Couldn't open file for appending: %s\n", file);
}



time_t sec_now(void) {
    time_t timenw;
    time_t ret;

    ret = time(&timenw);
    return(ret);
}



char *get_time(char *time_here) {
    struct tm *time_now;
    time_t timenw;

    (void)time(&timenw);
    time_now = localtime(&timenw);
    (void)strftime(time_here,MAX_TIME,"%m%d%Y%H%M%S",time_now);
    return(time_here);
}



///////////////////////////////////  Utilities  ////////////////////////////////////////////////////


/*
 *  Check for valid path and change it to TAPR format
 */
 
// DK7IN:  I saw /V or /1  is this defined somewhere ???
int valid_path(char *path) {
    int i,len,hops,j;
    int type,ast;
    char ch;

    len  = (int)strlen(path);
    type = 0;                           // 0: unknown, 1: AEA '>', 2: TAPR2 ',', 3: mixed
    hops = 1;
    ast  = 0;
    for (i=0,j=0; i<len; i++) {
        ch = path[i];
        if (ch == '>' || ch == ',') {   // found digi call separator
            if (ast > 1 || (ast == 1 && i-j > 10) || (ast == 0 && (i == j || i-j > 9)))
                return(0);              // more than one asterisk or wrong call size
            ast = 0;                    // reset asterisk counter
            j = i+1;                    // set to start of next call
            if (ch == ',')
                type |= 0x02;           // set TAPR2 flag
            else
                type |= 0x01;           // set AEA flag
            hops++;                     // count hops
        } else {                        // digi call character or asterisk
            if (ch == '*')
                ast++;                  // count asterisk
            else
                if ((ch <'A' || ch > 'Z') && (ch <'0' || ch > '9') && ch != '-')
                    return(0);          // wrong character in path
        }
    }
    if (ast > 1 || (ast == 1 && i-j > 10) || (ast == 0 && (i == j || i-j > 9)))
        return(0);                      // more than one asterisk or wrong call size

    if (type == 0x03)
        return(0);                      // wrong format, both '>' and ',' in path

    if (hops > 9)                       // [APRS Reference chapter 3]
        return(0);                      // too much hops, destination + 0-8 digipeater addresses

    if (type == 0x01) {
        for (i=0; i<len; i++) {
            if (path[i] == '>')
                path[i] = ',';          // exchange separator character
        }
    }
    return(1);
}


/*
 *  Check for a valid AX.25 call
 *      Valid calls consist of up to 6 uppercase alphanumeric characters
 *      plus optinal SSID (four-bit integer)       [APRS Reference, AX.25 Reference]
 */
int valid_call(char *call) {
    int len, ok;
    int i, del, has_num, has_chr;
    char c;

    has_num = 0;
    has_chr = 0;
    ok      = 1;
    len = (int)strlen(call);

    if (len > 9 || len == 0)
        return(0);                              // wrong size

    del = 0;
    for (i=len-2;ok && i>0 && i>=len-3;i--) {   // search for optional SSID
        if (call[i] =='-')
            del = i;                            // found the delimiter
    }
    if (del) {                                  // we have a SSID, so check it
        if (len-del == 2) {                     // 2 char SSID
            if (call[del+1] < '1' || call[del+1] > '9')                         //  -1 ...  -9
                del = 0;
        } else {                                // 3 char SSID
            if (call[del+1] != '1' || call[del+2] < '0' || call[del+2] > '5')   // -10 ... -15
                del = 0;
        }
    }
    if (del)
        len = del;                              // length of base call
    for (i=0;ok && i<len;i++) {                 // check for uppercase alphanumeric
        c = call[i];
        if (c >= 'A' && c <= 'Z')
            has_chr = 1;                        // we need at least one char
        else if (c >= '0' && c <= '9')
            has_num = 1;                        // we need at least one number
        else
            ok = 0;                             // wrong character in call
    }

//    if (!has_num || !has_chr)                 // with this we also discard NOCALL etc.
    if (!has_chr)                               
        ok = 0;
    ok = (ok && strcmp(call,"NOCALL") != 0);    // check for errors
    ok = (ok && strcmp(call,"ERROR!") != 0);
    ok = (ok && strcmp(call,"WIDE")   != 0);
    ok = (ok && strcmp(call,"RELAY")  != 0);
    ok = (ok && strcmp(call,"MAIL")   != 0);
    return(ok);
}


/*
 *  Check for a valid object name
 */
int valid_object(char *name) {
    int len, i;
    
    // max 9 printable ASCII characters, case sensitive   [APRS Reference]
    len = (int)strlen(name);
    if (len > 9 || len == 0)
        return(0);                      // wrong size

    for (i=0;i<len;i++)
        if (!isprint((int)name[i]))
            return(0);                  // not printable

    return(1);
}


/*
 *  Check for a valid internet name
 *  accept only a few well formed names...
 */
int valid_inet_name(char *name, char *info, char *origin) {
    int len, i, ok;
    char *ptr;
    
    len = (int)strlen(name);

    if (len > 9 || len == 0)            // max 9 printable ASCII characters
        return(0);                      // wrong size

    for (i=0;i<len;i++)
        if (!isprint((int)name[i]))
            return(0);                  // not printable

    if (len > 5 && strncmp(name,"aprsd",5) == 0) {
        strcpy(origin,"INET");
        return(1);                      // aprsdXXXX is ok
    }

    if (len == 6) {                     // check for NWS
        ok = 1;
        for (i=0;i<6;i++)
            if (name[i] <'A' || name[i] > 'Z')  // 6 uppercase characters
                ok = 0;
        ok = ok && (info != NULL);      // check if we can test info
        if (ok) {
            ptr = strstr(info,":NWS-"); // NWS data in info field
            ok = (ptr != NULL);
        }
        if (ok) {
            strcpy(origin,"INET-NWS");
            return(1);                      // weather alerts
        }
    }

    // other acceptable internet names...
    // don't be too generous here, there is a lot of garbage on the internet

    return(0);                          // ignore all other
    
}


/*
 *  Ssubstring function WITH a terminating NULL char, needs a string of at least size+1
 */
void substr(char *dest, char *src, int size) {

    strncpy(dest,src,(size_t)size);
    dest[size] = '\0';
}



char *to_upper(char *data) {
    int i;

    for(i=strlen(data)-1;i>=0;i--)
        if(isalpha((int)data[i]))
            data[i]=toupper((int)data[i]);

    return(data);
}



int is_num_chr(char ch) {
    return((int)(ch >= '0' && ch <= '9'));
}



int is_num_or_sp(char ch) {
    return((int)((ch >= '0' && ch <= '9') || ch == ' '));
}



int is_xnum_or_dash(char *data, int max){
    int i;
    int ok;

    ok=1;
    for(i=0; i<max;i++)
        if(!(isxdigit((int)data[i]) || data[i]=='-')) {
            ok=0;
            break;
        }

    return(ok);
}


// not used
int is_num_stuff(char *data, int max) {
    int i;
    int ok;

    ok=1;
    for(i=0; i<max;i++)
        if(!(isdigit((int)data[i]) || data[i]=='-' || data[i]=='+' || data[i]=='.')) {
            ok=0;
            break;
        }

    return(ok);
}


// not used
int is_num(char *data) {
    int i;
    int ok;

    ok=1;
    for(i=strlen(data)-1;i>=0;i--)
        if(!isdigit((int)data[i])) {
            ok=0;
            break;
        }

    return(ok);
}


//--------------------------------------------------------------------
//Removes all control codes ( < 1Ch ) from a string and set the 8th bit to zero
void removeCtrlCodes(char *cp) {
    int i,j;
    int len = (int)strlen(cp);
    unsigned char *ucp = (unsigned char *)cp;

    for (i=0, j=0; i<=len; i++) {
        ucp[i] &= 0x7f;                 // Clear 8th bit
        if ( (ucp[i] >= (unsigned char)0x1c)           // Check for printable plus the Mic-E codes
                || ((char)ucp[i] == '\0') )   // or terminating 0
            ucp[j++] = ucp[i] ;        // Copy to temp if printable
    }
}


//--------------------------------------------------------------------
//Removes all control codes ( <0x20 or >0x7e ) from a string, including
// CR's, LF's, tab's, etc.
//
void makePrintable(char *cp) {
    int i,j;
    int len = (int)strlen(cp);
    unsigned char *ucp = (unsigned char *)cp;

    for (i=0, j=0; i<=len; i++) {
        ucp[i] &= 0x7f;                 // Clear 8th bit
        if ( ((ucp[i] >= (unsigned char)0x20) && (ucp[i] <= (unsigned char)0x7e))
              || ((char)ucp[i] == '\0') )     // Check for printable or terminating 0
            ucp[j++] = ucp[i] ;        // Copy to (possibly) new location if printable
    }
}



//----------------------------------------------------------------------
// Implements safer mutex locking.  Posix-compatible mutex's will lock
// up a thread if lock's/unlock's aren't in perfect pairs.  They also
// allow one thread to unlock a mutex that was locked by a different
// thread.
// Remember to keep this thread-safe.  Need dynamic storage (no statics)
// and thread-safe calls.
//
// In order to keep track of process ID's in a portable way, we probably
// can't use the process ID stored inside the pthread_mutex_t struct.
// There's no easy way to get at it.  For that reason we'll create
// another struct that contains both the process ID and a pointer to
// the pthread_mutex_t, and pass pointers to those structs around in
// our programs instead.  The new struct is "xastir_mutex".
//


// Function to initialize the new xastir_mutex
// objects before use.
//
void init_critical_section(xastir_mutex *lock) {
#ifdef MUTEX_DEBUG
    pthread_mutexattr_t attr;

    // Note that this stuff is not POSIX, so is considered non-portable.
    // Exists in Linux Threads implementation only?
    (void)pthread_mutexattr_setkind_np(&attr, PTHREAD_MUTEX_ERRORCHECK_NP);
    (void)pthread_mutexattr_init(&attr);
    (void)pthread_mutex_init(&lock->lock, &attr);
#else
    (void)pthread_mutex_init(&lock->lock, NULL); // Set to default POSIX-compatible type
#endif

    lock->threadID = 0;  // No thread has locked it yet
}





// Function which uses xastir_mutex objects to lock a
// critical shared section of code or variables.  Makes
// sure that only one thread can access the critical section
// at a time.  If there are no problems, it returns zero.
//
int begin_critical_section(xastir_mutex *lock, char *text) {
    pthread_t calling_thread;
    int problems;
#ifdef MUTEX_DEBUG
    int ret;
#endif

    problems = 0;

    // Note that /usr/include/bits/pthreadtypes.h has the
    // definition for pthread_mutex_t, and it includes the
    // owner thread number:  _pthread_descr pthread_mutex_t.__m_owner
    // It's probably not portable to use it though.

    // Get thread number we're currently running under
    calling_thread = pthread_self();

    if (pthread_equal( lock->threadID, calling_thread ))
    {
        // We tried to lock something that we already have the lock on.
        printf("%s:This thread already has the lock on this resource\n", text);
        problems++;
    }

    //if (lock->threadID != 0)
    //{
        // We tried to lock something that another thread has the lock on.
        // This is normal operation.  The pthread_mutex_lock call below
        // will put our process to sleep until the mutex is unlocked.
    //}

    // Try to lock it.
#ifdef MUTEX_DEBUG
    // Instead of blocking the thread, we'll loop here until there's
    // no deadlock, printing out status as we go.
    do {
        ret = pthread_mutex_lock(&lock->lock);
        if (ret == EDEADLK) {   // Normal operation.  At least two threads want this lock.
            printf("%s:pthread_mutex_lock(&lock->lock) == EDEADLK\n", text);
            sched_yield();  // Yield the cpu to another thread
        }
        else if (ret == EINVAL) {
            printf("%s:pthread_mutex_lock(&lock->lock) == EINVAL\n", text);
            printf("Forgot to initialize the mutex object...\n");
            problems++;
            sched_yield();  // Yield the cpu to another thread
        }
    } while (ret == EDEADLK);

    if (problems)
        printf("Returning %d to calling thread\n", problems);
#else
    pthread_mutex_lock(&lock->lock);
#endif

    // Note:  This can only be set AFTER we already have the mutex lock.
    // Otherwise there may be contention with other threads for setting
    // this variable.
    //
    lock->threadID = calling_thread;    // Save the threadID that we used
                                        // to lock this resource

    return(problems);
}





// Function which ends the locking of a critical section
// of code.  If there are no problems, it returns zero.
//
int end_critical_section(xastir_mutex *lock, char *text) {
    pthread_t calling_thread;
    int problems;
#ifdef MUTEX_DEBUG
    int ret;
#endif

    problems = 0;

    // Get thread number we're currently running under
    calling_thread = pthread_self();

    if (lock->threadID == 0)
    {
        // We have a problem.  This resource hasn't been locked.
        printf("%s:Trying to unlock a resource that hasn't been locked:%ld\n", text, lock->threadID);
        problems++;
    }

    if (! pthread_equal( lock->threadID, calling_thread ))
    {
        // We have a problem.  We just tried to unlock a mutex which
        // a different thread originally locked.  Not good.
        printf("%s:Trying to unlock a resource that a different thread locked originally: %ld:%ld\n", text, lock->threadID, calling_thread);
        problems++;
    }


    // We need to clear this variable BEFORE we unlock the mutex, 'cuz
    // other threads might be waiting to lock the mutex.
    //
    lock->threadID = 0; // We're done with this lock.


    // Try to unlock it.  Compare the thread identifier we used before to make
    // sure we should.

#ifdef MUTEX_DEBUG
    // EPERM error: We're trying to unlock something that we don't have a lock on.
    ret = pthread_mutex_unlock(&lock->lock);

    if (ret == EPERM) {
        printf("%s:pthread_mutex_unlock(&lock->lock) == EPERM\n", text);
        problems++;
    }
    else if (ret == EINVAL) {
        printf("%s:pthread_mutex_unlock(&lock->lock) == EINVAL\n", text);
        printf("Someone forgot to initialize the mutex object\n");
        problems++;
    }

    if (problems)
        printf("Returning %d to calling thread\n", problems);
#else
    (void)pthread_mutex_unlock(&lock->lock);
#endif

    return(problems);
}


