/*
 *	Xenophilia GTK+ Theme Engine
 * 
 *  xeno_color.c:
 *		Routines for modifying colors and remapping the colors in XPM data
 *		to the colors in the GtkStyle.
 *
 *	Copyright  1999-2002 Johan Hanson <misagon@bahnhof.se>
 *	
 *	This library is free software; you can redistribute it and/or
 *	modify it under the terms of the GNU Library General Public
 *	License as published by the Free Software Foundation; either
 *	version 2 of the License, or (at your option) any later version.
 *	
 *	This library is distributed in the hope that it will be useful,
 *	but WITHOUT ANY WARRANTY; without even the implied warranty of
 *	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *	Library General Public License for more details.
 *	
 *	You should have received a copy of the GNU Library General Public
 *	License along with this library; if not, write to the 
 *	Free Software Foundation, Inc., 59 Temple Place - Suite 330, 
 *	Boston, MA  02111-1307  USA.
 */

#include "xeno_color.h"
#include <math.h>

#define PIXMAP_IMAGE_TYPE(width, height)	GDK_IMAGE_NORMAL
#define MASK_IMAGE_TYPE(width, height)		GDK_IMAGE_NORMAL


/*
 *	Globals
 */

const guchar xeno_dither_table[16][16] = {
	{ 0x00,0xc0,0x30,0xf0,0x0c,0xcc,0x3c,0xfc,0x03,0xc3,0x33,0xf3,0x0f,0xcf,0x3f,0xff },
	{ 0x80,0x40,0xb0,0x70,0x8c,0x4c,0xbc,0x7c,0x83,0x43,0xb3,0x73,0x8f,0x4f,0xbf,0x7f },
	{ 0x20,0xe0,0x10,0xd0,0x2c,0xec,0x1c,0xdc,0x23,0xe3,0x13,0xd3,0x2f,0xef,0x1f,0xdf },
	{ 0xa0,0x60,0x90,0x50,0xac,0x6c,0x9c,0x5c,0xa3,0x63,0x93,0x53,0xaf,0x6f,0x9f,0x5f },
	{ 0x08,0xc8,0x38,0xf8,0x04,0xc4,0x34,0xf4,0x0b,0xcb,0x3b,0xfb,0x07,0xc7,0x37,0xf7 },
	{ 0x88,0x48,0xb8,0x78,0x84,0x44,0xb4,0x74,0x8b,0x4b,0xbb,0x7b,0x87,0x47,0xb7,0x77 },
	{ 0x28,0xe8,0x18,0xd8,0x24,0xe4,0x14,0xd4,0x2b,0xeb,0x1b,0xdb,0x27,0xe7,0x17,0xd7 },
	{ 0xa8,0x68,0x98,0x58,0xa4,0x64,0x94,0x54,0xab,0x6b,0x9b,0x5b,0xa7,0x67,0x97,0x57 },
	{ 0x02,0xc2,0x32,0xf2,0x0e,0xce,0x3e,0xfe,0x01,0xc1,0x31,0xf1,0x0d,0xcd,0x3d,0xfd },
	{ 0x82,0x42,0xb2,0x72,0x8e,0x4e,0xbe,0x7e,0x81,0x41,0xb1,0x71,0x8d,0x4d,0xbd,0x7d },
	{ 0x22,0xe2,0x12,0xd2,0x2e,0xee,0x1e,0xde,0x21,0xe1,0x11,0xd1,0x2d,0xed,0x1d,0xdd },
	{ 0xa2,0x62,0x92,0x52,0xae,0x6e,0x9e,0x5e,0xa1,0x61,0x91,0x51,0xad,0x6d,0x9d,0x5d },
	{ 0x0a,0xca,0x3a,0xfa,0x06,0xc6,0x36,0xf6,0x09,0xc9,0x39,0xf9,0x05,0xc5,0x35,0xf5 },
	{ 0x8a,0x4a,0xba,0x7a,0x86,0x46,0xb6,0x76,0x89,0x49,0xb9,0x79,0x85,0x45,0xb5,0x75 },
	{ 0x2a,0xea,0x1a,0xda,0x26,0xe6,0x16,0xd6,0x29,0xe9,0x19,0xd9,0x25,0xe5,0x15,0xd5 },
	{ 0xaa,0x6a,0x9a,0x5a,0xa6,0x66,0x96,0x56,0xa9,0x69,0x99,0x59,0xa5,0x65,0x95,0x55 }
};

/*
#define GAMMA_LUT_SIZE		32
#define GAMMA_LUT_SIZE_F	32.0

#define _CURVE(lut, subdiv, src, dst) { \
	gfloat t, u; \
	gint   i; \
	i = (gint)(u = floor(t = (src) * (subdiv))); \
	(dst) = LERP(lut[i], lut[i+1], t - u); \
}

#define GAMMA(src, dst)		_CURVE(gamma_lut, GAMMA_LUT_SIZE_F, src, dst)
#define LINEAR(src, dst)	_CURVE(inv_gamma_lut, GAMMA_LUT_SIZE_F, src, dst)

const static gfloat gamma_lut[GAMMA_LUT_SIZE + 1] = {
	0.0F,       0.2069383F, 0.2835781F, 0.3409684F, 0.3886015F, 0.4300852F, 0.4672464F, 0.5011599F,
	0.5325205F, 0.5618074F, 0.5893677F, 0.6154619F, 0.6402916F, 0.6640164F, 0.6867650F, 0.7086435F,
	0.7297400F, 0.7501288F, 0.7698734F, 0.7890282F, 0.8076406F, 0.8257521F, 0.8433989F, 0.8606134F,
	0.8774243F, 0.8938573F, 0.9099355F, 0.9256798F, 0.9411092F, 0.9562408F, 0.9710904F, 0.9856724F,
	1.0F
};

const static gfloat inv_gamma_lut[GAMMA_LUT_SIZE + 1] = {
	0.0F,		0.0004882F, 0.0022435F, 0.0054744F, 0.0103086F, 0.0168424F, 0.0251537F, 0.0353090F,
	0.0473661F, 0.0613767F, 0.0773873F, 0.0954407F, 0.1155763F, 0.1378305F, 0.1622376F, 0.1888299F,
	0.2176376F, 0.2486896F, 0.2820132F, 0.3176347F, 0.3555789F, 0.3958699F, 0.4385306F, 0.4835832F,
	0.5310492F, 0.5809490F, 0.6333028F, 0.6881297F, 0.7454486F, 0.8052776F, 0.8676344F, 0.9325363F,
	1.0F
};
*/
/*
#define GAMMA(src, dst)		((dst) = (src) * (src))
#define LINEAR(src, dst)	((dst) = sqrt(src))
*/

#define GAMMA(src, dst)		((dst) = (src))
#define LINEAR(src, dst)	((dst) = (src))


/*
 *	XenoColor
 */

const XenoColor xeno_color_black = { { 0.0F, 0.0F, 0.0F } };
const XenoColor xeno_color_white = { { 1.0F, 1.0F, 1.0F } };

void
xeno_color_init (XenoColor *color, gfloat red, gfloat green, gfloat blue)
{
	color->v[0] = red;
	color->v[1] = green;
	color->v[2] = blue;
}

#define XENO_COLOR_RED(color)		((color)->v[0])
#define XENO_COLOR_GREEN(color)		((color)->v[1])
#define XENO_COLOR_BLUE(color)		((color)->v[2])

void
xeno_color_to_gdk (const XenoColor *src, GdkColor *dst)
{
	gint red, green, blue;
	
	red   = XENO_COLOR_RED(src)   * 65535.0F;
	green = XENO_COLOR_GREEN(src) * 65535.0F;
	blue  = XENO_COLOR_BLUE(src)  * 65535.0F;
	
	(dst)->red   = CLAMP(red,   0, 0x0ffff);
	(dst)->green = CLAMP(green, 0, 0x0ffff);
	(dst)->blue  = CLAMP(blue,  0, 0x0ffff);
}

void xeno_color_shade (const XenoColor *src, gfloat k, XenoColor *dst) {
	if (k > 1.0) {
		k -= 1.0;
		dst->v[0] = LERP(src->v[0], 1.0F, k);
		dst->v[1] = LERP(src->v[1], 1.0F, k);
		dst->v[2] = LERP(src->v[2], 1.0F, k);
	} else {
		dst->v[0] = k * src->v[0];
		dst->v[1] = k * src->v[1];
		dst->v[2] = k * src->v[2];
	}
}

XENO_INLINE guint16 xeno_dither_component (gfloat f, gint shift, guchar dither) {
	gint32	value;
	
	shift += 8;
	value = f * ((1 << shift) - 1);
	if ((guchar)value > dither)
		value += 0x0100;
	
	value = value << (16 - shift);
	return CLAMP(value, 0x0000, 0x0ffff);
}

void xeno_color_dither (const XenoColor *src, const GdkVisual *visual, gint x, gint y, GdkColor *dst) {
	guchar dither = xeno_dither_table[x & 0x0f][y & 0x0f];
	dst->red   = xeno_dither_component (src->v[0],   visual->red_prec,   dither);
	dst->green = xeno_dither_component (src->v[1],	visual->green_prec,	dither);
	dst->blue  = xeno_dither_component (src->v[2],	visual->blue_prec,  dither);
}

#if 0
void xeno_color_gamma (const XenoColor *src, XenoColor *dst) {
  #if 1
	GAMMA(src->v[0], dst->v[0]);
	GAMMA(src->v[1], dst->v[1]);
	GAMMA(src->v[2], dst->v[2]);
  #else
	const static gfloat	f = 1.0/2.4;
	const gfloat *sp;
	gfloat *dp;
	gfloat s, d;
	gint i;
	
	sp = (const gfloat *)src;
	dp = (gfloat *)dst;
	
	for (i = 0; i < 3; ++i) {
		s = *(sp++);
		s = CLAMP(s, 0.0, 1.0);
		if (s <= 0.00304) {
			d = s * 12.92F;
		} else {
			d = 1.055 * pow(s, f) - 0.055;
		}
		*(dp++) = d;
	}
  #endif
}

void xeno_color_linear (const XenoColor *src, XenoColor *dst) {
  #if 1
	LINEAR(src->v[0],   dst->v[0]);
	LINEAR(src->v[1], dst->v[1]);
	LINEAR(src->v[2],  dst->v[2]);
  #else
	const static gfloat f = 1.0/12.92;
	const static gfloat g = 1.0/1.055;
	const gfloat *sp;
	gfloat       *dp;
	gfloat       s, d;
	gint         i;

	sp = (const gfloat *)src;
	dp = (gfloat *)dst;
	
	for (i = 0; i < 3; ++i) {
		s = *(sp++);
		s = CLAMP(s, 0.0, 1.0);
		if (s <= 0.03928) {
			d = s * f;
		} else {
			d = pow((s + 0.055F) * g, 2.4);
		}
		*(dp++) = d;
	}
  #endif
}
#endif

/*
 *	XenoPixmap
 */

void
xeno_pixmap_unref (XenoPixmap *pixmap)
{
	g_return_if_fail (pixmap != NULL);
	
	gdk_pixmap_unref ((GdkPixmap *)pixmap);
}

void
xeno_pixmap_get_size	(XenoPixmap	*pixmap,
						 gint		*width_p,
						 gint		*height_p)
{
	g_return_if_fail (pixmap != NULL);
	
	gdk_window_get_size ((GdkPixmap *)pixmap, width_p, height_p);
}

void
xeno_pixmap_mask_unref (XenoPixmapMask *mask)
{
	g_return_if_fail (mask != NULL);
	
	return;
  /*
	gdk_bitmap_unref ((GdkBitmap *)mask);
  */
}

void
xeno_draw_pixmap	(GdkDrawable	*drawable,
					 GdkGC			*gc,
					 GdkRectangle	*area,
					 XenoPixmap		*pixmap,
					 XenoPixmapMask	*mask,
					 gint			src_x,
					 gint			src_y,
					 gint			x,
					 gint			y,
					 gint			width,
					 gint			height)
{
	g_return_if_fail (drawable != NULL);
	g_return_if_fail (gc != NULL);
	g_return_if_fail (pixmap != NULL);
	
	if (area) {
		gint new_x, new_y;
		
		new_x = MAX(x, area->x);
		new_y = MAX(y, area->y);
		width  = MIN(x + width,  area->x + area->width)  - new_x; if (width <= 0) return;
		height = MIN(y + height, area->y + area->height) - new_y; if (height <= 0) return;
		src_x += new_x - x;
		src_y += new_y - y;
		x = new_x;
		y = new_y;
	}
	
	gdk_gc_set_clip_mask (gc, (GdkBitmap *)mask);
	gdk_gc_set_clip_origin (gc, x - src_x, y - src_y);
	gdk_draw_pixmap (drawable, gc, (GdkPixmap *)pixmap, src_x, src_y, x, y, width, height);
	if (mask)
		gdk_gc_set_clip_mask (gc, NULL);
}

GdkPixmap *
xeno_pixmap_get_gdk_pixmap	(XenoPixmap *pixmap)
{
	g_return_val_if_fail (pixmap != NULL, NULL);
	
	return (GdkPixmap *) gdk_pixmap_ref ((GdkPixmap *)pixmap);
}


/*
 *	XenoImageBuffer
 */

XenoImageBuffer	*
xeno_image_buffer_new  (guint16	 width,
						guint16	 height)
{
	XenoImageBuffer	*buffer;
	
	buffer = g_new (XenoImageBuffer, 1);
	if (buffer)
		xeno_image_buffer_init (buffer, width, height);
	
	return buffer;
}						

void
xeno_image_buffer_destroy (XenoImageBuffer *buffer)
{
	g_return_if_fail (buffer != NULL);
	
	xeno_image_buffer_finalize (buffer);
	g_free (buffer);
}

void
xeno_image_buffer_init (XenoImageBuffer	*buffer,
						guint16			width,
						guint16			height)
{
	gint i, size;

	g_return_if_fail (buffer != NULL);
	
	buffer->width = width;
	buffer->height = height;
	
	size = width * height;
	buffer->data = g_new (XenoTexel, size);
	for (i = 0; i < size; ++i)
		xeno_texel_clear (&buffer->data[i]);
}

void
xeno_image_buffer_finalize (XenoImageBuffer *buffer)
{
	g_return_if_fail (buffer != NULL);
	
	g_free (buffer->data);
}

/* Convert image buffer into format for rendering
   To do:
   - dithering
   - prime candidate for MMX/SSE/whatever
*/
static void
xeno_image_buffer_render_truecolor (const XenoImageBuffer	*buffer,
									GdkImage				*image,
									const XenoColor			*bg_color)
{
	XenoColor	linear_bg;
	const XenoTexel	*src;
	GdkVisual	*visual;
	XenoColor	color;
	gfloat		red_factor, green_factor, blue_factor;
	gfloat		alpha;
	guint32		pixel, bg_pixel;
	gint		u, v;

	g_return_if_fail (buffer != NULL);
	g_return_if_fail (buffer->data != NULL);
	g_return_if_fail (bg_color != NULL);
	g_return_if_fail (image != NULL);
	g_return_if_fail (!xeno_theme_pseudocolor);

	visual = xeno_theme_visual;
	
	red_factor	 = visual->red_mask;
	green_factor = visual->green_mask;
	blue_factor	 = visual->blue_mask;
	
	xeno_color_linear (bg_color, &linear_bg);
	bg_pixel = (  (visual->red_mask   & (gint)(bg_color->v[0] * red_factor))
				| (visual->green_mask & (gint)(bg_color->v[1] * green_factor))
				| (visual->blue_mask  & (gint)(bg_color->v[2] * blue_factor)));
	
	src = buffer->data;
	for (v = 0; v < buffer->height; ++v) {
		for (u = 0; u < buffer->width; ++u) {
			pixel = bg_pixel;
			alpha = src->v[3];
			if (alpha > 0.0F) {
				color = *XENO_COLOR(src);
				
				alpha = 1.0F - alpha;
				if (alpha > 0.0F)
					xeno_color_combine (&color, &linear_bg, alpha, &color);
				
				xeno_color_gamma (&color, &color);
				
				pixel = (  (visual->red_mask   & (gint)(XENO_COLOR_RED(&color)   * red_factor))
						 | (visual->green_mask & (gint)(XENO_COLOR_GREEN(&color) * green_factor))
						 | (visual->blue_mask  & (gint)(XENO_COLOR_BLUE(&color)  * blue_factor)));
			}
			++src;
			
			gdk_image_put_pixel (image, u, v, pixel);
		}
	}
}

static void
xeno_image_buffer_render_pseudocolor   (const XenoImageBuffer	*buffer,
										GdkImage				*image,
										const XenoColor			*bg_color)
{
	XenoColor	linear_bg;
	GdkColor	gdk_color;
	GdkVisual	*visual;
	const XenoTexel	*src;
	XenoColor	color;
	gfloat		alpha;
	guint32		pixel, bg_pixel;
	gint		u, v;
	
	g_return_if_fail (buffer != NULL);
	g_return_if_fail (buffer->data != NULL);
	g_return_if_fail (bg_color != NULL);
	g_return_if_fail (image != NULL);
	g_return_if_fail (xeno_theme_pseudocolor);

	visual = xeno_theme_visual;
	xeno_color_linear (bg_color, &linear_bg);
	xeno_color_to_gdk (bg_color, &gdk_color);
	gdk_colormap_alloc_color (xeno_theme_colormap, &gdk_color, FALSE, TRUE);
	bg_pixel = gdk_color.pixel;
	
	src = buffer->data;
	for (v = 0; v < buffer->height; ++v) {
		for (u = 0; u < buffer->width; ++u) {
			pixel = bg_pixel;
			alpha = src->v[3];
			if (alpha > 0.0F) {
				color = *XENO_COLOR(src);
				
				alpha = 1.0F - alpha;
				if (alpha > 0.0F)
					xeno_color_combine (&color, &linear_bg, alpha, &color);
				
				xeno_color_gamma (&color, &color);
				xeno_color_to_gdk (&color, &gdk_color);
				gdk_colormap_alloc_color (xeno_theme_colormap, &gdk_color, FALSE, TRUE);
				pixel = gdk_color.pixel;
			}
			++src;
			
			gdk_image_put_pixel (image, u, v, pixel);
		}
	}
}

XenoPixmap *
xeno_image_buffer_render (const XenoImageBuffer	*buffer,
						  const XenoColor		*bg_color)
{
	GdkPixmap	*pixmap;
	GdkImage	*image;
	GdkGC		*gc;
	
	g_return_val_if_fail (buffer != NULL, NULL);
	g_return_val_if_fail (buffer->data != NULL, NULL);
	g_return_val_if_fail (bg_color != NULL, NULL);
	
	pixmap = gdk_pixmap_new (NULL, buffer->width, buffer->height, xeno_theme_visual->depth);
	if (pixmap) {
		image = gdk_image_new (PIXMAP_IMAGE_TYPE(buffer->width, buffer->height),
							   xeno_theme_visual, buffer->width, buffer->height);
		
		if (image) {
			if (xeno_theme_pseudocolor) {
				xeno_image_buffer_render_pseudocolor (buffer, image, bg_color);
			} else {
				xeno_image_buffer_render_truecolor (buffer, image, bg_color);
			}
			
			gc = gdk_gc_new (pixmap);
			if (gc)
				gdk_draw_image (pixmap, gc, image, 0,0, 0,0, buffer->width, buffer->height);
			
			gdk_image_destroy (image);
			if (gc) {
				gdk_gc_unref (gc);
				return (XenoPixmap *)pixmap; /* succeed */
			}
			/* fail */
		}
		gdk_pixmap_unref (pixmap);
	}
	g_warning ("Failed to render image buffer");
	return NULL;
}

XenoPixmapMask *
xeno_image_buffer_render_mask (const XenoImageBuffer *buffer)
{
	const XenoTexel	*src;
	XenoPixmapMask	*mask;
	guchar	*data, *dst;
	gint	u, v, bytes_per_line;
	
	g_return_val_if_fail (buffer != NULL, NULL);
	g_return_val_if_fail (buffer->data != NULL, NULL);
	
	bytes_per_line = (buffer->width + 7) >> 3;
	data = g_malloc0 (buffer->height * bytes_per_line);
	
	src = buffer->data;
	for (v = 0; v < buffer->height; ++v) {
		dst = &data[v * bytes_per_line];
		for (u = 0; u < buffer->width; ++u) {
			if ((src++)->v[3] > 0.0)
				dst[u >> 3] |= 1 << (u & 0x07);
		}
	}
	
	mask = (XenoPixmapMask *) gdk_bitmap_create_from_data (NULL, data, buffer->width, buffer->height);
	g_free (data);
	
	return mask;
}


/*
 *	XenoImage
 */

void
xeno_image_render	(const XenoImage	*image,
					 XenoImageBuffer	*image_buffer,
					 guint16			x,
					 guint16			y,
					 XenoPenCallback	pen_callback,
					 gpointer			user_data)
{
	const XenoImageLayer *layer;
	XenoColor	color;
	XenoTexel	*dst;
	gint		stride;
	
	g_return_if_fail (image != NULL);
	g_return_if_fail (image_buffer != NULL);
	g_return_if_fail (pen_callback != NULL);
	g_return_if_fail (x + image->width <= image_buffer->width);
	g_return_if_fail (y + image->height <= image_buffer->height);
	
	for (layer = image->layers; layer != &image->layers[image->n_layers]; ++layer) {
		pen_callback (layer->pen, layer->modifier, &color, user_data);
		xeno_color_linear (&color, &color);
		
		dst = &image_buffer->data[(layer->y + y) * image_buffer->width + layer->x + x];
		stride = image_buffer->width - layer->width;
		
		if (xeno_theme_pseudocolor && layer->bitmap) {
			const guchar *src;
			gint	u, v;
			guchar	t;
			
			src = layer->bitmap;
			for (v = 0; v < layer->height; ++v) {
				for (u = 0; u < layer->width; ++u) {
					if ((u & 0x07) == 0)
						t = *(src++);
					
					if ((t & 0x80) != 0) {
						*XENO_COLOR(dst) = color;
						dst->v[3] = 1.0F;
					}
					t <<= 1;
					
					++dst;
				}
				dst += stride;
			}
		} else if (layer->alpha) {
			const gfloat f = 1.0/255.0;
			const guchar *src;
			gfloat	intensity;
			gint	u, v;
			guchar	t;
			
			src = layer->alpha;
			for (v = 0; v < layer->height; ++v) {
				for (u = 0; u < layer->width; ++u) {
					t = *(src++);
					if (t != 0) {
						intensity = t * f;
						xeno_color_combine (XENO_COLOR(dst), &color, intensity, XENO_COLOR(dst));
						dst->v[3] += intensity;
					}
					++dst;
				}
				dst += stride;
			}
		}
	}
}

/*
static void
xeno_image_render_pseudocolor	(const XenoImage	*image,
								 XenoImageBuffer	*image_buffer,
								 guint16			x,
								 guint16			y,
								 XenoPenCallback	pen_callback,
								 gpointer			user_data)
{
	XenoColor		*colors;
	XenoTexel		*dst;
	const guchar	*src;
	gint			u, v, c, stride;

	g_return_if_fail (image->pixels != NULL);
	g_return_if_fail (image->pens != NULL);
	g_return_if_fail (image->n_pens > 0);
	
	colors = alloca (sizeof(XenoColor) * image->n_pens);
	if (!colors)
		return;

	for (c = 0; c < image->n_pens; ++c) {
		pen_callback (image->pens[c].pen, image->pens[c].modifier, &colors[c], user_data);
		xeno_color_linear (&colors[c], &colors[c]);
	}
	
	dst = &image_buffer->data[y * image_buffer->width + x];
	stride = image_buffer->width - image->width;
	
	src = image->pixels;
	for (v = 0; v < image->height; ++v) {
		for (u = 0; u < image->width; ++u) {
			c = *(src++);
			if (c != 0) {
				*XENO_COLOR(dst) = colors[c - 1];
				dst->v[3] = 1.0F;
			}
			++dst;
		}
		dst += stride;
	}
}

void
xeno_image_render	(const XenoImage	*image,
					 XenoImageBuffer	*image_buffer,
					 guint16			x,
					 guint16			y,
					 XenoPenCallback	pen_callback,
					 gpointer			user_data)
{
	g_return_if_fail (image != NULL);
	g_return_if_fail (image_buffer != NULL);
	g_return_if_fail (pen_callback != NULL);
	g_return_if_fail (x + image->width <= image_buffer->width);
	g_return_if_fail (y + image->height <= image_buffer->height);
	
	if (xeno_theme_pseudocolor && image->pixels && image->pens) {
		xeno_image_render_pseudocolor (image, image_buffer, x, y, pen_callback, user_data);
	} else {
		xeno_image_render_truecolor (image, image_buffer, x, y, pen_callback, user_data);
	}
}
*/


/*
 *	Misc
 */

void
xeno_color_from_pixmap (XenoColor	*color,
						GdkPixmap	*pixmap)
{
	XenoColor	sum;
	GdkImage	*image;
	GdkVisual	*visual;
	gint		x, y, width, height, size;
	guint		pixel;
	
	g_return_if_fail (color != NULL);
	g_return_if_fail (pixmap != NULL);
	g_return_if_fail (pixmap != (GdkPixmap *)GDK_PARENT_RELATIVE);
	
	visual = gdk_window_get_visual (pixmap);
	if (visual == NULL)
		visual = xeno_theme_visual;
	
	gdk_window_get_size (pixmap, &width, &height);
	size = width * height;
	
	if ((image = gdk_image_get (pixmap, 0, 0, width, height)) == NULL)
		return;
	
	if (visual->type == GDK_VISUAL_TRUE_COLOR) {
		XenoColor	tmp;
		gfloat		rf, gf, bf;
		gulong		white_pixel;
		
		white_pixel = visual->red_mask | visual->green_mask | visual->blue_mask;
		rf = 1.0 / visual->red_mask;
		gf = 1.0 / visual->green_mask;
		bf = 1.0 / visual->blue_mask;
		
		for (y = 0; y < height; ++y) {
			for (x = 0; x < width; ++x) {
				pixel = gdk_image_get_pixel (image, x, y) & white_pixel;
				if (pixel) {
					if (pixel == white_pixel) {
						--size;
					} else {
						tmp.v[0] = (pixel & visual->red_mask) * rf;
						tmp.v[1] = (pixel & visual->green_mask) * gf;
						tmp.v[2] = (pixel & visual->blue_mask) * bf;
						
						xeno_color_linear (&tmp, &tmp);
						xeno_color_add (&tmp, &sum, &sum);
					}
				}
			}
		}	
	} else {
		XenoColor	tmp;
		GdkColor	indexed;
		GdkColormap *colormap;
		guint32		*histogram;
		guint32		black_pixel, white_pixel;
		gint		i;
		
		colormap = xeno_theme_colormap;
		gdk_color_white (colormap, &indexed); white_pixel = indexed.pixel;
		gdk_color_black (colormap, &indexed); black_pixel = indexed.pixel;
		histogram = g_new0(guint32, colormap->size);
		
		for (y = 0; y < height; ++y) {
			for (x = 0; x < width; ++x) {
				pixel = gdk_image_get_pixel (image, x, y);
				if (pixel == white_pixel) {
					--size;
				} else if (pixel != black_pixel) {
					for (i = 0; i < MAX(colormap->size - 1, 0); ++i)
						if (colormap->colors[i].pixel == pixel)
							break;
					
					histogram[i] += 1;
				}
			}
		}
		for (i = 0; i < colormap->size; ++i) {
			if (histogram[i] != 0) {
				xeno_color_from_gdk (&tmp, &colormap->colors[i]);
				xeno_color_linear (&tmp, &tmp);
				xeno_color_combine (&sum, &tmp, histogram[i], &sum);
			}
		}
		g_free (histogram);
	}
	gdk_image_destroy (image);
	
	if (size == 0) {
		color->v[0] = color->v[1] = color->v[2] = 1.0;
	} else {
		gfloat f = 1.0 / size;
		xeno_color_init (color, sum.v[0] * f, sum.v[1] * f, sum.v[2] * f);
	}
}

/* end */

