/*
 *  SingIt Lyrics Displayer
 *  Copyright (C) 2000 - 2002 Jan-Marek Glogowski <glogow@stud.fbi.fh-darmstadt.de>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */


#include <X11/Xlib.h>
#include <gdk/gdkx.h>
#include <gtk/gtk.h>
#include <stdio.h>
#include <string.h>
#include <stdlib.h>

#include "singit_macros.h"
#include "singit_macros_private.h"

#include "singit_wgt_karaoke.h"
#include "singit_tools.h"

#define left_right_line_border ((skw->sizes[BALL_DIAMETER] / 2) + 1)
#define font_height(a) ((a)->ascent + (a)->descent)
#define SKW_FONT ((skw->private_font) ? skw->private_font : GTK_WIDGET(skw)->style->font)

/*
	Uncomment to enable extended debug messages
*/
// #define WIDGET_DEBUG

/*
	All sizes in the sizes attribute
	All sizes are computed inside the calc_karaoke_widget_sizes funtion
*/
enum {

	MAX_LINE_WIDTH,			// Longest line in pixel (depends on the font)
	BALL_DIAMETER,			// Can guess 3 times ;-)
	BALL_LINE_HEIGHT,		// The height of the ball line
	TEXT_LINE_HEIGHT,		// Same for the text
	LEFT_RIGHT_LINE_BORDER,		// Left and right text offset
	WIDGET_MIN_HEIGHT,		// Minimum height of the widget
	WIDGET_MIN_WIDTH,		// Minimum width of the widget
	LEFT_RIGHT_WIDGET_BORDER,	// Left and right border around the widget
	TOP_BOTTOM_WIDGET_BORDER,       // Top and bottom border around the widget
	TOP_LINES_HEIGHT,		// Is the height of the lines above the active line
	LAST_VALUE
};

/*
	Referesh enums for the calc_karaoke_widget_sizes funtion
	They are used to set the refresh state (singit_karaoke_widget_refresh)
*/
enum {

	SONG_CHG = 1 << 1,		// Song has changed
	FONT_CHG = 1 << 2,		// Font has changed
	LINE_CHG = 1 << 3		// No of lines has changed
};

enum {
	ARG_0
};

void calc_karaoke_widget_sizes(SingitKaraokeWidget *skw, gint change, gboolean resized);
void singit_karaoke_widget_build_lyric_pixmaps(SingitKaraokeWidget *skw);
void singit_karaoke_widget_update_ball(SingitKaraokeWidget *skw, GdkRectangle *main_area);
void singit_karaoke_widget_update_progess_bar(SingitKaraokeWidget *skw, gchar *text, GdkRectangle *main_area);
void draw_line_rects(SingitKaraokeWidget *skw, GdkPixmap *pixmap, GdkGC *draw_gc);

static GtkWidgetClass *parent_class = NULL;

static void singit_karaoke_widget_class_init (SingitKaraokeWidgetClass *klass);
static void singit_karaoke_widget_init (SingitKaraokeWidget *skw);

static gboolean singit_karaoke_expose_event (GtkWidget *widget, GdkEventExpose *event);
static void singit_karaoke_widget_size_request (GtkWidget *widget, GtkRequisition *requisition);
static void singit_karaoke_widget_size_allocate (GtkWidget *widget, GtkAllocation *allocation);
static void singit_karaoke_widget_realize (GtkWidget *widget);
static void singit_karaoke_widget_unrealize(GtkWidget *widget);

static void singit_karaoke_widget_set_arg(GtkObject *object, GtkArg *arg, guint arg_id);
static void singit_karaoke_widget_get_arg(GtkObject *object, GtkArg *arg, guint arg_id);
static void singit_karaoke_widget_destroy(GtkObject *object);
static void singit_karaoke_widget_finalize(GtkObject *object);

GtkType singit_karaoke_widget_get_type (void)
{
	static GtkType singit_karaoke_widget_type = 0;

	if (!singit_karaoke_widget_type) {

		static const GtkTypeInfo singit_karaoke_widget_info =
		{
			"SingitKaraokeWidget",
			sizeof (SingitKaraokeWidget),
			sizeof (SingitKaraokeWidgetClass),
			(GtkClassInitFunc) singit_karaoke_widget_class_init,
			(GtkObjectInitFunc) singit_karaoke_widget_init,
			/* reserved_1 */ NULL,
			/* reserved_2 */ NULL,
			(GtkClassInitFunc) NULL,
		};

		singit_karaoke_widget_type = gtk_type_unique(GTK_TYPE_WIDGET, &singit_karaoke_widget_info);
	}

	return singit_karaoke_widget_type;
}

static void singit_karaoke_widget_class_init (SingitKaraokeWidgetClass *klass)
{
	GtkObjectClass *object_class = (GtkObjectClass*) klass;
	GtkWidgetClass *widget_class = (GtkWidgetClass*) klass;
	parent_class = gtk_type_class(GTK_TYPE_WIDGET);

	object_class->set_arg = singit_karaoke_widget_set_arg;
	object_class->get_arg = singit_karaoke_widget_get_arg;
	object_class->destroy = singit_karaoke_widget_destroy;
	object_class->finalize = singit_karaoke_widget_finalize;

	widget_class->expose_event = singit_karaoke_expose_event;
	widget_class->size_request = singit_karaoke_widget_size_request;
	widget_class->size_allocate = singit_karaoke_widget_size_allocate;
	widget_class->realize = singit_karaoke_widget_realize;
	widget_class->unrealize = singit_karaoke_widget_unrealize;
}

static void singit_karaoke_widget_init (SingitKaraokeWidget *skw)
{
	#ifdef WIDGET_DEBUG
	g_print("Init\n");
	#endif

//	GTK_WIDGET_UNSET_FLAGS(skw, GTK_NO_WINDOW);
//	GTK_WIDGET_SET_FLAGS(skw, GTK_CAN_FOCUS);

	skw->song = NULL;
	skw->pbp_start = skw->pbp_start_last = skw->pbp_offset = skw->pbp_offset_last =
		skw->pbp_offset_max = 0;
	skw->lines = 3;
	skw->top_lines = 1;
	skw->current = NULL;

	skw->mode = skwm_line_scroll;
	skw->use_ball = TRUE;
	skw->double_buffer_pixmap = NULL;

	skw->ball_y_pos = 0;
	skw->ball_y_pos_last = 1;

	skw->private_color[0].red = 0;
	skw->private_color[0].green = 0;
	skw->private_color[0].blue = 0;

	skw->private_color[1].red = 65535;
	skw->private_color[1].green = 65535;
	skw->private_color[1].blue = 65535;

	skw->private_color[2].red = 65535;
	skw->private_color[2].green = 65535;
	skw->private_color[2].blue = 0;

	gdk_color_alloc(gtk_widget_get_colormap(GTK_WIDGET(skw)), &skw->private_color[0]);
	gdk_color_alloc(gtk_widget_get_colormap(GTK_WIDGET(skw)), &skw->private_color[1]);
	gdk_color_alloc(gtk_widget_get_colormap(GTK_WIDGET(skw)), &skw->private_color[2]);

	skw->show_empty_lines = FALSE;
	skw->maxLineWidth = FALSE;
	skw->last_time = -1;
	skw->line_seperator_high = 5;
	skw->active_line_seperator_high = skw->line_seperator_high * 2;

	skw->sizes = g_new(gint, LAST_VALUE);
	g_get_current_time(&skw->last_rtc_time);
	skw->sizes[WIDGET_MIN_HEIGHT] = 60;
	skw->sizes[WIDGET_MIN_WIDTH] = 80;

	skw->private_font = NULL;
	skw->private_font_name = NULL;
}

static void singit_karaoke_widget_destroy (GtkObject *object)
{
	SingitKaraokeWidget *skw;

	#ifdef WIDGET_DEBUG
	g_print("Destroy\n");
	#endif

	g_return_if_fail (object != NULL);
	g_return_if_fail (IS_SINGIT_KARAOKE_WIDGET (object));

	skw = SINGIT_KARAOKE_WIDGET (object);

	// Cleanup connected signals

/*	if (skw->adjustment)
		gtk_signal_disconnect_by_data (GTK_OBJECT(skw->adjustment), skw);*/

	if (GTK_OBJECT_CLASS(parent_class)->destroy)
		GTK_OBJECT_CLASS(parent_class)->destroy (object);
}

static void singit_karaoke_widget_finalize (GtkObject *object)
{
	SingitKaraokeWidget *skw;

	g_return_if_fail (object != NULL);
	g_return_if_fail (IS_SINGIT_KARAOKE_WIDGET (object));

	#ifdef WIDGET_DEBUG
	g_print("Finalize\n");
	#endif

	skw = SINGIT_KARAOKE_WIDGET (object);

	l_song_detach(skw->song, TRUE);

	if (skw->double_buffer_pixmap)
		{ gdk_pixmap_unref(skw->double_buffer_pixmap); }

	if (skw->ball_pixmap)
		{ gdk_pixmap_unref(skw->ball_pixmap); }

	if (skw->word_pos)
		{ g_free(skw->word_pos); }

	if (skw->private_font_name) {
		if (skw->private_font)
			{ gdk_font_unref(skw->private_font); }
		g_free(skw->private_font_name);
	}

	if (skw->sizes)
		{ g_free(skw->sizes); }

	if (GTK_OBJECT_CLASS(parent_class)->finalize)
		GTK_OBJECT_CLASS(parent_class)->finalize (object);
}

static void singit_karaoke_widget_set_arg(GtkObject *object, GtkArg *arg, guint arg_id)
{
	SingitKaraokeWidget *skw;

	skw = SINGIT_KARAOKE_WIDGET(object);

	switch (arg_id)
	{
	default:
		break;
	}
}

static void singit_karaoke_widget_get_arg(GtkObject *object, GtkArg *arg, guint arg_id)
{
	SingitKaraokeWidget *skw;

	skw = SINGIT_KARAOKE_WIDGET(object);

	switch (arg_id)
	{
	default:
		arg->type = GTK_TYPE_INVALID;
		break;
	}
}

static void singit_karaoke_widget_realize (GtkWidget *widget)
{
	SingitKaraokeWidget *skw;
	GdkWindowAttr attributes;
	gint attributes_mask;

	g_return_if_fail (widget != NULL);
	g_return_if_fail (IS_SINGIT_KARAOKE_WIDGET(widget));

	#ifdef WIDGET_DEBUG
	g_print("Realize\n");
	#endif

	GTK_WIDGET_SET_FLAGS (widget, GTK_REALIZED);
	skw = SINGIT_KARAOKE_WIDGET(widget);

	attributes.x = widget->allocation.x;
	attributes.y = widget->allocation.y;
	attributes.width = widget->allocation.width;
	attributes.height = widget->allocation.height;
	attributes.wclass = GDK_INPUT_OUTPUT;
	attributes.window_type = GDK_WINDOW_CHILD;
	attributes.event_mask = gtk_widget_get_events (widget) | GDK_EXPOSURE_MASK;
	attributes.visual = gtk_widget_get_visual (widget);
	attributes.colormap = gtk_widget_get_colormap (widget);

	attributes_mask = GDK_WA_X | GDK_WA_Y | GDK_WA_VISUAL | GDK_WA_COLORMAP;
	widget->window = gdk_window_new
		(gtk_widget_get_parent_window (widget), &attributes, attributes_mask);

	widget->style = gtk_style_attach (widget->style, widget->window);

	gdk_window_set_user_data (widget->window, widget);

	gtk_style_set_background (widget->style, widget->window, GTK_STATE_ACTIVE);

	calc_karaoke_widget_sizes(skw, SONG_CHG | FONT_CHG | LINE_CHG, FALSE);
}

static void singit_karaoke_widget_unrealize(GtkWidget *widget)
{
	SingitKaraokeWidget *skw;

	g_return_if_fail (widget != NULL);
	g_return_if_fail (IS_SINGIT_KARAOKE_WIDGET(widget));

	#ifdef WIDGET_DEBUG
	g_print("Unrealize\n");
	#endif

	skw = SINGIT_KARAOKE_WIDGET(widget);

/*	if (paned->xor_gc)
	{
		gdk_gc_destroy (paned->xor_gc);
		paned->xor_gc = NULL;
	}*/

	if (GTK_WIDGET_CLASS (parent_class)->unrealize)
		(* GTK_WIDGET_CLASS (parent_class)->unrealize) (widget);
}

static void singit_karaoke_widget_size_request (GtkWidget *widget, GtkRequisition *requisition)
{
	SingitKaraokeWidget *skw;

	g_return_if_fail(widget != NULL);
	g_return_if_fail(IS_SINGIT_KARAOKE_WIDGET(widget));
	g_return_if_fail(requisition != NULL);

	#ifdef WIDGET_DEBUG
	g_print("Size Request\n");
	#endif

	skw = SINGIT_KARAOKE_WIDGET(widget);

	if (skw->song) {
		requisition->width = skw->sizes[WIDGET_MIN_WIDTH];
		requisition->height = skw->sizes[WIDGET_MIN_HEIGHT];
	}
	else {
		if (GTK_WIDGET_CLASS(parent_class)->size_request)
			GTK_WIDGET_CLASS(parent_class)->size_request(widget, requisition);
	}

}

static void singit_karaoke_widget_size_allocate (GtkWidget *widget, GtkAllocation *allocation)
{
	SingitKaraokeWidget *skw;

	g_return_if_fail (widget != NULL);
	g_return_if_fail (IS_SINGIT_KARAOKE_WIDGET(widget));
	g_return_if_fail (allocation != NULL);

	#ifdef WIDGET_DEBUG
	g_print("Size Allocate\n");
	#endif

	widget->allocation = *allocation;

	skw = SINGIT_KARAOKE_WIDGET(widget);
	calc_karaoke_widget_sizes(skw, 0, TRUE);

	if (GTK_WIDGET_REALIZED (widget))
	{
		gdk_window_move_resize (widget->window,
			allocation->x, allocation->y,
			allocation->width, allocation->height);
	}
}

GtkWidget *singit_karaoke_widget_new (void)
{
	GtkWidget *karaoke_widget;

	#ifdef WIDGET_DEBUG
	g_print("New\n");
	#endif

	karaoke_widget = gtk_type_new(TYPE_SINGIT_KARAOKE_WIDGET);

	return karaoke_widget;
}

/*struct _GdkEventExpose
{
  GdkEventType type;
  GdkWindow *window;
  gint8 send_event;
  GdkRectangle area;
  gint count; // If non-zero, how many more events follow.
};*/

typedef enum {

	XLFD_FOUNDRY        = 0,
	XLFD_FAMILY         = 1,
	XLFD_WEIGHT         = 2,
	XLFD_SLANT          = 3,
	XLFD_SET_WIDTH      = 4,
	XLFD_ADD_STYLE      = 5,
	XLFD_PIXELS         = 6,
	XLFD_POINTS         = 7,
	XLFD_RESOLUTION_X   = 8,
	XLFD_RESOLUTION_Y   = 9,
	XLFD_SPACING        = 10,
	XLFD_AVERAGE_WIDTH  = 11,
	XLFD_CHARSET        = 12
}
FontField;

#define GTK_NUM_STYLE_PROPERTIES 5
#define XLFD_MAX_FIELD_LEN 64

typedef struct _FontStyle {

	guint16  properties[GTK_NUM_STYLE_PROPERTIES];
	gint	 pixel_sizes_index;
	guint16  npixel_sizes;
	gint	 point_sizes_index;
	guint16  npoint_sizes;
	guint8   flags;
}
FontStyle;

static gboolean is_xlfd_font_name (const gchar *fontname)
{
	gint i = 0;
	gint field_len = 0;

	while (*fontname) {
		if (*fontname++ == '-') {
			if (field_len > XLFD_MAX_FIELD_LEN) { return FALSE; }
			field_len = 0;
			i++;
		}
		else
			field_len++;
	}
	return (i == 14) ? TRUE : FALSE;
}

static gchar* get_xlfd_field (const gchar *fontname, FontField field_num, gchar *buffer)
{
	const gchar *t1, *t2;
	gint countdown, len, num_dashes;

	if (!fontname) { return NULL; }

	countdown = field_num;
	t1 = fontname;
	while (*t1 && (countdown >= 0))
		if (*t1++ == '-')
			countdown--;

	num_dashes = (field_num == XLFD_CHARSET) ? 2 : 1;
	for (t2 = t1; *t2; t2++) {
		if (*t2 == '-' && --num_dashes == 0) { break; }
	}

	if (t1 != t2) {
		/* Check we don't overflow the buffer */
		len = (long) t2 - (long) t1;
		if (len > XLFD_MAX_FIELD_LEN - 1) { return NULL; }
		strncpy (buffer, t1, len);
		buffer[len] = 0;

		/* Convert to lower case. */
		g_strdown (buffer);
	}
	else
		strcpy(buffer, "(nil)");

	return buffer;
}

/*
	Returns the line number of the longest line
*/
inline gint inl_get_max_line_width_nbr (SingitKaraokeWidget *skw, GdkFont *font)
{
	gint i = 0, strlength = 0, temp = 0, result = -1;
	LSong *my_song = l_song_attach(skw->song);

	if (my_song) {
		if (my_song->lyrics) {
			while (my_song->lyrics[i]) {
	  			temp = gdk_string_width(font, my_song->lyrics[i]);
	  			if (temp > strlength) {
					strlength = temp;
					result = i;
				}
				i++;
			}
		}
		l_song_detach(my_song, TRUE);
	}
	return result;
}

typedef struct _TestFontInfo {

	gint      longest_line_nbr;
	gint      font_pixel_size;
	gint      line_width;
	gchar    *base_font_name;
	gchar   **split_font_name;
	guint8    flags;
	gchar    *xfont_name;
}
TestFontInfo;

void set_optimal_font(SingitKaraokeWidget *skw)
{
	gchar **xfontnames;
	gchar **attrib_list;
	gchar *search_fond_str, *result_font = NULL;
	gint i = 0;
	gint num_fonts;
	gint pixels, points, res_x, res_y;
	gchar field_buffer[XLFD_MAX_FIELD_LEN];
	gchar *field;
	guint8 flags = 0;
	GdkFont *test_font = NULL;
	LSong *my_song = NULL;
	gint test_width = 0, test_height = 0;
	TestFontInfo font_info = { -1, 0, 0, NULL, NULL, 0, NULL };
	gchar *font_start, *font_end, *font_temp;
	gchar font_size[5] = { 0 };

	if (!(my_song = l_song_attach(skw->song))) { return; }

	if (!skw->private_font_name) { return; }

	attrib_list = g_strsplit(skw->private_font_name, "-", 16);
	while (attrib_list[i]) { i++; }
	if (i < 11) {
		g_strfreev(attrib_list);
		return;
	}
	g_free(attrib_list[7]);
	g_free(attrib_list[8]);
	attrib_list[7] = g_strdup("*");
	attrib_list[8] = g_strdup("*");
	search_fond_str = g_strjoinv("-", attrib_list);
	g_strfreev(attrib_list);

	field = get_xlfd_field (skw->private_font_name, XLFD_PIXELS, field_buffer);
	font_info.font_pixel_size = atoi(field);
	field = get_xlfd_field (skw->private_font_name, XLFD_POINTS, field_buffer);
	points = atoi(field);
	if (!font_info.font_pixel_size) { font_info.font_pixel_size = points * 10; }

	xfontnames = XListFonts (GDK_DISPLAY(), search_fond_str, 1000, &num_fonts);

	#ifdef WIDGET_DEBUG
	g_print("\nOld    Font: %s\nSearch Font: %s\nResults :\n", skw->private_font_name,
		search_fond_str);
	#endif
	g_free(search_fond_str);

	// Find the best test font by FONT_TYPE
	i = 0;
	while ((i < num_fonts) && !(flags & GTK_FONT_SCALABLE)) {
		if (is_xlfd_font_name(xfontnames[i])) {
			field = get_xlfd_field (xfontnames[i], XLFD_PIXELS, field_buffer);
			pixels = atoi(field);

			field = get_xlfd_field (xfontnames[i], XLFD_POINTS, field_buffer);
			points = atoi(field);

			field = get_xlfd_field (xfontnames[i], XLFD_RESOLUTION_X, field_buffer);
			res_x = atoi(field);

			field = get_xlfd_field (xfontnames[i], XLFD_RESOLUTION_Y, field_buffer);
			res_y = atoi(field);

			if (pixels == 0 && points == 0) {
				if (res_x == 0 && res_y == 0)
					flags = GTK_FONT_SCALABLE;
				else
					flags = GTK_FONT_SCALABLE_BITMAP;
			}
			else
				flags = GTK_FONT_BITMAP;

			if (font_info.flags < flags) {
				g_free(font_info.base_font_name);
				font_info.base_font_name = g_strdup(xfontnames[i]);
				font_info.flags = flags;
				font_info.font_pixel_size = pixels;
			}
		}
		i++;
	}
	XFreeFontNames(xfontnames);

	// Check which size fits best
	if (font_info.base_font_name && !(font_info.flags & GTK_FONT_BITMAP))
	{
		// Split base font at '-' and construct font_start and font_end
		font_info.split_font_name = g_strsplit(font_info.base_font_name, "-", 16);
		font_temp = font_info.split_font_name[7];
		font_info.split_font_name[7] = NULL;
		result_font = font_start = g_strjoinv("-", font_info.split_font_name);
		font_info.split_font_name[7] = font_temp;
		font_start = g_strconcat(result_font, "-", NULL);
		g_free(result_font);

		result_font = g_strjoinv("-", &font_info.split_font_name[8]);
		font_end = g_strconcat("-", result_font, NULL);
		g_free(result_font);

		i = 7;
		do
		{
			g_snprintf(&font_size[0], 5, "%i", i);
			result_font = g_strconcat(font_start, &font_size, font_end, NULL);
			if ((test_font = gdk_font_load(result_font)) != NULL) {

				test_width = font_height(test_font) / 2.5;
				if ((test_width % 2) == 0) { test_width++; }
				test_width = GTK_WIDGET(skw)->allocation.width - test_width;

				test_height = skw->lines * (font_height(test_font) + font_height(test_font) / 6 + 3)
					+ 2 * (font_height(test_font) / 3 + 1);
				if (skw->use_ball) { test_height += font_height(test_font); }
				else { test_height -= (font_height(test_font) / 6 + 1); }

				if (font_info.longest_line_nbr < 0)
					font_info.longest_line_nbr = inl_get_max_line_width_nbr(skw, test_font);
				font_info.line_width = gdk_string_width
					(test_font, my_song->lyrics[font_info.longest_line_nbr]);
				gdk_font_unref(test_font);

				if (font_info.line_width < test_width)
				{
					g_free(font_info.xfont_name);
					font_info.xfont_name = result_font;
				}
//				g_print("%i / %i\n", font_info.line_width, test_height);
			}
			else { g_free(result_font); }
			i++;
		}
		while ((font_info.line_width < test_width) && (test_height < GTK_WIDGET(skw)->allocation.height) && (test_font != NULL) && (i < 500));


		g_strfreev(font_info.split_font_name);
	}
	if (font_info.xfont_name) {
		#ifdef WIDGET_DEBUG
		g_print("Size: %i / Width: %i\n", i--, font_info.line_width);
		#endif
		singit_karaoke_widget_set_font(skw, font_info.xfont_name);
	}

	g_free(font_info.base_font_name);
	g_free(font_info.xfont_name);

	l_song_detach(my_song, TRUE);
}

/*
	Substracts two regions (rectangles) and return the intersection and
	eventually all 4 part rectangles from the first rectangle.
	The funtion is used to do a full repaint of the widget, if the widget has borders
	to prevent clear on used areas (doesn't flicker)
*/
static gint substract_region(GdkRectangle full, GdkRectangle part, GdkRectangle *result_top,
	GdkRectangle *result_left, GdkRectangle *result_right, GdkRectangle *result_bottom)
{
	gboolean result = 0;
	if (result_top && (full.y < part.y)) {
		(*result_top).x = full.x;
		(*result_top).y = full.y;
		(*result_top).width = full.width;
		(*result_top).height = part.y - full.y;
		result += 1;
	}
	if (result_left && (full.x < part.x)) {
		(*result_left).x = full.x;
		(*result_left).y = part.y;
		(*result_left).width = part.x - full.x;
		(*result_left).height = full.height + full.y - part.y;
		result += 2;
	}
	if (result_right && (full.x + full.width > part.x + part.width)) {
		(*result_right).x = part.x + part.width;
		(*result_right).y = part.y ;
		(*result_right).width = full.x + full.width - part.x - part.width;
		(*result_right).height = full.height + full.y - part.y;
		result += 4;
	}
	if (result_bottom && (full.y + full.height > part.y + part.height)) {
		(*result_bottom).x = part.x;
		(*result_bottom).y = part.y + part.height;
		(*result_bottom).width = part.width;
		(*result_bottom).height = full.y + full.height - part.y - part.height;
		result += 8;
	}
	return result;
}

static gboolean singit_karaoke_expose_event (GtkWidget *widget, GdkEventExpose *event)
{
	GdkGC *draw_gc = NULL;
	SingitKaraokeWidget *skw;
	GdkRectangle intersect_rectangle, widget_rectangle;
	GdkRectangle result_top, result_left, result_right, result_bottom;
	gint draw_sides;

	g_return_val_if_fail(widget != NULL, FALSE);
	g_return_val_if_fail(IS_SINGIT_KARAOKE_WIDGET(widget), FALSE);
	g_return_val_if_fail(event != NULL, FALSE);

	if (!GTK_WIDGET_DRAWABLE(widget)) { return FALSE; }
//	if (event->count > 0) { return FALSE; }

	skw = SINGIT_KARAOKE_WIDGET (widget);

	if (skw->freezers > 0) { return FALSE; }
/*	if (!GTK_WIDGET_REALIZED(GTK_WIDGET(skw)) || !widget->window) { return FALSE; }
	if (!gdk_window_is_visible(widget->window)) { return FALSE; }*/

	widget_rectangle.x = skw->sizes[LEFT_RIGHT_WIDGET_BORDER];
	widget_rectangle.y = skw->sizes[TOP_BOTTOM_WIDGET_BORDER];
	widget_rectangle.width = skw->sizes[WIDGET_MIN_WIDTH];
	widget_rectangle.height = skw->sizes[WIDGET_MIN_HEIGHT];

	if (!(skw->song && skw->double_buffer_pixmap &&
		gdk_rectangle_intersect(&event->area, &widget_rectangle, &intersect_rectangle)))
	{
		gdk_window_clear_area(widget->window, event->area.x, event->area.y,
			event->area.width, event->area.height);
		return FALSE;
	}

	draw_gc = gdk_gc_new (widget->window);
	gdk_gc_copy(draw_gc, GTK_WIDGET(skw)->style->black_gc);
	gdk_gc_set_fill (draw_gc, GDK_SOLID);
	gdk_gc_set_foreground (draw_gc, widget->style->text);

	/*
		Redraw all widget borders if needed
	*/
	if ((draw_sides = substract_region(event->area, intersect_rectangle,
		&result_top, &result_left, &result_right, &result_bottom)) != 0) {
		if (draw_sides & (1 << 0)) {
			gdk_window_clear_area(widget->window, result_top.x, result_top.y,
				result_top.width, result_top.height);
		}
		if (draw_sides & (1 << 1)) {
			gdk_window_clear_area(widget->window, result_left.x, result_left.y,
				result_left.width, result_left.height);
		}
		if (draw_sides & (1 << 2)) {
			gdk_window_clear_area(widget->window, result_right.x, result_right.y,
				result_right.width, result_right.height);
		}
		if (draw_sides & (1 << 3)) {
			gdk_window_clear_area(widget->window, result_bottom.x, result_bottom.y,
				result_bottom.width, result_bottom.height);
		}
	}

	/*
		Redraw the real widget
	 */
	gdk_draw_pixmap(widget->window, widget->style->fg_gc[widget->state],
		skw->double_buffer_pixmap,
		intersect_rectangle.x - skw->sizes[LEFT_RIGHT_WIDGET_BORDER],
		intersect_rectangle.y - skw->sizes[TOP_BOTTOM_WIDGET_BORDER],
		intersect_rectangle.x,
		intersect_rectangle.y,
		intersect_rectangle.width,
		intersect_rectangle.height);

	return FALSE;
}

/*
	Splits the active line to word sizes, so the ball knows where to drop
	This is needed because of different space sizes between the words

	Expl.:
	String: I'm here and there
	Splits: I'm| |here| |and| |there|

	You get 7 split position
	The ball jumps from the start to the middle of the first and second position...
 */
void singit_karaoke_widget_set_word_pos(SingitKaraokeWidget *skw, gchar *text)
{
	gchar **splitted_line = NULL, *text_dup;
	gint counter = 0;
	gint textpos = 0;
	GdkFont *my_font = SKW_FONT;

	if (skw->word_pos) {
		g_free(skw->word_pos);
		skw->word_pos = NULL;
	}
	if ((!text) || (strlen(text) == 0)) { return; }

	textpos = strlen(text);
	text_dup = g_strdup(text);
	splitted_line = g_strsplit (text, " ", 50);
	while (splitted_line[counter]) { counter++; }
	counter = counter * 2;
	skw->word_pos = g_new(gint, counter+1);
	skw->word_pos[counter] = 0;
	counter--;
	skw->word_pos[counter] = 0;
	counter--;
	while (counter >= 0) {
		text_dup[textpos] = '\0';
		if (textpos >= 0) {
			textpos -= (strlen(splitted_line[counter / 2]) + 1);
		}
		skw->word_pos[counter] = gdk_string_measure(my_font, text_dup);
		if (counter > 1)
				skw->word_pos[counter-1] = gdk_string_measure(my_font, (gchar*)(text_dup + textpos + 1));
		counter -= 2;
	}
	counter = 0;
	while (skw->word_pos[counter+1] != 0) {
		skw->word_pos[counter+1] = skw->word_pos[counter+2] - skw->word_pos[counter] - skw->word_pos[counter+1];
		counter += 2;
	}
	g_free(text_dup);
	g_strfreev(splitted_line);
}

/*
	Get the y position of the ball dependant on the word split positions
	and the x position of the progress bar
 */
void singit_karaoke_widget_set_curr_y_pos(SingitKaraokeWidget *skw)
{
/*
	This is the smallest range the ball can jump
	It's the pixel size of the characters "IM"
*/
#	define smallest_jump "IM"

	gint cur_word_pos, new_ball_pos_index = 0, min_size, new_pos = 0;
	gint real_height = skw->sizes[BALL_LINE_HEIGHT] - skw->sizes[BALL_DIAMETER];
	gboolean found = FALSE;
	GdkFont *my_font = SKW_FONT;

	if ((skw->pbp_offset != skw->pbp_offset_last) && (skw->pbp_offset == 0)) { return; }
	if (skw->pbp_offset + skw->pbp_start == 0) {
		skw->ball_y_pos = 0;
		return;
	}
	if (!skw->word_pos) {
		skw->ball_y_pos = skw->sizes[BALL_LINE_HEIGHT] - skw->sizes[BALL_DIAMETER];
		skw->pbp_offset = 0;
		skw->pbp_start = 0;
		return;
	}

	if (((skw->pbp_offset + skw->pbp_start) < skw->ball_start) ||
		((skw->pbp_offset + skw->pbp_start) > (skw->ball_start + skw->ball_offset_max))) {
		new_pos = 0;
		while (skw->word_pos[new_ball_pos_index * 2] && !found) {
			if (skw->word_pos[new_ball_pos_index * 2] > (skw->pbp_start + skw->pbp_offset)) {
				if (new_ball_pos_index == 0) { new_pos = 0; }
				else { new_pos = skw->word_pos[(new_ball_pos_index-1) * 2]; }
				new_ball_pos_index--;
				found = TRUE;
			}
			new_ball_pos_index++;
		}

		if (skw->pbp_start > new_pos) { skw->ball_start= skw->pbp_start; }
		else { skw->ball_start = new_pos; }

		new_pos = 0;
		min_size = gdk_string_measure(my_font, smallest_jump);
		while ((new_pos < min_size) && skw->word_pos[new_ball_pos_index * 2]) {
			new_pos = skw->word_pos[new_ball_pos_index * 2] - skw->ball_start;
			if (((skw->pbp_offset_max + skw->pbp_start) < (skw->ball_start + new_pos)) &&
				((skw->pbp_offset_max - skw->pbp_offset) > min_size))
				{ new_pos = skw->pbp_offset_max + skw->pbp_start
					- skw->ball_start; }
			new_ball_pos_index++;
		}
		skw->ball_offset_max = new_pos;
	}

	cur_word_pos = (skw->pbp_offset + skw->pbp_start) - skw->ball_start;
	if (skw->ball_offset_max == 0) {  skw->ball_y_pos = real_height; }
	else {
		new_pos = 2 * (cur_word_pos / (double) skw->ball_offset_max) * real_height;
		if (new_pos < real_height) { new_pos = real_height - new_pos; }
		else { new_pos = new_pos - real_height; }
		skw->ball_y_pos = new_pos;
	}
}

/*
	This creates a new double_buffer_pixmap dependant on the current
	song position
 */
void singit_karaoke_widget_set_time (SingitKaraokeWidget *skw, guint time)
{
	GList *new_item = NULL, *next_item;
	LSong *my_song = NULL;
	gchar *text = NULL, *startPos, *endPos = NULL;
	gchar oldChar;
	gboolean recheck = FALSE;
	gint bearings[2] = { 0 };
	double multi = 0.0;
	GdkRectangle area;
	GdkFont *my_font = SKW_FONT;

//	return;

	g_return_if_fail(skw != NULL);
	g_return_if_fail(IS_SINGIT_KARAOKE_WIDGET(skw));

	if (skw->last_time == time) { return; }
	else { skw->last_time = time; }

	my_song = l_song_attach(skw->song);
	if (my_song) {
		area.x = skw->sizes[LEFT_RIGHT_WIDGET_BORDER];
		area.y = skw->sizes[TOP_BOTTOM_WIDGET_BORDER];
		area.width = 0;
		area.height = 0;
		new_item = my_song->active_token;
		next_item = inl_l_song_get_next_token(my_song);
		if ((new_item) && (new_item != skw->current)) {
			text = tText(my_song, new_item);
			if (new_item != my_song->last_token) { recheck = (tPos(g_list_next(new_item))); }
			if ((tPos(new_item) > 0) || recheck) {
				startPos = text + tPos(new_item);
				oldChar = startPos[0];
				startPos[0] = '\0';
				gdk_string_extents (my_font, text,
					&bearings[0], &bearings[1], &skw->pbp_start, NULL, NULL);
				skw->pbp_start += gdk_string_measure(my_font, text);
				skw->pbp_start /= 2;
				startPos[0] = oldChar;
				recheck = TRUE;
				if (next_item) {
					recheck = (tLine(next_item) != tLine(new_item));
					if (recheck) { recheck = (tPos(next_item) == strlen(tText(my_song, new_item))); }
				}
				if (recheck) {
					gdk_string_extents (my_font, startPos,
						&bearings[0], &bearings[1], &skw->pbp_offset_max, NULL, NULL);
					skw->pbp_offset_max += gdk_string_measure(my_font, startPos);
					skw->pbp_offset_max /= 2;
				}
				else {
					if (next_item) {
						endPos = text + tPos(next_item);
						oldChar = endPos[0];
						endPos[0] = '\0';
					}
					gdk_string_extents (my_font, startPos,
						&bearings[0], &bearings[1], &skw->pbp_offset_max, NULL, NULL);
					skw->pbp_offset_max += gdk_string_measure(my_font, startPos);
					skw->pbp_offset_max /= 2;
					if (new_item != my_song->last_token) { endPos[0] = oldChar; }
				}
			}
			else {
				skw->pbp_start = skw->pbp_offset = 0;
				gdk_string_extents (my_font, text,
					&bearings[0], &bearings[1], &skw->pbp_offset_max, NULL, NULL);
				skw->pbp_offset_max += gdk_string_measure(my_font, text);
				skw->pbp_offset_max /= 2;
			}
			if (skw->current != new_item) {
				recheck = TRUE;
				if (skw->current && new_item)
					if (tLine(new_item) == tLine(skw->current)) { recheck = FALSE; }
				if (recheck) {
					if ((skw->current == g_list_previous(new_item)) && (skw->current != NULL) &&
						(!skw->show_empty_lines)) {
						if (!l_song_is_empty_item(my_song, skw->current)) {
							singit_karaoke_widget_build_lyric_pixmaps(skw);
							area.width = skw->sizes[WIDGET_MIN_WIDTH];
							area.height = skw->sizes[WIDGET_MIN_HEIGHT];
						}
					}
					else {
						singit_karaoke_widget_build_lyric_pixmaps(skw);
						area.width = skw->sizes[WIDGET_MIN_WIDTH];
						area.height = skw->sizes[WIDGET_MIN_HEIGHT];
					}
				}
				skw->current = new_item;
			}
		}

		if (new_item) {
			if (new_item != my_song->last_token) {
				multi = (time - tTime(new_item)) / (tTime(next_item) - tTime(new_item));
				if ((multi > 1.0) || (multi < 0.0)) { skw->pbp_offset = skw->pbp_offset_max; }
				else {
					skw->pbp_offset = skw->pbp_offset_max *
						(time - tTime(new_item)) / (tTime(next_item) - tTime(new_item));
				}
			}
			else { skw->pbp_offset = skw->pbp_offset_max; }
		}
		else {
			if ((my_song->first_token) && (new_item != skw->current)) {
				singit_karaoke_widget_build_lyric_pixmaps(skw);
				area.width = skw->sizes[WIDGET_MIN_WIDTH];
				area.height = skw->sizes[WIDGET_MIN_HEIGHT];
			}
			skw->current = new_item;
		}

		if ((skw->pbp_offset_last != skw->pbp_offset)) {
			if (skw->use_ball) {
				singit_karaoke_widget_set_curr_y_pos(skw);
				singit_karaoke_widget_update_ball(skw, &area);
			}
			if (new_item) {
				singit_karaoke_widget_update_progess_bar
					(skw, tText(my_song, new_item), &area);
			}
		}
		else if (skw->use_ball && (skw->ball_y_pos_last != skw->ball_y_pos)) {
			singit_karaoke_widget_set_curr_y_pos(skw);
			singit_karaoke_widget_update_ball(skw, &area);
		}

		if ((skw->freezers == 0) && (area.width != 0) && (area.height != 0)) {
			if (GTK_WIDGET_DRAWABLE(GTK_WIDGET(skw))) {
				gtk_widget_queue_draw_area(GTK_WIDGET(skw),
					area.x, area.y, area.width, area.height);
				skw->frames++;
			}
		}
		skw->pbp_offset_last = skw->pbp_offset;
		skw->pbp_start_last = skw->pbp_start;
		skw->ball_y_pos_last = skw->ball_y_pos;

		g_get_current_time(&skw->new_rtc_time);
		if ((skw->new_rtc_time.tv_sec - skw->last_rtc_time.tv_sec) > 5) {
			skw->fps = skw->frames / (gdouble) (skw->new_rtc_time.tv_sec - skw->last_rtc_time.tv_sec);
			skw->frames = 0;
			skw->last_rtc_time = skw->new_rtc_time;
		}
		l_song_detach(my_song, TRUE);
	}
}

/*
	Updates the ball part of the double_buffer_pixmap and return the
	united area which has to be repaintet
*/
void singit_karaoke_widget_update_ball(SingitKaraokeWidget *skw, GdkRectangle *main_area)
{
	GdkGC *draw_gc = NULL;
	gint pos_x, height;
	gint xpos[2];
	GdkRectangle area;

	if ((skw->ball_y_pos_last == skw->ball_y_pos) &&
		(skw->pbp_offset_last == skw->pbp_offset)) { return; }

	if (!(GTK_WIDGET(skw)->window && skw->double_buffer_pixmap)) { return; }

	draw_gc = gdk_gc_new (GTK_WIDGET(skw)->window);
	gdk_gc_set_fill (draw_gc, GDK_SOLID);
	gdk_gc_set_foreground (draw_gc, &skw->private_color[skwc_background]);

	pos_x = skw->pbp_start_last + skw->pbp_offset_last;
	xpos[0] = xpos[1] = pos_x;
	height = skw->sizes[TOP_LINES_HEIGHT];

	gdk_draw_rectangle(skw->double_buffer_pixmap, draw_gc, TRUE, pos_x,
		skw->ball_y_pos_last + height, skw->sizes[BALL_DIAMETER], skw->sizes[BALL_DIAMETER]);

	if (inl_l_song_get_next_token(skw->song)) {
		pos_x = skw->pbp_start + skw->pbp_offset;
		xpos[1] = pos_x;
		gdk_draw_pixmap(skw->double_buffer_pixmap, draw_gc, skw->ball_pixmap, 0, 0,
			pos_x, skw->ball_y_pos + height, skw->sizes[BALL_DIAMETER], skw->sizes[BALL_DIAMETER]);
	}

	gdk_gc_destroy(draw_gc);

	if (main_area && (skw->freezers == 0)) {
		area.y = skw->sizes[TOP_BOTTOM_WIDGET_BORDER] + height;
		area.height = skw->sizes[BALL_LINE_HEIGHT] + height;
		if (xpos[0] < xpos[1]) {
			area.x = skw->sizes[LEFT_RIGHT_WIDGET_BORDER] + xpos[0];
			area.width = xpos[1] - xpos[0] + skw->sizes[BALL_DIAMETER];
		}
		else {
			area.x = skw->sizes[LEFT_RIGHT_WIDGET_BORDER] + xpos[1];
			area.width = xpos[0] - xpos[1] + skw->sizes[BALL_DIAMETER];
		}
		gdk_rectangle_union(main_area, &area, main_area);
	}
}

/*
	Updates the progress bar part of the double_buffer_pixmap and return the
	united area which has to be repaintet
*/
void singit_karaoke_widget_update_progess_bar(SingitKaraokeWidget *skw, gchar *text, GdkRectangle *main_area)
{
	GdkGC *draw_gc = NULL;
	GtkWidget *widget = GTK_WIDGET(skw);
	GdkFont *my_font = SKW_FONT;
	GdkRectangle area;
	gint height;
	gint ranges[4];

	if (strlen(text) < 1) { return; }
	if (!(GTK_WIDGET(skw)->window && skw->double_buffer_pixmap)) { return; }

	area.height = skw->sizes[TEXT_LINE_HEIGHT];

	draw_gc = gdk_gc_new (widget->window);
	gdk_gc_set_fill (draw_gc, GDK_SOLID);

	height = skw->sizes[TOP_LINES_HEIGHT];
	if (skw->use_ball) { height += (skw->sizes[BALL_LINE_HEIGHT] + skw->line_seperator_high); }

	ranges[0] = skw->pbp_start_last + skw->pbp_offset_last;
	ranges[1] = skw->pbp_start + skw->pbp_offset;
	ranges[2] = MIN(ranges[0], ranges[1]); // x
	ranges[3] = MAX(ranges[0], ranges[1]) - ranges[2]; // width

	if (ranges[0] > ranges[1]) {
		gdk_gc_set_foreground (draw_gc, &skw->private_color[skwc_background]);
		gdk_draw_rectangle(skw->double_buffer_pixmap,
			draw_gc, TRUE, left_right_line_border + ranges[2], height, ranges[3], area.height);
	}
	else if (ranges[0] < ranges[1]) {
		gdk_gc_set_foreground (draw_gc, &skw->private_color[skwc_active]);
		gdk_draw_rectangle(skw->double_buffer_pixmap,
			draw_gc, TRUE, left_right_line_border + ranges[2], height, ranges[3], area.height);
	}

	gdk_gc_set_foreground (draw_gc, &skw->private_color[skwc_text]);
	gdk_draw_string (skw->double_buffer_pixmap,
		my_font, draw_gc, left_right_line_border, height + my_font->ascent + 1, text);

	gdk_gc_destroy(draw_gc);

	if (main_area && (skw->freezers == 0)) {
		area.x = skw->sizes[LEFT_RIGHT_WIDGET_BORDER] + ranges[2];
		area.y = skw->sizes[TOP_BOTTOM_WIDGET_BORDER] + height;
		area.width = ranges[3];
		gdk_rectangle_union(main_area, &area, main_area);
	}
}

/*
	Creates all text lines
*/
void singit_karaoke_widget_build_lyric_pixmaps (SingitKaraokeWidget *skw)
{
	GList *item = 0;
	GdkGC *curgc = NULL;
	LSong *my_song = NULL;
	GdkFont *my_font = SKW_FONT;
	gint i, height;

//	return;

	if (!(GTK_WIDGET(skw)->window && skw->double_buffer_pixmap &&
		GTK_WIDGET_REALIZED(GTK_WIDGET(skw)))) { return; }

	curgc = gdk_gc_new(GTK_WIDGET(skw)->window);
	gdk_gc_set_foreground (curgc, &skw->private_color[skwc_background]);
	gdk_gc_set_fill (curgc, GDK_SOLID);

	draw_line_rects(skw, skw->double_buffer_pixmap, curgc);

	my_song = l_song_attach(skw->song);
	if (my_song) {

		skw->pbp_start_last = skw->pbp_offset_last = 0;

		if (skw->mode == skwm_line_scroll) {
			skw->ball_offset_max = skw->ball_start = 0;
			item = my_song->active_token;

			i = skw->top_lines;
			height = skw->sizes[TOP_LINES_HEIGHT];
			if (skw->use_ball) {
				height += (skw->sizes[BALL_LINE_HEIGHT] + skw->line_seperator_high);
			}

			if (item) {
				if ((!skw->show_empty_lines) && (l_song_is_empty_item(my_song, item)))
					item = l_song_find_next_lyric_line(my_song, item, FALSE, NULL);

				if (item) {
					singit_karaoke_widget_set_word_pos(skw, tText(my_song, item));

					while ((i > 0) && item) {
						item = l_song_find_prev_lyric_line(my_song, item, skw->show_empty_lines, NULL);
						i--;
					}
					if (!item) {
						item = my_song->first_token;
						i++;
					}
					if (i < skw->top_lines) {
						height = i * (skw->sizes[TEXT_LINE_HEIGHT] + skw->line_seperator_high);
					}
					else if (i > skw->top_lines) {
						height = i * (skw->sizes[TEXT_LINE_HEIGHT] + skw->line_seperator_high) +
							2 * skw->active_line_seperator_high;
						if (skw->use_ball) {
							height += (skw->sizes[BALL_LINE_HEIGHT] + skw->line_seperator_high);
						}
					}
				}
			}
			else {
				item = my_song->first_token;
				if (skw->show_empty_lines) {
					i++;
					height = i * (skw->sizes[TEXT_LINE_HEIGHT] + skw->line_seperator_high) +
						2 * skw->active_line_seperator_high;
					if (skw->use_ball) {
						height += (skw->sizes[BALL_LINE_HEIGHT] + skw->line_seperator_high);
					}
				}
			}

			gdk_gc_set_foreground (curgc, &skw->private_color[skwc_text]);
			while (i < skw->lines) {
				if (item) {
					gdk_draw_string (skw->double_buffer_pixmap, my_font,
						curgc, left_right_line_border, height + my_font->ascent + 1,
						tText(my_song, item));
					item = l_song_find_next_lyric_line(my_song, item, skw->show_empty_lines, NULL);
				}
				if (i == (skw->top_lines - 1)) {
					height += skw->active_line_seperator_high;
					if (skw->use_ball) {
						height += (skw->sizes[BALL_LINE_HEIGHT] + skw->line_seperator_high);
					}
				}
				else if (i == skw->top_lines) {
					height += skw->active_line_seperator_high;
				}

				i++;
				height += (skw->sizes[TEXT_LINE_HEIGHT] + skw->line_seperator_high);
			}
		}
		my_song = l_song_detach(my_song, TRUE);
	}

	gdk_gc_destroy(curgc);
}

void singit_karaoke_widget_refresh (SingitKaraokeWidget *skw, gint add_changer)
{
	static gint changer = 0;

	changer |= add_changer;

	if (skw->freezers != 0) { return; }
	if (!skw->song) { return; }

	#ifdef WIDGET_DEBUG
	g_print("Refresh\n");
	#endif

	calc_karaoke_widget_sizes(skw, changer, FALSE);
	skw->current = NULL;
	singit_karaoke_widget_build_lyric_pixmaps(skw);
	if (skw->song && skw->song->active_token)
		singit_karaoke_widget_update_progess_bar
			(skw, tText(skw->song, skw->song->active_token), FALSE);

	if (GTK_WIDGET_DRAWABLE(GTK_WIDGET(skw)))
		gtk_widget_set_usize((GtkWidget*) skw,
			skw->sizes[WIDGET_MIN_WIDTH], skw->sizes[WIDGET_MIN_HEIGHT]);
/*	if (GTK_WIDGET_DRAWABLE(GTK_WIDGET (skw)))
		{ gtk_widget_queue_resize(GTK_WIDGET(skw)); }*/

	changer = 0;

	#ifdef WIDGET_DEBUG
	g_print("Refresh End\n");
	#endif
}

void singit_karaoke_widget_set_song (SingitKaraokeWidget *skw, LSong *new_song)
{
	LSong *attached_song;

	g_return_if_fail(skw != NULL);
	g_return_if_fail(IS_SINGIT_KARAOKE_WIDGET(skw));

	#ifdef WIDGET_DEBUG
	g_print("Set Song\n");
	#endif

	attached_song = l_song_attach(new_song);

	if (attached_song != skw->song) {
		l_song_detach(skw->song, TRUE);
		skw->song = attached_song;

		skw->current = NULL;
		skw->pbp_start = skw->pbp_offset = 0;
		skw->pbp_start_last = skw->pbp_offset_last = 0;
		skw->last_time = -1;
		skw->ball_y_pos_last = skw->ball_y_pos = 0;

		singit_karaoke_widget_refresh(skw, SONG_CHG);
	}
	else { l_song_detach(attached_song, TRUE); }
}

void singit_karaoke_widget_set_lines(SingitKaraokeWidget *skw, guint value)
{
	g_return_if_fail(skw != NULL);
	g_return_if_fail(IS_SINGIT_KARAOKE_WIDGET(skw));

	#ifdef WIDGET_DEBUG
	g_print("Set Lines\n");
	#endif

//	return;

	if ((value != skw->lines) && (value > 0)) {
		skw->lines = value;
		if (!(skw->top_lines < value))
			{ skw->top_lines = value - 1; }
		singit_karaoke_widget_refresh(skw, LINE_CHG);
	}
}

guint singit_karaoke_widget_get_lines(SingitKaraokeWidget *skw)
{
	#ifdef WIDGET_DEBUG
	g_print("Get Lines\n");
	#endif

	g_return_val_if_fail(skw != NULL, 0);
	g_return_val_if_fail(IS_SINGIT_KARAOKE_WIDGET(skw), 0);

	return skw->lines;
}

void singit_karaoke_widget_set_toplines(SingitKaraokeWidget *skw, guint value)
{
	g_return_if_fail(skw != NULL);
	g_return_if_fail(IS_SINGIT_KARAOKE_WIDGET(skw));

	#ifdef WIDGET_DEBUG
	g_print("Set Toplines\n");
	#endif

//	return;

	if ((skw->top_lines != value) && (skw->top_lines < (skw->lines-1))) {
		skw->top_lines = value;
		singit_karaoke_widget_refresh(skw, 0);
	}
}

guint singit_karaoke_widget_get_toplines(SingitKaraokeWidget *skw)
{
	#ifdef WIDGET_DEBUG
	g_print("Get Toplines\n");
	#endif

	g_return_val_if_fail(skw != NULL, 0);
	g_return_val_if_fail(IS_SINGIT_KARAOKE_WIDGET(skw), 0);

	return skw->top_lines;
}

void singit_karaoke_widget_set_color (SingitKaraokeWidget *skw, SingitKaraokeWidgetColor item, gchar *color)
{
	GdkColor new_color;

	g_return_if_fail(skw != NULL);
	g_return_if_fail(IS_SINGIT_KARAOKE_WIDGET(skw));

	#ifdef WIDGET_DEBUG
	g_print("Set Color : ");
	#endif

//	return;

	if (!color) { return; }

	if (gdk_color_parse(color, &new_color)) {
		skw->private_color[item] = new_color;
		gdk_color_alloc(gtk_widget_get_colormap(GTK_WIDGET(skw)), &skw->private_color[item]);
		#ifdef WIDGET_DEBUG
		g_print("Ok\n");
		#endif
	}
	#ifdef WIDGET_DEBUG
	else {
		g_print("Failed");
	}
	#endif
}

void singit_karaoke_widget_set_font (SingitKaraokeWidget *skw, const gchar *font)
{
	GdkFont *new_font = NULL;

	g_return_if_fail(skw != NULL);
	g_return_if_fail(IS_SINGIT_KARAOKE_WIDGET(skw));

	#ifdef WIDGET_DEBUG
	g_print("Set Font] : ");
	#endif

//	return;

	g_return_if_fail(font != NULL);

	if ((new_font = gdk_font_load(font)) != NULL) {
		if (skw->private_font) { gdk_font_unref(skw->private_font); }
		skw->private_font = new_font;

		g_free(skw->private_font_name);
		skw->private_font_name = g_strdup(font);

		#ifdef WIDGET_DEBUG
		g_print("Ok\n");
		#endif

		singit_karaoke_widget_refresh(skw, FONT_CHG);
	}
	#ifdef WIDGET_DEBUG
	else {
		g_print("Failed\n");
	}
	#endif
}

void singit_karaoke_widget_set_show_empty_lines (SingitKaraokeWidget *skw, const gboolean value)
{
	g_return_if_fail(skw != NULL);
	g_return_if_fail(IS_SINGIT_KARAOKE_WIDGET(skw));

	#ifdef WIDGET_DEBUG
	g_print("Set Empty Lines\n");
	#endif

//	return;

	if (value != skw->show_empty_lines) {

		skw->show_empty_lines = value;
	}
}

void singit_karaoke_widget_set_jumping_ball (SingitKaraokeWidget *skw, const gboolean value)
{
	g_return_if_fail(skw != NULL);
	g_return_if_fail(IS_SINGIT_KARAOKE_WIDGET(skw));

	#ifdef WIDGET_DEBUG
	g_print("Set Jumping Ball\n");
	#endif

//	return;

	if (value != skw->use_ball) {

		skw->use_ball = value;
		singit_karaoke_widget_refresh(skw, 0);
	}
}

void singit_karaoke_widget_freeze(SingitKaraokeWidget *skw)
{ skw->freezers++; }

void singit_karaoke_widget_thaw(SingitKaraokeWidget *skw)
{
	if (skw->freezers > 0) { skw->freezers--; }
	if (skw->freezers == 0) { singit_karaoke_widget_refresh(skw, 0); }
}

void singit_karaoke_widget_set_optimal_font (SingitKaraokeWidget *skw)
{
	g_return_if_fail(skw != NULL);
	g_return_if_fail(IS_SINGIT_KARAOKE_WIDGET(skw));

	#ifdef WIDGET_DEBUG
	g_print("Set Optimal Font Size\n");
	#endif

//	return;

	singit_karaoke_widget_freeze(skw);
	set_optimal_font(skw);
	singit_karaoke_widget_thaw(skw);
}


/* Helper funtions */

void create_new_ball_pixmap(SingitKaraokeWidget *skw)
{
	GdkGC *draw_gc = NULL;

	g_return_if_fail(skw != NULL);
	g_return_if_fail(IS_SINGIT_KARAOKE_WIDGET(skw));

	if (!GTK_WIDGET_REALIZED(skw)) { return; }

	#ifdef WIDGET_DEBUG
	g_print("New Ball\n");
	#endif

	if (skw->ball_pixmap)
		{ gdk_pixmap_unref(skw->ball_pixmap); }

	if (skw->sizes[BALL_DIAMETER] <= 0) {
		skw->ball_pixmap = NULL;
		return;
	}

	draw_gc = gdk_gc_new (GTK_WIDGET(skw)->window);
	gdk_gc_set_fill (draw_gc, GDK_SOLID);
	gdk_gc_set_foreground (draw_gc, &skw->private_color[skwc_background]);

	skw->ball_pixmap = gdk_pixmap_new(GTK_WIDGET(skw)->window,
		skw->sizes[BALL_DIAMETER], skw->sizes[BALL_DIAMETER], -1);
	gdk_draw_rectangle(skw->ball_pixmap, draw_gc, TRUE, 0, 0,
		skw->sizes[BALL_DIAMETER], skw->sizes[BALL_DIAMETER]);

	gdk_gc_set_foreground (draw_gc, &skw->private_color[skwc_text]);
	gdk_draw_arc (skw->ball_pixmap, draw_gc, TRUE, 0, 0,
			skw->sizes[BALL_DIAMETER], skw->sizes[BALL_DIAMETER], 0, 360 * 64);


	if (skw->sizes[BALL_DIAMETER] > 5) {
		gdk_gc_set_foreground (draw_gc, &skw->private_color[skwc_active]);

		gdk_draw_arc (skw->ball_pixmap, draw_gc, TRUE, 2, 2,
			skw->sizes[BALL_DIAMETER] - 4, skw->sizes[BALL_DIAMETER] - 4, 0, 360 * 64);
	}

	gdk_gc_unref(draw_gc);
}

void draw_line_rects(SingitKaraokeWidget *skw, GdkPixmap *pixmap, GdkGC *draw_gc)
{
	gint height, i;

	g_return_if_fail(skw != NULL);
	g_return_if_fail(IS_SINGIT_KARAOKE_WIDGET(skw));

	if (!(pixmap && draw_gc)) { return; }

	height = 0;
	for (i = 0; i < skw->top_lines; i++) {
		gdk_draw_rectangle(pixmap, draw_gc,
			TRUE, 0,height, skw->sizes[WIDGET_MIN_WIDTH],
			skw->sizes[TEXT_LINE_HEIGHT]);
		height += skw->sizes[TEXT_LINE_HEIGHT] + skw->line_seperator_high;
	}
	height += skw->active_line_seperator_high;

	if (skw->use_ball) {
		gdk_draw_rectangle(pixmap, draw_gc,
			TRUE, 0, height, skw->sizes[WIDGET_MIN_WIDTH],
			skw->sizes[BALL_LINE_HEIGHT]);
		height += skw->sizes[BALL_LINE_HEIGHT] + skw->line_seperator_high;
	}

	for (i = 0; i < (skw->lines - skw->top_lines); i++) {
		gdk_draw_rectangle(pixmap, draw_gc,
			TRUE, 0,height, skw->sizes[WIDGET_MIN_WIDTH],
			skw->sizes[TEXT_LINE_HEIGHT]);
		height += skw->sizes[TEXT_LINE_HEIGHT] + skw->line_seperator_high;
		if (i == 0) { height += skw->active_line_seperator_high; }
	}
}

void create_new_doublebuffer_pixmap(SingitKaraokeWidget *skw)
{
	GdkGC *draw_gc;

	g_return_if_fail(skw != NULL);
	g_return_if_fail(IS_SINGIT_KARAOKE_WIDGET(skw));

	if (!GTK_WIDGET_REALIZED(skw)) { return; }

	#ifdef WIDGET_DEBUG
	g_print("New Doublebuffer\n");
	#endif

	if (skw->double_buffer_pixmap)
		{ gdk_pixmap_unref(skw->double_buffer_pixmap); }

	if ((skw->sizes[WIDGET_MIN_WIDTH] <= 0) || (skw->sizes[WIDGET_MIN_HEIGHT] <= 0)) {
		skw->double_buffer_pixmap = NULL;
		return;
	}

	skw->double_buffer_pixmap = gdk_pixmap_new(GTK_WIDGET(skw)->window,
		skw->sizes[WIDGET_MIN_WIDTH], skw->sizes[WIDGET_MIN_HEIGHT], -1);
	gdk_draw_rectangle(skw->double_buffer_pixmap, GTK_WIDGET(skw)->style->bg_gc[0], TRUE,
		0, 0, skw->sizes[WIDGET_MIN_WIDTH], skw->sizes[WIDGET_MIN_HEIGHT]);

	draw_gc = gdk_gc_new (GTK_WIDGET(skw)->window);
	gdk_gc_set_fill (draw_gc, GDK_SOLID);
	gdk_gc_set_foreground (draw_gc, &skw->private_color[skwc_background]);

	draw_line_rects(skw, skw->double_buffer_pixmap, draw_gc);

	gdk_gc_unref(draw_gc);
}

guint get_max_line_width (SingitKaraokeWidget *skw)
{
	gint i = 0, strlength;
	guint line_width = 0;
	GdkFont *my_font = SKW_FONT;
	LSong *my_song = l_song_attach(skw->song);

	if (my_song) {
		if (my_song->lyrics) {
			while (my_song->lyrics[i]) {
				strlength = strlen(my_song->lyrics[i]);
				if (strlength > 0) {
		  			strlength = gdk_string_width(my_font, my_song->lyrics[i]);
		  			if (strlength > line_width) { line_width = strlength; }
				}
				i++;
			}
		}
		l_song_detach(my_song, TRUE);
	}
	return line_width;
}

void calc_karaoke_widget_sizes(SingitKaraokeWidget *skw, gint change, gboolean resized)
{
	GdkFont *my_font = SKW_FONT;

	if (!resized) {
		#ifdef WIDGET_DEBUG
		g_print("Calc Sizes - Changes\n");
		#endif

		if (change & FONT_CHG) {

			skw->sizes[BALL_LINE_HEIGHT] = font_height(my_font);

			skw->sizes[TEXT_LINE_HEIGHT] = skw->sizes[BALL_LINE_HEIGHT] + 2;

			skw->sizes[BALL_DIAMETER] = (font_height(my_font) / 2.5);
			if ((skw->sizes[BALL_DIAMETER] % 2) == 0) { skw->sizes[BALL_DIAMETER]++; }

			skw->sizes[LEFT_RIGHT_LINE_BORDER] = ((skw->sizes[BALL_DIAMETER] / 2) + 1);

			skw->line_seperator_high = font_height(my_font) / 6 + 1;
			skw->active_line_seperator_high = font_height(my_font) / 3 + 1;

			create_new_ball_pixmap(skw);
		}


		skw->sizes[WIDGET_MIN_HEIGHT] =
			skw->lines * (skw->sizes[TEXT_LINE_HEIGHT] + skw->line_seperator_high) +
			2 * skw->active_line_seperator_high;
		if (skw->use_ball)
			skw->sizes[WIDGET_MIN_HEIGHT] += skw->sizes[BALL_LINE_HEIGHT];
		else
			skw->sizes[WIDGET_MIN_HEIGHT] -= skw->line_seperator_high;

		if (change & (FONT_CHG | SONG_CHG))
			{ skw->sizes[MAX_LINE_WIDTH] = get_max_line_width(skw); }

		skw->sizes[WIDGET_MIN_WIDTH] = skw->sizes[MAX_LINE_WIDTH] + 2 * skw->sizes[LEFT_RIGHT_LINE_BORDER];

		skw->sizes[TOP_LINES_HEIGHT] = skw->top_lines * (skw->sizes[TEXT_LINE_HEIGHT] + skw->line_seperator_high)
			+ skw->active_line_seperator_high;

		create_new_doublebuffer_pixmap(skw);
	}
	else {
		#ifdef WIDGET_DEBUG
		g_print("Calc Sizes - Resize\n");
		#endif

		skw->sizes[LEFT_RIGHT_WIDGET_BORDER] =
			MAX(0, (GTK_WIDGET(skw)->allocation.width - skw->sizes[WIDGET_MIN_WIDTH]) / 2);

		skw->sizes[TOP_BOTTOM_WIDGET_BORDER] =
			MAX(0, (GTK_WIDGET(skw)->allocation.height - skw->sizes[WIDGET_MIN_HEIGHT]) / 2);
	}
}

gdouble singit_karaoke_widget_get_frames_per_second (SingitKaraokeWidget *skw)
{
	#ifdef WIDGET_DEBUG
	g_print("FPS\n");
	#endif

	g_return_val_if_fail(skw != NULL, 0.0);
	g_return_val_if_fail(IS_SINGIT_KARAOKE_WIDGET(skw), 0.0);

	return skw->fps;
}
