/*
 *  SingIt Lyrics Displayer
 *  Copyright (C) 2000 - 2003 Jan-Marek Glogowski <glogow@stud.fbi.fh-darmstadt.de>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */


#include <gtk/gtk.h>
#include <gdk/gdkkeysyms.h>

#include <sys/types.h>
#include <sys/stat.h>
#include <dirent.h>
#include <unistd.h>
#include <stdio.h>
#include <string.h>

#include <xmms/plugin.h>
#include <xmms/xmmsctrl.h>

#include "singit_debug.h"
#include "singit/config_gen.h"
#include "singit/macros.h"
#include "singit_tools.h"

#include "list_genres.h"
#include "list_languages.h"
#include "singit_id3lib_wrapper.h"
#include "singit_tag_manager.h"
#include "singit_tag_manager_private.h"
#include "mpeg_header_info.h"

static GtkWidget *singit_tag_manager_win = NULL;

static GtkWidget *id3_main_vbox,
		 *id3_file_name_entry,
		 *id3_notebook,
			*id3_v1_vbox, *id3_mode_v1_label,
				*id3_mode_frame, *id3_modes_table,
					*id3_v1_select_radiobutton,
  					*id3_v2_select_radiobutton,
					*id3_mode_vseparator,
					*id3_mode_strict_radiobutton,
					*id3_mode_auto_radiobutton,
				*id3_v1_frame, *id3_v1_table,
					*id3_v1_title_label, *id3_v1_title_entry, *id3_v1_title_ind_button,
					*id3_v1_artist_label, *id3_v1_artist_entry, *id3_v1_artist_ind_button,
					*id3_v1_album_label, *id3_v1_album_entry, *id3_v1_album_ind_button,
					*id3_v1_comment_label, *id3_v1_comment_entry, *id3_v1_comment_ind_button,
					*id3_v1_year_track_hbox,
						*id3_v1_year_label, *id3_v1_year_entry,
						*id3_v1_track_label, *id3_v1_track_spinbutton,
					*id3_v1_genre_label, *id3_v1_genre_combo, *id3_v1_genre_combo_entry,
			*id3_v2_table, *id3_v2_label,
				*id3_v2_isrc_label, *id3_v2_isrc_entry,
				*id3_v2_language_label, *id3_v2_language_combo, *id3_v2_language_combo_entry,
				*id3_v2_lyric_writers_label, *id3_v2_lyric_writers_scrolledwindow, *id3_v2_lyric_writers_text,
  				*id3_v2_comment_label, *id3_v2_comment_edit_vbox,
					*id3_v2_comment_scrolledwindow, *id3_v2_comment_text,
					*id3_v2_comment_language_combo, *id3_v2_comment_language_combo_entry,
		*id3_button_hbox,
			*id3_save_button,
			*id3_remove_button,
			*id3_cancel_button;

static GtkTooltips *tooltips;
static GtkObject *id3_v1_track_spinbutton_adj;
static GSList	*id3_modes_group = NULL,
		*id3_strict_modes_group = NULL;

Id3DlgStatus id3dlg_status;

extern VisPlugin singit_vp;

static void singit_language_entry_event(GtkEntry *entry, gpointer user_data)
{
	gchar *text, *language;
	static gboolean is_running = FALSE;
	static gint last_length = -1;
//	GList *item = GTK_LIST(GTK_COMBO(user_data)->list)->children;
	guint i = 0, str_len;

	if (is_running) { return; }
	else { is_running = TRUE; }

#ifdef CODEDEBUG
	DEBUG(DLV_ALL, ("dlg_singit_tag_manager.c [singit_language_entry_event]\n"));
#endif

	text = gtk_entry_get_text (entry);
	str_len = strlen(text);
	if ((gint) str_len < last_length) {
		last_length = str_len;
		is_running = FALSE;
		return;
	}
	last_length = gtk_editable_get_position(GTK_EDITABLE(entry));

//	list_languages[i].language_string
	while (i < list_languages_count) {
		if (strlen(list_languages[i].language_string) >= strlen(text)) {
			language = g_strndup(list_languages[i].language_string, strlen(text));
			if (g_strcasecmp(language, text) == 0) {
				gtk_entry_set_text(GTK_ENTRY(entry), list_languages[i].language_string);
				gtk_editable_set_position(GTK_EDITABLE(entry), str_len);
//				printf("language-part : %s / %i \n", language, str_len);
				gtk_editable_select_region (GTK_EDITABLE(entry),
				       str_len, -1);
				g_free(language);
				is_running = FALSE;
				return;
			}
			g_free(language);
		}
		i++;
	}
	is_running = FALSE;
}

static void id3_v1_entry_changed_event (GtkEntry *entry, gpointer user_data)
{
	gchar* text;
	text = gtk_entry_get_text (entry);
	gtk_widget_set_sensitive (GTK_WIDGET(user_data), (strlen(text) > 30));
}

static void id3_v1_ind_button_clicked (GtkButton *button, gpointer user_data)
{
	gchar* text;
	text = g_strndup(gtk_entry_get_text(GTK_ENTRY(user_data)), 30);
	gtk_entry_set_text(GTK_ENTRY(user_data), text);
	gtk_widget_set_sensitive (GTK_WIDGET(button), FALSE);
	g_free(text);
}

static void id3_mode_changed_button_clicked_event (GtkButton *button, gpointer user_data)
{
	switch ((int) user_data) {
	case 0:
		id3dlg_status.useId1 = TRUE;
		if (id3dlg_status.beStrict) {
			gtk_entry_set_max_length(GTK_ENTRY(id3_v1_title_entry), 30);
			gtk_entry_set_max_length(GTK_ENTRY(id3_v1_artist_entry), 30);
			gtk_entry_set_max_length(GTK_ENTRY(id3_v1_album_entry), 30);
			gtk_entry_set_max_length(GTK_ENTRY(id3_v1_comment_entry), 30);
		}
		break;
	case 1:
		id3dlg_status.useId1 = FALSE;
		gtk_entry_set_max_length(GTK_ENTRY(id3_v1_title_entry), 0);
		gtk_entry_set_max_length(GTK_ENTRY(id3_v1_artist_entry), 0);
		gtk_entry_set_max_length(GTK_ENTRY(id3_v1_album_entry), 0);
		gtk_entry_set_max_length(GTK_ENTRY(id3_v1_comment_entry), 0);
		break;
	case 2:
		id3dlg_status.beStrict = TRUE;
		if (id3dlg_status.useId1) {
			gtk_entry_set_max_length(GTK_ENTRY(id3_v1_title_entry), 30);
			gtk_entry_set_max_length(GTK_ENTRY(id3_v1_artist_entry), 30);
			gtk_entry_set_max_length(GTK_ENTRY(id3_v1_album_entry), 30);
			gtk_entry_set_max_length(GTK_ENTRY(id3_v1_comment_entry), 30);
		}
		break;
	case 3:
		id3dlg_status.beStrict = FALSE;
		gtk_entry_set_max_length(GTK_ENTRY(id3_v1_title_entry), 0);
		gtk_entry_set_max_length(GTK_ENTRY(id3_v1_artist_entry), 0);
		gtk_entry_set_max_length(GTK_ENTRY(id3_v1_album_entry), 0);
		gtk_entry_set_max_length(GTK_ENTRY(id3_v1_comment_entry), 0);
		break;
	}
}

static gboolean on_id3_v1_year_entry_focus_out_event (GtkWidget *widget, GdkEventFocus *event, gpointer user_data)
{
	return TRUE;
}

static void fill_genre_combo (GtkWidget *combo)
{
	int i = 0;
	static GList *items = NULL;

	if (items == NULL) {
		while (list_genres[i] != NULL) {
			items = g_list_append(items,
				(gchar*) list_genres[i]);
			i++;
		}
	}
	gtk_combo_set_popdown_strings (GTK_COMBO (combo), items);
}

static void fill_language_combo (GtkWidget *combo)
{
	gint i;
	static GList *items = NULL;

	if (items == NULL) {
		for (i = 0; i < list_languages_count; i++) {
			items = g_list_append(items,
				(gchar*) list_languages[i].language_string);
		}
	}
	gtk_combo_set_popdown_strings (GTK_COMBO (combo), items);
}

static gboolean singit_tag_manager_set_entry_from_field (ID3Frame *id3_frame, GtkEntry *entry_widget)
{
	gchar *dyn_text;
	luint num_chars;
	ID3Field *id3_field;

	if ((id3_field = ID3Frame_GetField(id3_frame, ID3FN_TEXT)) != NULL) {
		num_chars = ID3Field_Size(id3_field);
		dyn_text = g_malloc(num_chars + 1);
		dyn_text[num_chars] = '\0';
		ID3Field_GetASCII_WRP(id3_field, dyn_text, num_chars);
		gtk_entry_set_text(entry_widget, dyn_text);
		g_free(dyn_text);
		return TRUE;
	}
	return FALSE;
}

static gboolean singit_tag_manager_set_genre (ID3Frame *id3_frame, GtkCombo *combo_widget)
{
	gchar *dyn_text;
	luint num_chars;
	ID3Field *id3_field;
	guint genre_nbr = 0;

	if ((id3_field = ID3Frame_GetField(id3_frame, ID3FN_TEXT)) != NULL) {
		num_chars = ID3Field_Size(id3_field);
		dyn_text = g_malloc(num_chars + 1);
		ID3Field_GetASCII_WRP(id3_field, dyn_text, num_chars);
		dyn_text[num_chars-1] = '\0';
		genre_nbr = atoi(dyn_text+1);
//		printf("%i : %s\n", genre_nbr, list_genres[genre_nbr]);
		gtk_list_select_item (GTK_LIST(combo_widget->list), genre_nbr+1);
		g_free(dyn_text);
		return TRUE;
	}
	return FALSE;
}

#define MAX_STR_LEN 100
/*
static void label_set_text(GtkWidget * label, char *str, ...)
{
	va_list args;
	gchar tempstr[MAX_STR_LEN];

	va_start(args, str);
	g_vsnprintf(tempstr, MAX_STR_LEN, str, args);
	va_end(args);

	gtk_label_set_text(GTK_LABEL(label), tempstr);
}
*/
/*
static void file_info_http(char *filename)
{
	gtk_widget_set_sensitive(id3_frame, FALSE);
	if (mpg123_filename && !strcmp(filename, mpg123_filename) &&
	    mpg123_bitrate != 0)
	{
		set_mpeg_level_label(mpg123_mpeg25, mpg123_lsf, mpg123_layer);
		label_set_text(mpeg_bitrate, _("Bitrate: %d kb/s"), mpg123_bitrate);
		label_set_text(mpeg_samplerate, _("Samplerate: %ld Hz"), mpg123_frequency);
		label_set_text(mpeg_flags, "%s", channel_mode_name(mpg123_mode));
	}
}
*/

static void fill_from_id3_info()
{
//	size_t nItems, nDataSize, nSymbol, nTimestamp, len, bytesleft, bytestoparse;
//	char *sText, *sDesc, *sLang, *sMimeType, *sFileName, *sOwner, *format;
//	uchar* bin, p;
	ID3Frame *id3_frame;
	ID3_FrameID id3_frame_id;
	ID3Field *id3_field;
	gint parsed_num = 0;
	gchar *dyn_text, *pos;
	guint num_chars;
#if (ID3LIB_MAJOR >= 3) && (ID3LIB_MINOR >= 8)
	ID3TagIterator *frame_iter;
#else
	size_t nFrames
#endif

#if (ID3LIB_MAJOR >= 3) && (ID3LIB_MINOR >= 8)
	frame_iter = ID3Tag_CreateIterator(id3dlg_status.id3tag);
	while ((id3_frame = ID3TagIterator_GetNext(frame_iter)) != NULL) {
#else
	for (nFrames = 0; nFrames < ID3Tag_NumFrames(id3dlg_status.id3tag); nFrames++) {
		id3_frame = ID3Tag_GetFrameNum(id3dlg_status.id3tag, nFrames);
#endif
		if (NULL != id3_frame) {
			id3_frame_id = ID3Frame_GetID(id3_frame);
			switch (id3_frame_id) {
			case ID3FID_ALBUM:
				singit_tag_manager_set_entry_from_field
					(id3_frame, GTK_ENTRY(id3_v1_album_entry));
				break;
//			case ID3FID_COMPOSER:
			case ID3FID_CONTENTTYPE:
				singit_tag_manager_set_genre(id3_frame, GTK_COMBO(id3_v1_genre_combo));
				break;
//			case ID3FID_DATE:
//			case ID3FID_PLAYLISTDELAY:
//			case ID3FID_ENCODEDBY:
//			case ID3FID_LYRICIST:
//			case ID3FID_CONTENTGROUP:
			case ID3FID_TITLE:
				singit_tag_manager_set_entry_from_field
					(id3_frame, GTK_ENTRY(id3_v1_title_entry));
				break;
//			case ID3FID_SUBTITLE:
//			case ID3FID_INITIALKEY:
			case ID3FID_LANGUAGE:
				if ((id3_field = ID3Frame_GetField(id3_frame, ID3FN_LANGUAGE)) != NULL) {
/*					num_chars = ID3Field_Size(id3_field);
					dyn_text = g_malloc(num_chars + 1);
					ID3Field_GetASCII_WRP(id3_field, dyn_text, num_chars, 0);
					gtk_spin_button_set_value(GTK_SPIN_BUTTON(id3_v2_track_spinbutton), parsed_num);
					g_free(dyn_text);
					gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(id3_v2_track_checkbutton), TRUE);*/
				}
				break;
			case ID3FID_LEADARTIST: singit_tag_manager_set_entry_from_field(id3_frame, GTK_ENTRY(id3_v1_artist_entry)); break;
//			case ID3FID_BAND:
//			case ID3FID_CONDUCTOR:
//			case ID3FID_PARTINSET:
//			case ID3FID_PUBLISHER:
			case ID3FID_TRACKNUM:
				if ((id3_field = ID3Frame_GetField(id3_frame, ID3FN_TEXT)) != NULL) {
					num_chars = ID3Field_Size(id3_field);
					dyn_text = g_malloc(num_chars + 1);
					ID3Field_GetASCII_WRP(id3_field, dyn_text, num_chars);
					// dyn_text[num_chars-1] = '\0';
					pos = strstr(dyn_text, "/");
					if (pos == NULL) {
						parsed_num = atoi(dyn_text);
					}
					else {
						pos = '\0';
						parsed_num = atoi(dyn_text);

					}
					gtk_spin_button_set_value(GTK_SPIN_BUTTON(id3_v1_track_spinbutton), parsed_num);
					g_free(dyn_text);
				}
				break;
			case ID3FID_ISRC:
				singit_tag_manager_set_entry_from_field(id3_frame, GTK_ENTRY(id3_v2_isrc_entry));
				break;
			case ID3FID_YEAR:
				singit_tag_manager_set_entry_from_field(id3_frame, GTK_ENTRY(id3_v1_year_entry));
				break;
			case ID3FID_USERTEXT:
/*				sText = ID3_GetString(myFrame, ID3FN_TEXT),
				sDesc = ID3_GetString(myFrame, ID3FN_DESCRIPTION);
				delete [] sText;
				delete [] sDesc;*/
				break;
			case ID3FID_COMMENT:
				singit_tag_manager_set_entry_from_field(id3_frame, GTK_ENTRY(id3_v1_comment_entry));
				break;
			case ID3FID_UNSYNCEDLYRICS:
/*				sText = ID3_GetString(myFrame, ID3FN_TEXT),
				sDesc = ID3_GetString(myFrame, ID3FN_DESCRIPTION),
				sLang = ID3_GetString(myFrame, ID3FN_LANGUAGE);
				g_free(sText);
				delete [] sDesc;
				delete [] sLang;*/
				break;
			case ID3FID_INVOLVEDPEOPLE:
/*				nItems = myFrame->Field(ID3FN_TEXT).GetNumTextItems();
				for (size_t nIndex = 0; nIndex < nItems; nIndex++) {
					char *sPeople = ID3_GetString(myFrame, ID3FN_TEXT, nIndex);
					delete [] sPeople;
					if (nIndex + 1 < nItems) {
					// Fill
					}
				}*/
				break;
			case ID3FID_GENERALOBJECT:
/*				char *sMimeType = ID3_GetString(myFrame, ID3FN_TEXT),
				     *sDesc = ID3_GetString(myFrame, ID3FN_DESCRIPTION),
				     *sFileName = ID3_GetString(myFrame, ID3FN_FILENAME);
				nDataSize = ID3Field_Size(ID3Frame_GetField(myFrame, ID3FN_DATA));
				delete [] sMimeType;
				delete [] sDesc;
				delete [] sFileName;*/
				break;
			case ID3FID_UNIQUEFILEID:
/*				char *sOwner = ID3_GetString(myFrame, ID3FN_TEXT);
				nDataSize = ID3Field_Size(ID3Frame_GetField(myFrame, ID3FN_DATA));
				delete [] sOwner;*/
				break;
			case ID3FID_SYNCEDLYRICS:
/*				sDesc = ID3_GetString(myFrame, ID3FN_DESCRIPTION);
				sLang = ID3_GetString(myFrame, ID3FN_LANGUAGE);
				nTimestamp = myFrame->Field(ID3FN_TIMESTAMPFORMAT).Get();
				nRating = myFrame->Field(ID3FN_CONTENTTYPE).Get();
				format = (2 == nTimestamp) ? "ms" : "frames";
				switch (nRating) {
				case ID3CT_OTHER:    cout << "Other"; break;
				case ID3CT_LYRICS:   cout << "Lyrics"; break;
				case ID3CT_TEXTTRANSCRIPTION:     cout << "Text transcription"; break;
				case ID3CT_MOVEMENT: cout << "Movement/part name"; break;
				case ID3CT_EVENTS:   cout << "Events"; break;
				case ID3CT_CHORD:    cout << "Chord"; break;
				case ID3CT_TRIVIA:   cout << "Trivia/'pop up' information"; break;
				}
				nDataSize = ID3Field_Size(ID3Frame_GetField(myFrame, ID3FN_DATA));
				bin = myFrame->Field(ID3FN_DATA).GetBinary();
				p = bin;
				while (p < bin + nDataSize) {
					len = strlen((char *)p);
					cout << p;
					p += len + 1;
					if (p < bin + size) {
						bytesleft = bin + size - p,
						bytestoparse = MIN(sizeof(uint32), bytesleft);
						p += 4;
					}
				}
				delete [] sDesc;
				delete [] sLang;*/
				break;
			default:
			break;
			}
		}
	}

#if (ID3LIB_MAJOR >= 3) && (ID3LIB_MINOR >= 8)
	ID3TagIterator_Delete(frame_iter);
#endif
}

static gchar* get_comment(vorbis_comment *vc, const gchar *label)
{
	gchar *tag;
	if (vc && (tag = vorbis_comment_query(vc, (gchar*) label, 0)) != NULL)
		return convert_from_utf8(tag);
	else
		return g_strdup("");
}

static void fill_from_oggvorbis_info()
{
	vorbis_comment *vcomment = ov_comment(id3dlg_status.ov_file, -1);

	gtk_entry_set_text(GTK_ENTRY(id3_v1_title_entry), get_comment(vcomment, "title"));
	gtk_entry_set_text(GTK_ENTRY(id3_v1_artist_entry), get_comment(vcomment, "artist"));
	gtk_entry_set_text(GTK_ENTRY(id3_v1_album_entry), get_comment(vcomment, "album"));
	gtk_entry_set_text(GTK_ENTRY(id3_v1_year_entry), get_comment(vcomment, "date"));
	gtk_entry_set_text(GTK_ENTRY(id3_v1_track_spinbutton), get_comment(vcomment, "tracknumber"));
	gtk_spin_button_set_value(GTK_SPIN_BUTTON(id3_v1_track_spinbutton),
		atoi(get_comment(vcomment, "tracknumber")));

	gtk_entry_set_text(GTK_ENTRY(id3_v2_isrc_entry), get_comment(vcomment, "isrc"));
	gtk_entry_set_text(GTK_ENTRY(id3_v1_comment_entry), get_comment(vcomment, "comment"));
}

static void singit_tag_manager_fill_dialog(gchar *file_name)
{

#ifdef CODEDEBUG
	DEBUG(DLV_ALL, ("dlg_singit_tag_manager.c [singit_tag_manager_fill_dialog]\n"));
#endif

	gtk_entry_set_text(GTK_ENTRY(id3_file_name_entry), file_name);
	gtk_tooltips_set_tip(tooltips, id3_file_name_entry, file_name, NULL);

	fill_genre_combo(id3_v1_genre_combo);
	fill_language_combo(id3_v2_language_combo);
	fill_language_combo(id3_v2_comment_language_combo);

	switch (id3dlg_status.info_type) {
	case IT_VORBIS:
		fill_from_oggvorbis_info();
		break;
	case IT_MPEG:
		fill_from_id3_info();
		break;
	default:
		break;
	}
}

static void singit_tag_manager_init()
{
	SingitConfigGen *scg_o;

#ifdef CODEDEBUG
	DEBUG(DLV_ALL, ("dlg_singit_tag_manager.c [singit_tag_manager_init]\n"));
#endif

	id3dlg_status.id3v2Usage = 0;
	id3dlg_status.id3tag = NULL;

	id3dlg_status.ov_file = NULL;
	id3dlg_status.mpeg_info = NULL;

	id3dlg_status.info_type = IT_ID3;

	scg_o = STATUS->config;
	g_return_if_fail(scg_o != NULL);
	singit_config_gen_attach(scg_o);
}

static void singit_tag_manager_cleanup()
{
	SingitConfigGen *scg_o;
#ifdef CODEDEBUG
	DEBUG(DLV_ALL, ("singit_config.c [singit_tag_manager_cleanup]\n"));
#endif

	if (id3dlg_status.id3tag != NULL) {
		ID3Tag_Delete(id3dlg_status.id3tag);
		id3dlg_status.id3tag = NULL;
	}

	if (id3dlg_status.ov_file != NULL) {
		ov_clear(id3dlg_status.ov_file);
		id3dlg_status.ov_file = NULL;
	}

	if (id3dlg_status.mpeg_info != NULL) {
		mpeg_header_info_free(id3dlg_status.mpeg_info);
		id3dlg_status.mpeg_info = NULL;
	}

	scg_o = STATUS->config;
	singit_config_gen_detach(&scg_o);

	id3_modes_group = NULL;
	id3_strict_modes_group = NULL;
	id3dlg_status.id3v2Usage = 0;

	// Free static string
	transadd(NULL, NULL);
}

void singit_tag_manager_hide(void)
{
	if (!singit_tag_manager_win) return;

#ifdef CODEDEBUG
	DEBUG(DLV_ALL, ("singit_config.c [singit_tag_manager_hide]\n"));
#endif

	singit_tag_manager_cleanup();

	gtk_widget_destroy(singit_tag_manager_win);
}

static void singit_tag_manager_cancel_clicked(GtkButton *button, gpointer user_data)
{
	singit_tag_manager_cleanup();

	/* close and destroy window */
	gtk_widget_destroy(singit_tag_manager_win);
}

static gint dlg_singit_id3_delete_event( GtkWidget *widget, GdkEvent  *event, gpointer data )
{
	singit_tag_manager_cleanup();

	/* close and destroy window */
	gtk_widget_destroy(singit_tag_manager_win);

	return (FALSE);
}

#define id3_widget_descr(widget, widget_var_name, show) \
	create_std_wgt_description(widget, widget_var_name, show, singit_tag_manager_win)

static void singit_tag_manager_page_v1()
{
	id3_v1_vbox = gtk_vbox_new (FALSE, 3);
	id3_widget_descr(id3_v1_vbox, "id3_v1_vbox", TRUE);
	gtk_container_set_border_width (GTK_CONTAINER (id3_v1_vbox), 3);

	id3_mode_v1_label = gtk_label_new (_("id3 mode / v1.x"));
	id3_widget_descr (id3_mode_v1_label, "id3_mode_v1_label", TRUE);
	gtk_notebook_append_page(GTK_NOTEBOOK(id3_notebook), id3_v1_vbox, id3_mode_v1_label);

	id3_mode_frame = gtk_frame_new (_("id3 mode"));
	id3_widget_descr(id3_mode_frame, "id3_mode_frame", TRUE);
	gtk_box_pack_start (GTK_BOX (id3_v1_vbox), id3_mode_frame, FALSE, TRUE, 0);

	id3_modes_table = gtk_table_new (2, 3, FALSE);
	id3_widget_descr(id3_modes_table, "id3_modes_table", TRUE);
	gtk_container_set_border_width(GTK_CONTAINER (id3_modes_table), 5);
	gtk_container_add (GTK_CONTAINER (id3_mode_frame), id3_modes_table);
	gtk_table_set_row_spacings (GTK_TABLE (id3_modes_table), 2);
	gtk_table_set_col_spacings (GTK_TABLE (id3_modes_table), 2);

	id3_v1_select_radiobutton = gtk_radio_button_new_with_label (id3_modes_group, _("Use id3 tag v1"));
	id3_modes_group = gtk_radio_button_group (GTK_RADIO_BUTTON (id3_v1_select_radiobutton));
	id3_widget_descr(id3_v1_select_radiobutton, "id3_v1_select_radiobutton", TRUE);
	gtk_table_attach (GTK_TABLE (id3_modes_table), id3_v1_select_radiobutton, 0, 1, 0, 1,
		(GtkAttachOptions) (GTK_FILL), (GtkAttachOptions) (0), 0, 0);
	gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (id3_v1_select_radiobutton), TRUE);
  	gtk_signal_connect (GTK_OBJECT (id3_v1_select_radiobutton), "clicked",
		GTK_SIGNAL_FUNC (id3_mode_changed_button_clicked_event), (gpointer) 0);

	id3_v2_select_radiobutton = gtk_radio_button_new_with_label (id3_modes_group, _("Use id3 tag v2.x.x"));
	id3_modes_group = gtk_radio_button_group (GTK_RADIO_BUTTON (id3_v2_select_radiobutton));
	id3_widget_descr(id3_v2_select_radiobutton, "id3_v2_select_radiobutton", TRUE);
	gtk_table_attach (GTK_TABLE (id3_modes_table), id3_v2_select_radiobutton, 0, 1, 1, 2,
		(GtkAttachOptions) (GTK_FILL), (GtkAttachOptions) (0), 0, 0);
	gtk_signal_connect (GTK_OBJECT (id3_v2_select_radiobutton), "clicked",
		GTK_SIGNAL_FUNC (id3_mode_changed_button_clicked_event), (gpointer) 1);

	id3_mode_vseparator = gtk_vseparator_new ();
	id3_widget_descr(id3_mode_vseparator, "id3_mode_vseparator", TRUE);
	gtk_table_attach (GTK_TABLE (id3_modes_table), id3_mode_vseparator, 1, 2, 0, 2,
		(GtkAttachOptions) (GTK_EXPAND | GTK_FILL), (GtkAttachOptions) (GTK_FILL), 0, 0);

	id3_mode_strict_radiobutton = gtk_radio_button_new_with_label (id3_strict_modes_group, _("Be strict"));
	id3_strict_modes_group = gtk_radio_button_group (GTK_RADIO_BUTTON (id3_mode_strict_radiobutton));
	id3_widget_descr(id3_mode_strict_radiobutton, "id3_mode_strict_radiobutton", TRUE);
	gtk_table_attach (GTK_TABLE (id3_modes_table), id3_mode_strict_radiobutton, 2, 3, 0, 1,
		(GtkAttachOptions) (GTK_FILL), (GtkAttachOptions) (0), 0, 0);
	gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (id3_mode_strict_radiobutton), TRUE);
	gtk_signal_connect (GTK_OBJECT (id3_mode_strict_radiobutton), "clicked",
		GTK_SIGNAL_FUNC (id3_mode_changed_button_clicked_event), (gpointer) 2);

	id3_mode_auto_radiobutton = gtk_radio_button_new_with_label (id3_strict_modes_group, _("Allow auto mode change"));
	id3_strict_modes_group = gtk_radio_button_group (GTK_RADIO_BUTTON (id3_mode_auto_radiobutton));
	id3_widget_descr(id3_mode_auto_radiobutton, "id3_mode_auto_radiobutton", TRUE);
	gtk_table_attach (GTK_TABLE (id3_modes_table), id3_mode_auto_radiobutton, 2, 3, 1, 2,
		(GtkAttachOptions) (GTK_FILL), (GtkAttachOptions) (0), 0, 0);
	gtk_signal_connect (GTK_OBJECT (id3_mode_auto_radiobutton), "clicked",
		GTK_SIGNAL_FUNC (id3_mode_changed_button_clicked_event), (gpointer) 3);

	id3_v1_frame = gtk_frame_new (_("id3v1"));
	id3_widget_descr(id3_v1_frame, "id3_v1_frame", TRUE);
	gtk_box_pack_start (GTK_BOX (id3_v1_vbox), id3_v1_frame, FALSE, TRUE, 0);

	id3_v1_table = gtk_table_new (5, 3, FALSE);
	id3_widget_descr(id3_v1_table, "id3_v1_table", TRUE);
	gtk_container_add (GTK_CONTAINER (id3_v1_frame), id3_v1_table);
	gtk_container_set_border_width (GTK_CONTAINER (id3_v1_table), 2);
	gtk_table_set_row_spacings (GTK_TABLE (id3_v1_table), 4);
	gtk_table_set_col_spacings (GTK_TABLE (id3_v1_table), 4);

	id3_v1_title_label = gtk_label_new (transadd(_("Title"), ":"));
	id3_widget_descr(id3_v1_title_label, "id3_v1_title_label", TRUE);
	gtk_table_attach (GTK_TABLE (id3_v1_table), id3_v1_title_label, 0, 1, 0, 1,
		(GtkAttachOptions) (GTK_EXPAND | GTK_FILL), (GtkAttachOptions) (GTK_EXPAND | GTK_FILL), 0, 0);
	gtk_misc_set_alignment (GTK_MISC (id3_v1_title_label), 1, 0.5);

	id3_v1_title_entry = gtk_entry_new ();
	id3_widget_descr (id3_v1_title_entry, "id3_v1_title_entry", TRUE);
	gtk_table_attach (GTK_TABLE (id3_v1_table), id3_v1_title_entry, 1, 2, 0, 1,
		(GtkAttachOptions) (GTK_EXPAND | GTK_FILL), (GtkAttachOptions) (0), 0, 0);

	id3_v1_artist_label = gtk_label_new (transadd(_("Artist"), ":"));
	id3_widget_descr(id3_v1_artist_label, "id3_v1_artist_label", TRUE);
	gtk_table_attach (GTK_TABLE (id3_v1_table), id3_v1_artist_label, 0, 1, 1, 2,
		(GtkAttachOptions) (GTK_EXPAND | GTK_FILL), (GtkAttachOptions) (0), 0, 0);
	gtk_misc_set_alignment (GTK_MISC (id3_v1_artist_label), 1, 0.5);

	id3_v1_artist_entry = gtk_entry_new ();
	id3_widget_descr (id3_v1_artist_entry, "id3_v1_artist_entry", TRUE);
	gtk_table_attach (GTK_TABLE (id3_v1_table), id3_v1_artist_entry, 1, 2, 1, 2,
		(GtkAttachOptions) (GTK_EXPAND | GTK_FILL),
		(GtkAttachOptions) (0), 0, 0);

	id3_v1_album_label = gtk_label_new (transadd(_("Album"), ":"));
	id3_widget_descr(id3_v1_album_label, "id3_v1_album_label", TRUE);
	gtk_table_attach (GTK_TABLE (id3_v1_table), id3_v1_album_label, 0, 1, 2, 3,
		(GtkAttachOptions) (GTK_EXPAND | GTK_FILL), (GtkAttachOptions) (0), 0, 0);
	gtk_misc_set_alignment (GTK_MISC (id3_v1_album_label), 1, 0.5);

	id3_v1_album_entry = gtk_entry_new ();
	id3_widget_descr (id3_v1_album_entry, "id3_v1_album_entry", TRUE);
	gtk_table_attach (GTK_TABLE (id3_v1_table), id3_v1_album_entry, 1, 2, 2, 3,
		(GtkAttachOptions) (GTK_EXPAND | GTK_FILL),
		(GtkAttachOptions) (0), 0, 0);

	id3_v1_comment_label = gtk_label_new (transadd(_("Comment"), ":"));
	id3_widget_descr(id3_v1_comment_label, "id3_v1_comment_label", TRUE);
	gtk_table_attach (GTK_TABLE (id3_v1_table), id3_v1_comment_label, 0, 1, 3, 4,
		(GtkAttachOptions) (GTK_FILL), (GtkAttachOptions) (GTK_FILL), 0, 0);
	gtk_misc_set_alignment (GTK_MISC (id3_v1_comment_label), 1, 0.5);

	id3_v1_comment_entry = gtk_entry_new ();
	id3_widget_descr (id3_v1_comment_entry, "id3_v1_comment_entry", TRUE);
	gtk_table_attach (GTK_TABLE (id3_v1_table), id3_v1_comment_entry, 1, 2, 3, 4,
		(GtkAttachOptions) (GTK_EXPAND | GTK_FILL), (GtkAttachOptions) (0), 0, 0);
	gtk_tooltips_set_tip (tooltips, id3_v1_comment_entry, _("v1: Comment / v2: comment content"), NULL);

	id3_v1_year_label = gtk_label_new (transadd(_("Year"), ":"));
	id3_widget_descr (id3_v1_year_label, "id3_v1_year_label", TRUE);
	gtk_table_attach (GTK_TABLE (id3_v1_table), id3_v1_year_label, 0, 1, 4, 5,
		(GtkAttachOptions) (GTK_FILL), (GtkAttachOptions) (GTK_FILL), 0, 0);
	gtk_misc_set_alignment (GTK_MISC (id3_v1_year_label), 1, 0.5);

	id3_v1_year_track_hbox = gtk_hbox_new (FALSE, 2);
	id3_widget_descr (id3_v1_year_track_hbox, "id3_v1_year_track_hbox", TRUE);
	gtk_table_attach (GTK_TABLE (id3_v1_table), id3_v1_year_track_hbox, 1, 2, 4, 5,
		(GtkAttachOptions) (GTK_FILL),
		(GtkAttachOptions) (GTK_EXPAND | GTK_FILL), 0, 0);

	id3_v1_year_entry = gtk_entry_new_with_max_length (4);
	id3_widget_descr (id3_v1_year_entry, "id3_v1_year_entry", TRUE);
	gtk_box_pack_start (GTK_BOX (id3_v1_year_track_hbox), id3_v1_year_entry, FALSE, FALSE, 0);
	gtk_widget_set_usize (id3_v1_year_entry, 57, -2);

	id3_v1_track_label = gtk_label_new (transadd(_("Track"), " #"));
	id3_widget_descr (id3_v1_track_label, "id3_v1_track_label", TRUE);
	gtk_box_pack_start (GTK_BOX (id3_v1_year_track_hbox), id3_v1_track_label, TRUE, TRUE, 0);
	gtk_misc_set_alignment (GTK_MISC (id3_v1_track_label), 1, 0.5);

	id3_v1_track_spinbutton_adj = gtk_adjustment_new (0, 0, 99, 1, 10, 10);
	id3_v1_track_spinbutton = gtk_spin_button_new (GTK_ADJUSTMENT (id3_v1_track_spinbutton_adj), 1, 0);
	id3_widget_descr (id3_v1_track_spinbutton, "id3_v1_track_spinbutton", TRUE);
	gtk_box_pack_start (GTK_BOX (id3_v1_year_track_hbox), id3_v1_track_spinbutton, FALSE, FALSE, 0);

	id3_v1_genre_label = gtk_label_new (transadd(_("Genre"), ":"));
	id3_widget_descr (id3_v1_genre_label, "id3_v1_genre_label", TRUE);
	gtk_table_attach (GTK_TABLE (id3_v1_table), id3_v1_genre_label, 0, 1, 5, 6,
		(GtkAttachOptions) (GTK_FILL),
		(GtkAttachOptions) (GTK_EXPAND | GTK_FILL), 0, 0);
	gtk_misc_set_alignment (GTK_MISC (id3_v1_genre_label), 1, 0.5);

	id3_v1_genre_combo = gtk_combo_new ();
	id3_widget_descr (id3_v1_genre_combo, "id3_v1_genre_combo", TRUE);
	gtk_table_attach (GTK_TABLE (id3_v1_table), id3_v1_genre_combo, 1, 2, 5, 6,
		(GtkAttachOptions) (GTK_FILL),
		(GtkAttachOptions) (GTK_EXPAND | GTK_FILL), 0, 0);

	id3_v1_genre_combo_entry = GTK_COMBO (id3_v1_genre_combo)->entry;
	id3_widget_descr (id3_v1_genre_combo_entry, "id3_v1_genre_combo_entry", TRUE);

	id3_v1_title_ind_button = gtk_button_new_with_label (_("v1"));
	id3_widget_descr (id3_v1_title_ind_button, "id3_v1_title_ind_button", TRUE);
	gtk_table_attach (GTK_TABLE (id3_v1_table), id3_v1_title_ind_button, 2, 3, 0, 1,
		(GtkAttachOptions) (0),
		(GtkAttachOptions) (0), 0, 0);
	gtk_widget_set_sensitive (id3_v1_title_ind_button, FALSE);
	gtk_tooltips_set_tip (tooltips, id3_v1_title_ind_button, _("Reduce size to id3v1 (30)"), NULL);

	id3_v1_artist_ind_button = gtk_button_new_with_label (_("v1"));
	id3_widget_descr (id3_v1_artist_ind_button, "id3_v1_artist_ind_button", TRUE);
	gtk_table_attach (GTK_TABLE (id3_v1_table), id3_v1_artist_ind_button, 2, 3, 1, 2,
		(GtkAttachOptions) (0),
		(GtkAttachOptions) (0), 0, 0);
	gtk_widget_set_sensitive (id3_v1_artist_ind_button, FALSE);
	gtk_tooltips_set_tip (tooltips, id3_v1_artist_ind_button, _("Reduce size to id3v1 (30)"), NULL);

	id3_v1_album_ind_button = gtk_button_new_with_label (_("v1"));
	id3_widget_descr (id3_v1_album_ind_button, "id3_v1_album_ind_button", TRUE);
	gtk_table_attach (GTK_TABLE (id3_v1_table), id3_v1_album_ind_button, 2, 3, 2, 3,
		(GtkAttachOptions) (0),
		(GtkAttachOptions) (0), 0, 0);
	gtk_widget_set_sensitive (id3_v1_album_ind_button, FALSE);
	gtk_tooltips_set_tip (tooltips, id3_v1_album_ind_button, _("Reduce size to id3v1 (30)"), NULL);

	id3_v1_comment_ind_button = gtk_button_new_with_label (_("v1"));
	id3_widget_descr (id3_v1_comment_ind_button, "id3_v1_comment_ind_button", TRUE);
	gtk_table_attach (GTK_TABLE (id3_v1_table), id3_v1_comment_ind_button, 2, 3, 4, 5,
		(GtkAttachOptions) (0),
		(GtkAttachOptions) (0), 0, 0);
	gtk_widget_set_sensitive (id3_v1_comment_ind_button, FALSE);
	gtk_tooltips_set_tip (tooltips, id3_v1_comment_ind_button, _("Reduce size to id3v1 (30)"), NULL);

	gtk_entry_set_max_length(GTK_ENTRY(id3_v1_title_entry), 30);
	gtk_entry_set_max_length(GTK_ENTRY(id3_v1_artist_entry), 30);
	gtk_entry_set_max_length(GTK_ENTRY(id3_v1_album_entry), 30);
	gtk_entry_set_max_length(GTK_ENTRY(id3_v1_comment_entry), 30);

	gtk_signal_connect (GTK_OBJECT (id3_v1_title_entry), "changed",
		GTK_SIGNAL_FUNC (id3_v1_entry_changed_event), id3_v1_title_ind_button);
	gtk_signal_connect (GTK_OBJECT (id3_v1_artist_entry), "changed",
		GTK_SIGNAL_FUNC (id3_v1_entry_changed_event), id3_v1_artist_ind_button);
	gtk_signal_connect (GTK_OBJECT (id3_v1_album_entry), "changed",
		GTK_SIGNAL_FUNC (id3_v1_entry_changed_event), id3_v1_album_ind_button);
	gtk_signal_connect (GTK_OBJECT (id3_v1_comment_entry), "changed",
		GTK_SIGNAL_FUNC (id3_v1_entry_changed_event), id3_v1_comment_ind_button);

	gtk_signal_connect (GTK_OBJECT (id3_v1_title_ind_button), "clicked",
		GTK_SIGNAL_FUNC (id3_v1_ind_button_clicked), id3_v1_title_entry);
	gtk_signal_connect (GTK_OBJECT (id3_v1_artist_ind_button), "clicked",
		GTK_SIGNAL_FUNC (id3_v1_ind_button_clicked), id3_v1_artist_entry);
	gtk_signal_connect (GTK_OBJECT (id3_v1_album_ind_button), "clicked",
		GTK_SIGNAL_FUNC (id3_v1_ind_button_clicked), id3_v1_album_entry);
	gtk_signal_connect (GTK_OBJECT (id3_v1_comment_ind_button), "clicked",
		GTK_SIGNAL_FUNC (id3_v1_ind_button_clicked), id3_v1_comment_entry);

	gtk_signal_connect (GTK_OBJECT (id3_v1_year_entry), "focus_out_event",
		GTK_SIGNAL_FUNC (on_id3_v1_year_entry_focus_out_event), NULL);
}

static void singit_tag_manager_page_v2()
{
	id3_v2_table = gtk_table_new (5, 2, FALSE);
	id3_widget_descr (id3_v2_table, "id3_v2_table", TRUE);
	gtk_container_set_border_width (GTK_CONTAINER (id3_v2_table), 4);
	gtk_table_set_row_spacings (GTK_TABLE (id3_v2_table), 3);
	gtk_table_set_col_spacings (GTK_TABLE (id3_v2_table), 2);

	id3_v2_label = gtk_label_new (_("id3v2.x.x"));
	id3_widget_descr (id3_v2_label, "id3_v2_label", TRUE);
	gtk_notebook_append_page(GTK_NOTEBOOK(id3_notebook), id3_v2_table, id3_v2_label);

	id3_v2_isrc_entry = gtk_entry_new_with_max_length (12);
	id3_widget_descr (id3_v2_isrc_entry, "id3_v2_isrc_entry", TRUE);
	gtk_table_attach (GTK_TABLE (id3_v2_table), id3_v2_isrc_entry, 1, 2, 2, 3,
		(GtkAttachOptions) (GTK_EXPAND | GTK_FILL),
		(GtkAttachOptions) (0), 0, 0);
	gtk_tooltips_set_tip (tooltips, id3_v2_isrc_entry,
		_("International Standard Recording Code"), NULL);

	id3_v2_language_combo = gtk_combo_new ();
	id3_widget_descr (id3_v2_language_combo, "id3_v2_language_combo", TRUE);
	gtk_table_attach (GTK_TABLE (id3_v2_table), id3_v2_language_combo, 1, 2, 0, 1,
		(GtkAttachOptions) (GTK_EXPAND | GTK_FILL), (GtkAttachOptions) (0), 0, 0);

	id3_v2_language_combo_entry = GTK_COMBO (id3_v2_language_combo)->entry;
	id3_widget_descr (id3_v2_language_combo_entry, "id3_v2_language_combo_entry", FALSE);
	gtk_signal_connect(GTK_OBJECT(id3_v2_language_combo_entry), "changed",
		GTK_SIGNAL_FUNC(singit_language_entry_event),
		(gpointer) id3_v2_language_combo);
	gtk_widget_show (id3_v2_language_combo_entry);

	id3_v2_language_label = gtk_label_new (transadd(_("Language"), ":"));
	id3_widget_descr (id3_v2_language_label, "id3_v2_language_label", TRUE);
	gtk_table_attach (GTK_TABLE (id3_v2_table), id3_v2_language_label, 0, 1, 0, 1,
		(GtkAttachOptions) (GTK_FILL), (GtkAttachOptions) (0), 0, 0);
	gtk_misc_set_alignment (GTK_MISC (id3_v2_language_label), 1, 0.5);

	id3_v2_isrc_label = gtk_label_new (transadd(_("ISRC"), ":"));
	id3_widget_descr (id3_v2_isrc_label, "id3_v2_isrc_label", TRUE);
	gtk_table_attach (GTK_TABLE (id3_v2_table), id3_v2_isrc_label, 0, 1, 2, 3,
		(GtkAttachOptions) (GTK_FILL), (GtkAttachOptions) (0), 0, 0);
	gtk_misc_set_alignment (GTK_MISC (id3_v2_isrc_label), 1, 0.5);

	id3_v2_lyric_writers_label = gtk_label_new (transadd(_("Lyric\nwriters"), ":"));
	gtk_label_set_justify(GTK_LABEL(id3_v2_lyric_writers_label), GTK_JUSTIFY_LEFT);
	id3_widget_descr (id3_v2_lyric_writers_label, "id3_v2_lyric_writers_label", TRUE);
	gtk_table_attach (GTK_TABLE (id3_v2_table), id3_v2_lyric_writers_label, 0, 1, 3, 4,
		(GtkAttachOptions) (GTK_FILL),
		(GtkAttachOptions) (GTK_EXPAND | GTK_FILL), 0, 0);
	gtk_misc_set_alignment (GTK_MISC (id3_v2_lyric_writers_label), 1, 0.01);

	id3_v2_lyric_writers_scrolledwindow = gtk_scrolled_window_new (NULL, NULL);
	id3_widget_descr (id3_v2_lyric_writers_scrolledwindow, "id3_v2_lyric_writers_scrolledwindow", TRUE);
	gtk_table_attach (GTK_TABLE (id3_v2_table), id3_v2_lyric_writers_scrolledwindow, 1, 2, 3, 4,
		(GtkAttachOptions) (GTK_FILL),
		(GtkAttachOptions) (GTK_FILL), 0, 0);
	gtk_scrolled_window_set_policy (GTK_SCROLLED_WINDOW (id3_v2_lyric_writers_scrolledwindow), GTK_POLICY_NEVER, GTK_POLICY_ALWAYS);

	id3_v2_lyric_writers_text = gtk_text_new (NULL, NULL);
	id3_widget_descr (id3_v2_lyric_writers_text, "id3_v2_lyric_writers_text", TRUE);
	gtk_container_add (GTK_CONTAINER (id3_v2_lyric_writers_scrolledwindow), id3_v2_lyric_writers_text);
	gtk_tooltips_set_tip (tooltips, id3_v2_lyric_writers_text, _("One writer per line !!!"), NULL);
	gtk_text_set_editable (GTK_TEXT (id3_v2_lyric_writers_text), TRUE);

	id3_v2_comment_label = gtk_label_new (transadd(_("Comment"), ":"));
	id3_widget_descr (id3_v2_comment_label, "id3_v2_comment_label", TRUE);
	gtk_table_attach (GTK_TABLE (id3_v2_table), id3_v2_comment_label, 0, 1, 4, 5,
		(GtkAttachOptions) (GTK_FILL),
		(GtkAttachOptions) (GTK_EXPAND | GTK_FILL), 0, 0);
	gtk_misc_set_alignment (GTK_MISC (id3_v2_comment_label), 1, 0.01);

	id3_v2_comment_edit_vbox = gtk_vbox_new (FALSE, 0);
	id3_widget_descr (id3_v2_comment_edit_vbox, "id3_v2_comment_edit_vbox", TRUE);
	gtk_table_attach (GTK_TABLE (id3_v2_table), id3_v2_comment_edit_vbox, 1, 2, 4, 5,
		(GtkAttachOptions) (GTK_FILL),
		(GtkAttachOptions) (GTK_FILL), 0, 0);

	id3_v2_comment_scrolledwindow = gtk_scrolled_window_new (NULL, NULL);
	id3_widget_descr (id3_v2_comment_scrolledwindow, "id3_v2_comment_scrolledwindow", TRUE);
	gtk_box_pack_start (GTK_BOX (id3_v2_comment_edit_vbox), id3_v2_comment_scrolledwindow, TRUE, TRUE, 0);
	gtk_widget_set_usize (id3_v2_comment_scrolledwindow, -2, 60);
	gtk_scrolled_window_set_policy (GTK_SCROLLED_WINDOW (id3_v2_comment_scrolledwindow), GTK_POLICY_NEVER, GTK_POLICY_ALWAYS);

	id3_v2_comment_text = gtk_text_new (NULL, NULL);
	id3_widget_descr (id3_v2_comment_text, "id3_v2_comment_text", TRUE);
	gtk_container_add (GTK_CONTAINER (id3_v2_comment_scrolledwindow), id3_v2_comment_text);
	gtk_tooltips_set_tip (tooltips, id3_v2_comment_text, _("Comment long text"), NULL);
	gtk_text_set_editable (GTK_TEXT (id3_v2_comment_text), TRUE);

	id3_v2_comment_language_combo = gtk_combo_new ();
	id3_widget_descr (id3_v2_comment_language_combo, "id3_v2_comment_language_combo", TRUE);
	gtk_box_pack_start (GTK_BOX (id3_v2_comment_edit_vbox), id3_v2_comment_language_combo, FALSE, FALSE, 0);

	id3_v2_comment_language_combo_entry = GTK_COMBO (id3_v2_comment_language_combo)->entry;
	id3_widget_descr (id3_v2_comment_language_combo_entry, "id3_v2_comment_language_combo_entry", TRUE);
	gtk_tooltips_set_tip (tooltips, id3_v2_comment_language_combo_entry, _("Language of the comment"), NULL);
}

void singit_tag_manager_show(void)
{
	SingitSong *my_song;
	gchar *file_name;

#ifdef CODEDEBUG
	DEBUG(DLV_ALL, ("singit_config.c [singit_tag_manager_show]\n"));
#endif

	if (singit_tag_manager_win) {
		gdk_window_raise(singit_tag_manager_win->window);
		return;
	}

	singit_tag_manager_init();

	// Do we have a song?
	if (!STATUS || ((my_song = singit_song_attach(STATUS->song)) == NULL)) {
		singit_tag_manager_cleanup();
		return;
	}
	else {
		file_name = g_strdup(my_song->song_filename);
		singit_song_detach(&my_song);
	}

	// Do we have a current playlist item?
	if (file_name == NULL) {
		file_name = xmms_remote_get_playlist_file(singit_vp.xmms_session,
			xmms_remote_get_playlist_pos(singit_vp.xmms_session));
		if (file_name == NULL)
			{ return; }
	}

	// Which filetype will we support?
	id3dlg_status.ov_file = get_oggvorbis_file_info(file_name, NULL);
	if (id3dlg_status.ov_file == NULL) {

		id3dlg_status.id3tag = get_mpeg_file_info
			(file_name, NULL, &id3dlg_status.mpeg_info);

		if (id3dlg_status.id3tag != NULL)
			{ id3dlg_status.info_type = IT_MPEG; }
		else { return; }
	}
	else { id3dlg_status.info_type = IT_VORBIS; }

	tooltips = gtk_tooltips_new ();

	singit_tag_manager_win = gtk_window_new (GTK_WINDOW_DIALOG);
	gtk_widget_set_name (singit_tag_manager_win, "singit_tag_manager_win");
	gtk_object_set_data (GTK_OBJECT (singit_tag_manager_win),
		"singit_tag_manager_win", singit_tag_manager_win);
	gtk_window_set_title (GTK_WINDOW (singit_tag_manager_win),
		_("id3v1/2.x.x tag editor"));
	gtk_window_set_policy
		(GTK_WINDOW (singit_tag_manager_win), FALSE, FALSE, FALSE);
	gtk_signal_connect(GTK_OBJECT(singit_tag_manager_win), "destroy",
		GTK_SIGNAL_FUNC(gtk_widget_destroyed), &singit_tag_manager_win);
	gtk_signal_connect(GTK_OBJECT(singit_tag_manager_win), "delete_event",
		GTK_SIGNAL_FUNC(dlg_singit_id3_delete_event), NULL);
	gtk_container_border_width(GTK_CONTAINER(singit_tag_manager_win), 5);

	id3_main_vbox = gtk_vbox_new (FALSE, 0);
	id3_widget_descr(id3_main_vbox, "id3_main_vbox", TRUE);
	gtk_container_add (GTK_CONTAINER (singit_tag_manager_win), id3_main_vbox);

	id3_file_name_entry = gtk_entry_new ();
	gtk_entry_set_editable(GTK_ENTRY(id3_file_name_entry), FALSE);
	id3_widget_descr(id3_file_name_entry, "id3_file_name_entry", TRUE);
	gtk_box_pack_start(GTK_BOX(id3_main_vbox), id3_file_name_entry, TRUE, TRUE, 4);

	id3_notebook = gtk_notebook_new ();
	id3_widget_descr(id3_notebook, "id3_notebook", TRUE);
	gtk_box_pack_start(GTK_BOX (id3_main_vbox), id3_notebook, TRUE, TRUE, 0);

// *
// * Append the pages to the notebook
// *
	singit_tag_manager_page_v1();
	singit_tag_manager_page_v2();

//	append_hierachial_view_page(id3_notebook, singit_tag_manager_win);

	switch (id3dlg_status.info_type) {
	case IT_MPEG:
		append_mpeg_info_page(id3_notebook, singit_tag_manager_win,
			id3dlg_status.id3tag, id3dlg_status.mpeg_info);
		break;
	case IT_VORBIS:
		append_oggvorbis_info_page(id3_notebook,
			singit_tag_manager_win, id3dlg_status.ov_file);
	default:
		break;
	}


// *
// * The bottom button bar
// *
	id3_button_hbox = gtk_hbutton_box_new();
	id3_widget_descr (id3_button_hbox, "id3_button_hbox", TRUE);
//	gtk_button_box_set_layout(GTK_BUTTON_BOX(id3_button_hbox), GTK_BUTTONBOX_END);
	gtk_button_box_set_spacing(GTK_BUTTON_BOX(id3_button_hbox), 10);
	gtk_box_pack_start (GTK_BOX (id3_main_vbox), id3_button_hbox, FALSE, FALSE, 5);

	id3_save_button = gtk_button_new_with_label (_("Save"));
	id3_widget_descr (id3_save_button, "id3_save_button", FALSE);
	GTK_WIDGET_SET_FLAGS(id3_save_button, GTK_CAN_DEFAULT);
	gtk_box_pack_start (GTK_BOX (id3_button_hbox), id3_save_button, TRUE, TRUE, 0);
	gtk_container_set_border_width (GTK_CONTAINER (id3_save_button), 1);
	gtk_widget_set_sensitive (id3_save_button, FALSE);

	id3_remove_button = gtk_button_new_with_label (_("Remove"));
	id3_widget_descr (id3_remove_button, "id3_remove_button", FALSE);
	GTK_WIDGET_SET_FLAGS(id3_remove_button, GTK_CAN_DEFAULT);
	gtk_box_pack_start (GTK_BOX (id3_button_hbox), id3_remove_button, TRUE, TRUE, 0);
	gtk_container_set_border_width (GTK_CONTAINER (id3_remove_button), 1);
	gtk_widget_set_sensitive (id3_remove_button, FALSE);

	id3_cancel_button = gtk_button_new_with_label (_("Cancel"));
	id3_widget_descr (id3_cancel_button, "id3_cancel_button", TRUE);
	GTK_WIDGET_SET_FLAGS(id3_cancel_button, GTK_CAN_DEFAULT);
	gtk_box_pack_start (GTK_BOX (id3_button_hbox), id3_cancel_button, TRUE, TRUE, 0);
	gtk_container_set_border_width (GTK_CONTAINER (id3_cancel_button), 1);
  	gtk_signal_connect_object (GTK_OBJECT (id3_cancel_button), "clicked",
		GTK_SIGNAL_FUNC (singit_tag_manager_cancel_clicked), NULL);

	gtk_widget_grab_default (id3_cancel_button);

	gtk_object_set_data (GTK_OBJECT (singit_tag_manager_win), "tooltips", tooltips);

	singit_tag_manager_fill_dialog(file_name);

	g_free(file_name);

	gtk_widget_show(singit_tag_manager_win);
}
