/* properties.c - properties box for xpenguins_applet
 * Copyright (C) 1999-2001  Robin Hogan
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */
#include <config.h>
#include <stdio.h>
#include <applet-widget.h>

#include "xpenguins.h"
#include "applet.h"

/* Apply all the settings in the property box */
static void 
property_apply_cb(GtkWidget *widget, void *data, XPenguinData *xp)
{
  /* Speed and number of toons */
  if (xp->overridetheme) {
    xp->number = xp->number_adj->value;
    xp->speed = xp->speed_adj->value;
  }
  else if (xp->loaded) {
    xp->number = (xp->theme).total;
    xp->speed = 1000/(xp->theme).delay;
  }

  /* Change theme? */
  if (xp->next_theme_name
      && strcmp(xp->next_theme_name, xp->theme_name) != 0) {
    if (xp->active) {
      gtk_timeout_remove(xp->timeout);
      xpenguins_exit();
    }
    if (xp->loaded) {
      xpenguins_free_theme(&(xp->theme));
      xp->loaded = FALSE;
    }
    if (xp->theme_name) {
      g_free(xp->theme_name);
    }
    xp->theme_name = xp->next_theme_name;
    xp->next_theme_name = NULL;
    if (!load_theme(xp)) {
      xp->active = FALSE;
      return;
    }
    xp->loaded = TRUE;
    if (!xp->overridetheme) {
      xp->number = (xp->theme).total;
      xp->speed = 1000/(xp->theme).delay;
    }
    xpenguins_set_number(xp->number);
    if (xp->active) {
      if (xpenguins_start(NULL))
	return;
      xp->timeout = gtk_timeout_add(1000/xp->speed, service_xpenguins, xp);
    }
  }
  else {
    xpenguins_set_number(xp->number);
  }

  /* Set the timeout */
  if (xp->active) {
    gtk_timeout_remove(xp->timeout);
    xp->timeout = gtk_timeout_add(1000/xp->speed, service_xpenguins, xp);
  }

  /* Do we ignore popup windows? */
  if (GTK_TOGGLE_BUTTON(xp->ignorepopups_check)->active) {
    xp->ignorepopups = TRUE;
  }
  else {
    xp->ignorepopups = FALSE;
  }
  xpenguins_ignorepopups(xp->ignorepopups);

  /* Do we show gory death sequences? */
  if (GTK_TOGGLE_BUTTON(xp->noblood_check)->active) {
    xp->noblood = TRUE;
  }
  else {
    xp->noblood = FALSE;
  }
  xpenguins_set_blood(!xp->noblood);
  return;
}

/* Destroy property box */
static gint
property_destroy_cb( GtkWidget *widget, gpointer data)
{
  XPenguinData *xp = data;
  xp->prop_window = NULL;
  return FALSE;
  widget = NULL;
}

/* A change has occurred - enable Apply button */
static void
update_properties_cb(GtkWidget *widget, gpointer data)
{
  XPenguinData *xp = data;
  gnome_property_box_changed(GNOME_PROPERTY_BOX(xp->prop_window));
}

/* Activate or deactivate the number and speed controls */
static void
update_overridetheme_cb(GtkWidget *widget, gpointer data)
{
  XPenguinData *xp = data;
  gnome_property_box_changed(GNOME_PROPERTY_BOX(xp->prop_window));
  if (GTK_TOGGLE_BUTTON(xp->overridetheme_check)->active) {
    xp->overridetheme = TRUE;
  }
  else {
    xp->overridetheme = FALSE;
  }
  gtk_widget_set_sensitive(GTK_WIDGET(xp->table), xp->overridetheme);
}

/* A theme has been selected in the theme list - display the preview
 * icon and text for it */
static void
property_select_theme_cb(GtkCList *widget, gint row, gint column,
			 GdkEventButton *event, gpointer data)
{
  XPenguinData *xp = data;
  gchar *theme_name = xp->theme_list[row];
  if (xp->next_theme_name) {
    g_free(xp->next_theme_name);
  }
  xp->next_theme_name = g_strdup(theme_name);

  /* Clear the theme information from a previous query */
  if (xp->theme_info) {
    xpenguins_free_list(xp->theme_info);
    xp->theme_info = NULL;
  }

  /* Get information about the theme */
  if ((xp->theme_info = xpenguins_theme_info(theme_name))) {
    /* Create the text information string - first add the theme name */
    GString *string = g_string_new(theme_name);
    /* If preview icon already exists, destroy it */
    if (xp->preview_pixmap) {
      gtk_widget_destroy(xp->preview_pixmap);
    }
    /* Display the icon if it exists */
    if ((xp->preview_pixmap 
	 = gnome_pixmap_new_from_file(xp->theme_info[PENGUIN_ICON]))) {
      gtk_box_pack_end(GTK_BOX(xp->preview), xp->preview_pixmap,
		       FALSE, FALSE, 5);
      gtk_widget_show(xp->preview_pixmap);
    }
    /* Add the date */
    if (*(xp->theme_info[PENGUIN_DATE])) {
      g_string_append(string, _(" (Last updated "));
      g_string_append(string, xp->theme_info[PENGUIN_DATE]);
      g_string_append(string, ")");
    }
    /* Add the artist(s) */
    if (*(xp->theme_info[PENGUIN_ARTIST])) {
      g_string_append(string, _("\nArtists: "));
      g_string_append(string, xp->theme_info[PENGUIN_ARTIST]);
    }
    /* Add the maintainer */
    if (*(xp->theme_info[PENGUIN_MAINTAINER])) {
      g_string_append(string, _("\nMaintainer: "));
      g_string_append(string, xp->theme_info[PENGUIN_MAINTAINER]);
    }
    /* Add the copyright */
    if (*(xp->theme_info[PENGUIN_COPYRIGHT])) {
      g_string_append(string, _("\nCopyright (C) "));
      g_string_append(string, xp->theme_info[PENGUIN_COPYRIGHT]);
    }
    /* Add the comment field */
    if (*(xp->theme_info[PENGUIN_COMMENT])) {
      g_string_append(string, _("\n"));
      g_string_append(string, xp->theme_info[PENGUIN_COMMENT]);
    }
    /* Display the information */
    gtk_label_set_text(GTK_LABEL(xp->info), string->str);
    g_string_free(string, FALSE);
  }
  else {
    /* No "about" file found for theme */
    GString *string = g_string_new(theme_name);
    g_string_append(string, _(" (no information available)"));
    gtk_label_set_text(GTK_LABEL(xp->info), string->str);
    g_string_free(string, FALSE);
    if (xp->preview_pixmap) {
      gtk_widget_destroy(xp->preview_pixmap);
    }
  }
  if (!xp->send_event) {
    gnome_property_box_changed(GNOME_PROPERTY_BOX(xp->prop_window));
  }
  else {
    xp->send_event = FALSE;
  }
}

/* Create the property box, from which the properties of xpenguins can
 * be changed */
void
property_show(AppletWidget *applet, gpointer data)
{
  static GnomeHelpMenuEntry help_entry
    = { "xpenguins_applet", "index.html#XPENGUINSAPPLET-PREFS" };
  XPenguinData *xp = data;
  GtkWidget *frame;
  GtkWidget *list;
  GtkWidget *label;
  GtkWidget *table;
  GtkWidget *scroll;
  GtkWidget *number_range, *speed_range;
  /* help_entry.name = gnome_app_id; */

  /* If property box already exists, bring it to the top and return */
  if(xp->prop_window) {
    gdk_window_raise(xp->prop_window->window);
    return;
  }
  /* Make a new property window */
  xp->prop_window = gnome_property_box_new();
  gtk_window_set_title(GTK_WINDOW(&GNOME_PROPERTY_BOX(xp->prop_window)->dialog.window),
		       _("XPenguins settings"));

  /* Property box contains two pages: "General" and "Themes" */
  /* First the "General" page, which is a vertical container */
  frame = gtk_vbox_new(FALSE, 5);

  /* Override the number and speed of toons in the current theme */
  xp->overridetheme_check = gtk_check_button_new_with_label(
	              _("Override number and speed of toons in theme"));
  gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(xp->overridetheme_check),
			       xp->overridetheme);
  gtk_box_pack_start_defaults(GTK_BOX(frame), xp->overridetheme_check);
  gtk_signal_connect(GTK_OBJECT(xp->overridetheme_check),"clicked",
		     GTK_SIGNAL_FUNC(update_overridetheme_cb), xp);
  gtk_widget_show(xp->overridetheme_check);

  /* The override button controls this table, which contains the
   * number and speed override controls */
  xp->table = table = gtk_table_new(2, 2, FALSE);
  gtk_widget_show(table);
  gtk_box_pack_start_defaults(GTK_BOX(frame), table);
  /* Set start sensitivity of table (i.e whether it is enabled or
   * disabled */
  gtk_widget_set_sensitive(GTK_WIDGET(xp->table), xp->overridetheme);

  /* Override the number of penguins */
  /* First the label */
  label = gtk_label_new(_("Number of toons:"));
  gtk_table_attach(GTK_TABLE(table), label, 0, 1, 0, 1,
		   GTK_FILL | GTK_SHRINK, GTK_FILL | GTK_SHRINK, 5, 5);
  gtk_widget_show(label);
  /* Then the slider */
  xp->number_adj = (GtkAdjustment *) gtk_adjustment_new(xp->number,
							1,110,1,1,10);
  number_range = gtk_hscale_new(GTK_ADJUSTMENT(xp->number_adj));
  gtk_table_attach(GTK_TABLE(table), number_range, 1, 2, 0, 1,
		   GTK_FILL | GTK_EXPAND, GTK_FILL | GTK_SHRINK, 5, 5);
  gtk_signal_connect(GTK_OBJECT(xp->number_adj),"value_changed",
		     GTK_SIGNAL_FUNC(update_properties_cb), xp);
  gtk_range_set_update_policy(GTK_RANGE(number_range),
			      GTK_UPDATE_DISCONTINUOUS);
  gtk_widget_show(number_range);

  /* Override the frame rate */
  /* First the label */
  label = gtk_label_new(_("Frame rate (Hz):"));
  gtk_table_attach(GTK_TABLE(table), label, 0, 1, 1, 2,
		   GTK_FILL | GTK_SHRINK , GTK_FILL | GTK_SHRINK, 5, 5);
  gtk_widget_show(label);
  /* Then the slider */
  xp->speed_adj = (GtkAdjustment *) gtk_adjustment_new(xp->speed,
						       1,101,1,1,1);
  speed_range = gtk_hscale_new(GTK_ADJUSTMENT(xp->speed_adj));
  gtk_table_attach(GTK_TABLE(table), speed_range, 1, 2, 1, 2,
		   GTK_FILL | GTK_EXPAND, GTK_FILL | GTK_SHRINK, 5, 5);
  gtk_signal_connect(GTK_OBJECT(xp->speed_adj),"value_changed",
		     GTK_SIGNAL_FUNC(update_properties_cb), xp);
  gtk_range_set_update_policy(GTK_RANGE(speed_range),
			      GTK_UPDATE_DISCONTINUOUS);
  gtk_widget_show(speed_range);

  /* Some check buttons: first one for ignoring popup windows
   * (e.g. tooltips) */
  xp->ignorepopups_check = gtk_check_button_new_with_label(
				     _("Ignore popup windows"));
  gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(xp->ignorepopups_check),
			       xp->ignorepopups);
  gtk_box_pack_start_defaults(GTK_BOX(frame), xp->ignorepopups_check);
  gtk_signal_connect(GTK_OBJECT(xp->ignorepopups_check),"clicked",
		     GTK_SIGNAL_FUNC(update_properties_cb), xp);
  gtk_widget_show(xp->ignorepopups_check);

  /* Then one for omitting all the gory images! */
  xp->noblood_check = gtk_check_button_new_with_label(
			    _("Do not show gory death sequences"));
  gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(xp->noblood_check),
			       xp->noblood);
  gtk_box_pack_start_defaults(GTK_BOX(frame), xp->noblood_check);
  gtk_signal_connect(GTK_OBJECT(xp->noblood_check),"clicked",
		     GTK_SIGNAL_FUNC(update_properties_cb), xp);
  gtk_widget_show(xp->noblood_check);

  /* Post the "General" page in the property box */
  label = gtk_label_new(_("General"));
  gtk_widget_show(frame);
  gnome_property_box_append_page(GNOME_PROPERTY_BOX(xp->prop_window),
				 frame, label);

  /* Now the "Themes" page, again a vertical container */
  frame = gtk_vbox_new(FALSE, 5);

  /* Preview themes in a horizontal container in which we put the
   * icon and the text information */
  xp->preview = gtk_hbox_new(FALSE, 5);
  gtk_box_pack_start_defaults(GTK_BOX(frame), xp->preview);
  xp->info = gtk_label_new(NULL);
  gtk_label_set_justify(GTK_LABEL(xp->info), GTK_JUSTIFY_LEFT);
  gtk_box_pack_end(GTK_BOX(xp->preview), xp->info, TRUE, TRUE, 2);
  gtk_widget_show(xp->preview);
  gtk_widget_show(xp->info);
  xp->preview_pixmap = NULL;

  /* The last widget is a scrolled window containing a list of 
   * the available themes */
  scroll = gtk_scrolled_window_new(NULL, NULL);
  gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(scroll),
				 GTK_POLICY_AUTOMATIC,
				 GTK_POLICY_AUTOMATIC);
  list = gtk_clist_new(1);
  gtk_clist_set_selection_mode(GTK_CLIST(list), GTK_SELECTION_BROWSE);

  /* Clear the theme list from a previous call */
  if (xp->theme_list) {
    xpenguins_free_list(xp->theme_list);
    xp->theme_list = NULL;
  }
  /* Call the xpenguins library to find the available themes */
  xp->theme_list = xpenguins_list_themes(&(xp->nthemes));

  /* Append each theme to the list */
  if (xp->theme_list) {
    int i;
    int row = -1;
    for (i = 0; i < xp->nthemes; ++i) {
      gtk_clist_append(GTK_CLIST(list), xp->theme_list +i);
      /* See if the current theme is in the list */
      if (strcmp(xp->theme_list[i], xp->theme_name) == 0) {
	row = i;
      }
    }
    gtk_signal_connect(GTK_OBJECT(list), "select_row",
		       GTK_SIGNAL_FUNC(property_select_theme_cb), xp);
    if (row > -1) {
      /* Activate the entry corresponding to the current theme */
      xp->send_event = TRUE;
      gtk_clist_select_row(GTK_CLIST(list), row, 0);
    }
  }
  else {
    /* No themes found! */
    GString *str;
    str = g_string_new(_("No themes found in "));
    str = g_string_append(str, XPENGUINS_SYSTEM_DIRECTORY);
    str = g_string_append(str,
	  _("\nMake sure you have installed the `xpenguins' package"));
    gtk_label_set_text(GTK_LABEL(xp->info), str->str);
    g_string_free(str, TRUE);
  }
  /* Set list properties and pack */
  gtk_clist_set_column_width(GTK_CLIST(list), 0,
			     gtk_clist_optimal_column_width(GTK_CLIST(list),
							    0));
  gtk_box_pack_start(GTK_BOX(frame), scroll, FALSE, FALSE, 2);
  gtk_container_add(GTK_CONTAINER(scroll), list);
  gtk_widget_show(frame);

  /* Post the "Themes" page in the property box */
  label = gtk_label_new(_("Themes"));
  gnome_property_box_append_page(GNOME_PROPERTY_BOX(xp->prop_window),
				 frame, label);

  /* Callbacks for the various property-box buttons */
  gtk_signal_connect( GTK_OBJECT(xp->prop_window), "apply",
		      GTK_SIGNAL_FUNC(property_apply_cb), xp);

  gtk_signal_connect( GTK_OBJECT(xp->prop_window), "destroy",
		      GTK_SIGNAL_FUNC(property_destroy_cb), xp);

  gtk_signal_connect( GTK_OBJECT(xp->prop_window), "help",
		      GTK_SIGNAL_FUNC(gnome_help_pbox_goto),
		      &help_entry );

  gtk_widget_show_all(xp->prop_window);
}

