/*

    xpuyopuyo - pfile.c       Copyright(c) 1999,2000 Justin David Smith
    justins(at)chaos2.org     http://chaos2.org/
    
    Utilities for FILE I/O
    

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

*/


/* System includes */
#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <sys/stat.h>
#include <unistd.h>


/* Local includes */
#include <xpuyopuyo.h>
#include <pfile.h>
#include <psnprintf.h>



char *p_file_break(char *s, char k) {
/* p_file_break

   Finds a break in a string line, denoted by the character k.  The returned
   pointer points one past the breaking character if one is found, or else
   NULL is returned if the breaking character is not in the string.  The
   breaking character will also be NULL'ed, forming two separate strings. 
   */

   char *p;                /* Pointer to iterate with */
   
   if(!s) return(NULL);    /* Make sure s is not null */
   
   p = strchr(s, k);       /* Search for character in string */
   if(p != NULL) {
      /* Character was found; NULL it and return next */
      *p = '\0';
      return(p + 1);
   }
   
   /* Character was not found in the string; return NULL. */
   return(NULL);

}



char *p_local_file_name(char *filename, const char *base, int size) {

   /* Construct filename for configuration file */
   p_snprintf(filename, size, "%s/%s/%s", getenv("HOME"), P_XPUYOPUYO_DIR, base);
   return(filename);

}



char *p_local_theme_file_name(char *filename, const char *base, int size) {

   /* Construct filename for configuration file */
   p_snprintf(filename, size, "%s/%s/%s/%s", getenv("HOME"), P_XPUYOPUYO_DIR, P_LOCAL_THEME_DIR, base);
   return(filename);

}



char *p_merge_path_base(char *filename, const char *path, const char *base, int size) {

   /* Construct filename for configuration file */
   p_snprintf(filename, size, "%s/%s", path, base);
   return(filename);

}



FILE *p_file_open(const char *filename, const char *flag) {
/* p_file_open

   Opens a configuration file.  The user's home directory and xpuyopuyo 
   directory are prepended to the filename given.  */

   char buf[P_IO_BUFFER];        /* Temporary buffer to read into */
   FILE *f;                      /* File input stream */
   
   /* Construct filename for configuration file */
   p_local_file_name(buf, filename, P_IO_BUFFER);

   /* Attempt to open the file */   
   f = fopen(buf, flag);
   if(f == NULL) {
      fprintf(stderr, "Cannot open file %s in mode %s\n", buf, flag);
   } 
   return(f);

}



char *p_file_read_line(char *buf, int size, FILE *f) {

   char *p;

   do {
      p = fgets(buf, size, f);
   } while(p != NULL && (*p == '#' || *p == '\0'));
   if(p == NULL) return(NULL);
   while(*p != '\0' && *p != '\n') p++;
   *p = '\0';
   return(buf);

}



void p_make_config_dir(void) {
/* make_xpuyopuyo_dir */

   char filename[P_IO_BUFFER];      /* Write temporary filename to */
   struct stat s;                   /* Status structure to test existence */

   /* Write the directory name out -  $HOME/.xpuyopuyo  */
   p_snprintf(filename, P_IO_BUFFER, "%s/%s", getenv("HOME"), P_XPUYOPUYO_DIR);

   /* Check for existence of the directory */
   if(stat(filename, &s) < 0) {
      /* Does not yet exist */
      printf("%s does not exist, creating ...\n", filename);
      if(mkdir(filename, 0755) < 0) {        
         /* Directory creation failed; this is inconvenient */
         fprintf(stderr, "Cannot make directory %s.  Data will not be saved between sessions.\n", filename);
      } 
   }

   /* Go looking for other files we need */
   if(stat(P_GLOBAL_DIR P_LICENSE_FILE, &s) < 0) {
      fprintf(stderr, "warning:  cannot find license %s, required by the application\n", P_GLOBAL_DIR P_LICENSE_FILE);
   }
   if(stat(P_GLOBAL_DIR P_MANUAL_FILE, &s) < 0) {
      fprintf(stderr, "warning:  cannot find manual %s, required by the application\n", P_GLOBAL_DIR P_MANUAL_FILE);
   }
                              
}
