/* $Header: /fridge/cvs/xscorch/sutil/sstr.c,v 1.5 2001/07/08 22:21:22 justins Exp $ */
/*
   
   xscorch - sstr.h           Copyright(c) 2000 Justin David Smith
   justins(at)chaos2.org      http://chaos2.org/
    
   Improved string manipulation functions
    

   This program is free software; you can redistribute it and/or modify 
   it under the terms of the GNU General Public License as published by 
   the Free Software Foundation; either version 2 of the License, or 
   (at your option) any later version.

   This program is distributed in the hope that it will be useful, 
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU 
   General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software Foundation, 
   Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

*/
#include <config.h>
#include <sstr.h>
#include <stdio.h>
#include <stdlib.h>
#include <stdarg.h>

#if HAVE_STRNLEN
   /* Required to get strnlen prototype on Linux */
   #define  __USE_GNU
#endif /* HAVE_STRNLEN */

#include <string.h>



int strnlenn(const char *src, int maxlen) {
/* strnlenn */

   #if HAVE_STRNLEN
      if(src == NULL || maxlen < 0) return(0);
      return(strnlen(src, maxlen));
   #else /* No strnlen function */
      const char *p = src;
      if(src == NULL || maxlen < 0) return(0);
      while(maxlen > 0 && *p != '\0') {
         ++p;
         --maxlen;
      }
      return(p - src);
   #endif /* HAVE_STRNLEN? */

}



const char *strstrn(const char *dest, const char *src) {
/* strstrn */

   if(dest != NULL && src != NULL) {
      return(strstr(dest, src));
   }
   return(NULL);
   
}



int strcmpn(const char *dest, const char *src) {
/* strcmpn */

   if(dest != NULL && src != NULL) {
      return(!strcmp(dest, src));
   }
   return(dest == src);

}



int strncmpn(const char *dest, const char *src, int size) {
/* strncmpn */

   if(dest != NULL && src != NULL) {
      return(!strncmp(dest, src, size));
   }
   return(dest == src);

}



void *memcpyn(void *dest, const void *src, int size) {
/* memcpyn */

   if(size > 0 && dest != NULL && src != NULL) {
      memcpy(dest, src, size);
      return(dest);
   }
   return(NULL);

}



char *strncpyn(char *dest, const char *src, int size) {
/* strncpyn */

   if(size > 0 && dest != NULL && src != NULL) {
      strncpy(dest, src, size);
      dest[size - 1] = '\0';
      return(dest);
   }
   return(NULL);

}



char *strncatn(char *dest, const char *src, int size) {
/* strncatn */

   if(size > 0 && dest != NULL && src != NULL) {
      strncat(dest, src, size);
      dest[size - 1] = '\0';
      return(dest);
   }
   return(NULL);

}



char *snprintfn(char *dest, int size, const char *fmt, ...) {
/* snprintfn */

   va_list args;
   
   if(size > 0 && dest != NULL && fmt != NULL) {
      va_start(args, fmt);
      vsnprintf(dest, size, fmt, args);
      va_end(args);
      dest[size - 1] = '\0';
      return(dest);
   }
   return(NULL);

}



void *memmoven(void *dest, const void *src, int size) {
/* memmoven */

   if(size > 0 && src != NULL && dest != NULL) {
      memmove(dest, src, size);
      return(dest);
   }
   return(NULL);

}



void *memsetn(void *dest, int ch, int size) {
/* memsetn */

   if(size > 0 && dest != NULL) {
      memset(dest, ch, size);
      return(dest);
   }
   return(NULL);

}



char *serror(int errn) {
/* serror */

   return(strerror(errn));
   
}



char *strunescape(char *src) {
/* strunescape
   Unescapes the given string.  Also removes unescaped double-quotes.  */

   char *orig;
   char *dest;
   
   if(src == NULL) return(NULL);
   
   orig = src;
   dest = src;
   while(*src != '\0') {
      if(*src != '\"') {
         /* Character is not a quote */
         if(*src == '\\') {
            /* Next character is escaped... */
            if(*(src + 1) == '\0') {
               *dest++ = '\\';
            } else switch(*++src) {
               case 'n':
                  *dest++ = '\n';
                  break;
               case 'r':
                  *dest++ = '\r';
                  break;
               case 't':
                  *dest++ = '\t';
                  break;
               default:
                  /* Just copy the escaped character */
                  *dest++ = *src;
            } /* Check for escape sequences */

         } else {
            /* Character is not escaped */
            *dest++ = *src;
         } /* Is character escaped? */
      } /* Character is not a quote */
      ++src;
   } /* While looking through src... */
   
   *dest = '\0';
   return(orig);

}



char *strescape(char *str, int size) {
/* strescape
   Escapes potentially troublesome characters in the string.  */

   char *buf;
   char *src;
   char *dest;
   
   if(str == NULL || size <= 0) return(NULL);
   
   buf = (char *)malloc(size + size + 2);
   if(buf == NULL) return(NULL);

   src = str;
   dest = buf;
   while(*src != '\0') {
      if(*src == '\n') {
         *dest++ = '\\';
         *dest++ = 'n';
      } else if(*src == '\r') {
         *dest++ = '\\';
         *dest++ = 'r';
      } else if(*src == '\t') {
         *dest++ = '\\';
         *dest++ = 't';
      } else if(*src == '\\' || *src == '\"') {
         *dest++ = '\\';
         *dest++ = *src;
      } else {
         *dest++ = *src;
      }
      ++src;
   }

   *dest = '\0';
   memcpy(str, buf, size);
   free(buf);
   return(str);

}



char *strtrim(char *src) {
/* strtrim
   Removes whitespace from the beginning and the end of the string.  */

   char *fnws;       /* Pointer to the first non-whitespace character */
   char *lnws;       /* Pointer to the last non-whitespace character */
   char *p;          /* Current character pointer */

   /* Sanity check */
   if(src == NULL) return(NULL);

   /* Scan for first nonwhitespace character */
   for(fnws = src; *fnws == ' ' || *fnws == '\t' || *fnws == '\n'; ++fnws) /* Just loop */;

   /* Scan for last nonwhitespace character */
   for(lnws = fnws - 1, p = fnws; *p != '\0'; ++p) {
      if(*p != ' ' && *p != '\t' && *p != '\n') lnws = p;
   }

   /* Make sure the string isn't empty */
   if(lnws < fnws) {
      *src = '\0';  
      return(src);
   }

   /* Move the string down and truncate, as needed */
   memmoven(src, fnws, lnws - fnws + 1);
   src[lnws - fnws + 1] = '\0';
   return(src);
   
}  
   
   
   
char *strscan(char *src, char ch) {
/* strscan
   Scans for occurrence of ch in the string.  Returns NULL if the character
   is not found, otherwise the first occurence is returned.  This function
   will ignore characters inside a quoted block, as well as escaped
   characters.  */

   char *p;
   int instring;

   if(src == NULL) return(NULL);

   p = src;
   instring = 0;
   while(*p != '\0') {
      if(*p == '\"') {
         instring = !instring;
      } else if(*p == '\\' && *(p + 1) != '\0') {
         ++p;
      } else if(!instring && *p == ch) return(p);
      ++p;
   }      
    
   return(NULL);

}



char *strchop(char *src, char ch) {
/* strchop
   Chops off the string after the first occurrence of ch.  */

   char *p;
   
   p = strscan(src, ch);
   if(p != NULL) *p = '\0';
   return(src);

}
