// file      : xsd/cxx/tree/tree-source.cxx
// author    : Boris Kolpackov <boris@codesynthesis.com>
// copyright : Copyright (c) 2005-2008 Code Synthesis Tools CC
// license   : GNU GPL v2 + exceptions; see accompanying LICENSE file

#include <cxx/tree/tree-source.hxx>

#include <xsd-frontend/semantic-graph.hxx>
#include <xsd-frontend/traversal.hxx>

#include <cult/containers/list.hxx>

#include <iostream>

using std::wcerr;

namespace CXX
{
  namespace Tree
  {
    namespace
    {
      struct List: Traversal::List, protected virtual Context
      {
        List (Context& c)
            : Context (c)
        {
        }

        virtual Void
        traverse (Type& l)
        {
          String name (ename (l));

          // If renamed name is empty then we do not need to generate
          // anything for this type.
          //
          if (renamed_type (l, name) && !name)
            return;

          os << "// " << name << endl
             << "//" << endl
             << endl;

          if (!options.value<CLI::suppress_parsing> ())
          {
            String base (L"::xsd::cxx::tree::list< " +
                         item_type_name (l) +
                         L", " + char_type + L" >");

            // c-tor (xercesc::DOMElement)
            //
            os << name << "::" << endl
               << name << " (const " << xerces_ns << "::DOMElement& e," << endl
               << flags_type << " f," << endl
               << container << "* c)" << endl
               << ": " << any_simple_type << " (e, f, c)," << endl
               << "  " << base << " (e, f, c)"
               << "{"
               << "}";

            // c-tor (xercesc::DOMAttr)
            //
            os << name << "::" << endl
               << name << " (const " << xerces_ns << "::DOMAttr& a," << endl
               << flags_type << " f," << endl
               << container << "* c)" << endl
               << ": " << any_simple_type << " (a, f, c)," << endl
               << "  " << base << " (a, f, c)"
               << "{"
               << "}";

            // c-tor (string const&, xercesc::DOMElement)
            //
            os << name << "::" << endl
               << name << " (const " << string_type << "& s," << endl
               << "const " << xerces_ns << "::DOMElement* e," << endl
               << flags_type << " f," << endl
               << container << "* c)" << endl
               << ": " << any_simple_type << " (s, e, f, c)," << endl
               << "  " << base << " (s, e, f, c)"
               << "{"
               << "}";
          }

          // _clone
          //
          os << name << "* " << name << "::" << endl
             << "_clone (" << flags_type << " f," << endl
             << container << "* c) const"
             << "{"
             << "return new " << name << " (*this, f, c);"
             << "}";

          // d-tor
          //
          os << name << "::" << endl
             << "~" << name << " ()"
             << "{"
             << "}";

          // Register with type factory map.
          //
          if (options.value<CLI::generate_polymorphic> () &&
              !l.context ().count ("anonymous"))
          {
            // Note that we are using the original type name.
            //
            os << "static" << endl
               << "::xsd::cxx::tree::type_factory_initializer< 0, " <<
              char_type << ", " << ename (l) << " >" << endl
               << "_xsd_" << ename (l) << "_type_factory_init (" << endl
               << L << "\"" << l.name () << "\"," << endl
               << L << "\"" << xml_ns_name (l) << "\");"
               << endl
               << endl;
          }
        }

      private:
        String
        item_type_name (Type& l)
        {
          std::wostringstream o;

          MemberTypeName type (*this, o);
          type.dispatch (l.argumented ().type ());

          return o.str ();
        }
      };


      struct Union: Traversal::Union, protected virtual Context
      {
        Union (Context& c)
            : Context (c)
        {
        }

        virtual Void
        traverse (Type& u)
        {
          String name (ename (u));

          // If renamed name is empty then we do not need to generate
          // anything for this type.
          //
          if (renamed_type (u, name) && !name)
            return;

          String const& base (xs_string_type);

          os << "// " << name << endl
             << "//" << endl
             << endl;

          if (!options.value<CLI::suppress_parsing> ())
          {
            // c-tor (xercesc::DOMElement)
            //
            os << name << "::" << endl
               << name << " (const " << xerces_ns << "::DOMElement& e," << endl
               << flags_type << " f," << endl
               << container << "* c)" << endl
               << ": " << base << " (e, f, c)"
               << "{"
               << "}";

            // c-tor (xercesc::DOMAttr)
            //
            os << name << "::" << endl
               << name << " (const " << xerces_ns << "::DOMAttr& a," << endl
               << flags_type << " f," << endl
               << container << "* c)" << endl
               << ": " << base << " (a, f, c)"
               << "{"
               << "}";

            // c-tor (string const&, xercesc::DOMElement)
            //
            os << name << "::" << endl
               << name << " (const " << string_type << "& s," << endl
               << "const " << xerces_ns << "::DOMElement* e," << endl
               << flags_type << " f," << endl
               << container << "* c)" << endl
               << ": " << base << " (s, e, f, c)"
               << "{"
               << "}";
          }

          // _clone
          //
          os << name << "* " << name << "::" << endl
             << "_clone (" << flags_type << " f," << endl
             << container << "* c) const"
             << "{"
             << "return new " << name << " (*this, f, c);"
             << "}";

          // Register with type factory map.
          //
          if (options.value<CLI::generate_polymorphic> () &&
              !u.context ().count ("anonymous"))
          {
            // Note that we are using the original type name.
            //
            os << "static" << endl
               << "::xsd::cxx::tree::type_factory_initializer< 0, " <<
              char_type << ", " << ename (u) << " >" << endl
               << "_xsd_" << ename (u) << "_type_factory_init (" << endl
               << L << "\"" << u.name () << "\"," << endl
               << L << "\"" << xml_ns_name (u) << "\");"
               << endl
               << endl;
          }
        }
      };


      // Enumeration mapping.
      //

      struct EnumeratorLiteral: Traversal::Enumerator,
                                protected virtual Context
      {
        EnumeratorLiteral (Context& c)
            : Context (c)
        {
        }

        virtual Void
        traverse (Type& e)
        {
          os << L << "\"" << escape_str (e.name ()) << "\"";
        }
      };


      //
      //
      struct LiteralInfo
      {
        LiteralInfo (String const& value, String const& name)
            : value_ (value), name_ (name)
        {
        }

        String value_;
        String name_;
      };

      Boolean
      operator< (LiteralInfo const& x, LiteralInfo const& y)
      {
        return x.value_ < y.value_;
      }

      typedef Cult::Containers::List<LiteralInfo> LiteralInfoList;


      // Populate LiteralInfoList
      //
      struct EnumeratorLiteralInfo: Traversal::Enumerator,
                                    protected virtual Context

      {
        EnumeratorLiteralInfo (Context& c, LiteralInfoList& list)
            : Context (c), list_ (list)
        {
        }

        virtual Void
        traverse (Type& e)
        {
          list_.push_back (LiteralInfo (e.name (), ename (e)));
        }

      private:
        LiteralInfoList& list_;
      };


      struct Enumeration: Traversal::Enumeration, protected virtual Context
      {
        Enumeration (Context& c)
            : Context (c), enumerator_literal_ (c)
        {
          names_enumerator_literal_ >> enumerator_literal_;
        }

        virtual Void
        traverse (Type& e)
        {
          String name (ename (e));

          // If renamed name is empty then we do not need to generate
          // anything for this type.
          //
          if (renamed_type (e, name) && !name)
            return;

          Boolean string_based (false);
          {
            IsStringBasedType t (string_based);
            t.dispatch (e);
          }

          SemanticGraph::Enumeration* be (0);
          Boolean enum_based (false);
          if (string_based)
          {
            IsEnumBasedType t (be);
            t.dispatch (e);

            enum_based = (be != 0);
          }

          String value;
          if (string_based)
            value = evalue (e);

          UnsignedLong enum_count (0);

          for (Type::NamesIterator i (e.names_begin ()), end (e.names_end ());
               i != end; ++i)
            ++enum_count;

          String base; // base type name
          {
            std::wostringstream o;

            BaseTypeName base_type (*this, o);
            Traversal::Inherits inherits_type (base_type);

            inherits (e, inherits_type);
            base = o.str ();
          }

          os << "// " << name << endl
             << "//" << endl
             << endl;

          if (!options.value<CLI::suppress_parsing> ())
          {
            // c-tor (xercesc::DOMElement)
            //
            os << name << "::" << endl
               << name << " (const " << xerces_ns << "::DOMElement& e," << endl
               << flags_type << " f," << endl
               << container << "* c)" << endl
               << ": " << base << " (e, f, c)"
               << "{";

            if (string_based)
              os << "_xsd_" << name << "_convert ();";

            os << "}";


            // c-tor (xercesc::DOMAttr)
            //
            os << name << "::" << endl
               << name << " (const " << xerces_ns << "::DOMAttr& a," << endl
               << flags_type << " f," << endl
               << container << "* c)" << endl
               << ": " << base << " (a, f, c)"
               << "{";

            if (string_based)
            {
              os << "_xsd_" << name << "_convert ();";
            }

            os << "}";


            // c-tor (string const&, xercesc::DOMElement)
            //
            os << name << "::" << endl
               << name << " (const " << string_type << "& s," << endl
               << "const " << xerces_ns << "::DOMElement* e," << endl
               << flags_type << " f," << endl
               << container << "* c)" << endl
               << ": " << base << " (s, e, f, c)"
               << "{";

            if (string_based)
            {
              os << "_xsd_" << name << "_convert ();";
            }

            os << "}";
          }

          // _clone
          //
          os << name << "* " << name << "::" << endl
             << "_clone (" << flags_type << " f," << endl
             << container << "* c) const"
             << "{"
             << "return new " << name << " (*this, f, c);"
             << "}";

          // convert
          //
          // @@ TODO: expected list
          //
          if (string_based)
          {
            String i_name (L"_xsd_" + name + L"_indexes_");

            os << name << "::" << value << " " <<
              name << "::" << endl
               << "_xsd_" << name << "_convert () const"
               << "{"
               << "::xsd::cxx::tree::enum_comparator< " << char_type <<
              " > c (_xsd_" << name << "_literals_);"
               << "const " << value << "* i (::std::lower_bound (" << endl
               << i_name << "," << endl
               << i_name << " + " << enum_count << "," << endl
               << "*this," << endl
               << "c));"
               << endl
               << "if (i == " << i_name << " + " << enum_count << " || " <<
              "_xsd_" << name << "_literals_[*i] != *this)"
               << "{"
               << "throw ::xsd::cxx::tree::unexpected_enumerator < " <<
              char_type << " > (*this);"
               << "}"
               << "return *i;"
               << "}";
          }

          // literals and indexes
          //
          if (string_based)
          {
            if (enum_based)
            {
              os << "const " << char_type << "* const* " << name << "::" << endl
                 << "_xsd_" << name << "_literals_ = " <<
                fq_name (*be) << "::_xsd_" << ename (*be) << "_literals_;"
                 << endl;
            }
            else
            {
              os << "const " << char_type << "* const " << name << "::" << endl
                 << "_xsd_" << name << "_literals_[" << enum_count << "] ="
                 << "{";

              names<Enumeration> (
                e, names_enumerator_literal_, 0, 0, 0, &Enumeration::comma);

              os << "};";
            }


            LiteralInfoList l;
            {
              EnumeratorLiteralInfo enumerator (*this, l);
              Traversal::Names names_enumerator (enumerator);
              names (e, names_enumerator);
            }

            l.sort ();

            os << "const " << name << "::" << value << " " <<
              name << "::" << endl
               << "_xsd_" << name << "_indexes_[" << enum_count << "] ="
               << "{";

            String fq_name (ns_scope + L"::" + name);

            for (LiteralInfoList::Iterator
                   b (l.begin ()), i (b), end (l.end ()); i != end; ++i)
            {
              if (i != b)
                os << "," << endl;

              os << fq_name << "::" << i->name_;
            }

            os << "};";
          }

          // Register with type factory map.
          //
          if (options.value<CLI::generate_polymorphic> () &&
              !e.context ().count ("anonymous"))
          {
            // Note that we are using the original type name.
            //
            os << "static" << endl
               << "::xsd::cxx::tree::type_factory_initializer< 0, " <<
              char_type << ", " << ename (e) << " >" << endl
               << "_xsd_" << ename (e) << "_type_factory_init (" << endl
               << L << "\"" << e.name () << "\"," << endl
               << L << "\"" << xml_ns_name (e) << "\");"
               << endl
               << endl;
          }
        }

        virtual Void
        comma (Type&)
        {
          os << "," << endl;
        }

      private:
        Traversal::Names names_enumerator_literal_;
        EnumeratorLiteral enumerator_literal_;

      };

      //
      //
      struct Member: Traversal::Member, protected virtual Context
      {
        Member (Context& c, String const& scope)
            : Context (c), scope_ (scope)
        {
        }

        virtual Void
        traverse (Type& m)
        {
	  if (skip (m))
            return;

          // default_value
          //
          if (m.default_ ())
          {
            Boolean simple (true);

            if (m.is_a<SemanticGraph::Element> ())
            {
              IsSimpleType test (simple);
              test.dispatch (m.type ());
            }

            if (simple)
            {
              Boolean fund (false);
              {
                IsFundamentalType t (fund);
                t.dispatch (m.type ());
              }

              String type (etype (m));
              String q_type (scope_ + L"::" + type);

              if (fund)
              {
                String ftype;
                {
                  std::wostringstream o;
                  MemberTypeName t (*this, o);
                  t.dispatch (m.type ());
                  ftype = o.str ();
                }

                os << "const " << q_type << " " << scope_ << "::" <<
                  edefault_value_member (m) << " (" << endl
                   << "::xsd::cxx::tree::traits< " << ftype << ", " <<
                  char_type << " >::create (" << endl
                   << string_type << " (" << L << "\"" <<
                  escape_str (m.value ()) << "\"), 0, 0, 0));"
                   << endl;
              }
              else
              {
                os << "const " << q_type << " " << scope_ << "::" <<
                  edefault_value_member (m) << " (" << endl
                   << string_type << " (" << L << "\"" <<
                  escape_str (m.value ()) << "\"), 0, 0, 0);"
                   << endl;
              }
            }
          }
        }

      private:
        String scope_;
      };


      struct Element: Traversal::Element, protected virtual Context
      {
        Element (Context& c)
            : Context (c)
        {
        }

        virtual Void
        traverse (Type& e)
        {
          if (skip (e))
            return;

          String const& aname (eaname (e));
          String const& mname (emname (e));

          String tr (etraits (e)); // traits type name
          String type (etype (e));

          // Check if we need to handle xsi:type and substitution groups.
          // If this element's type is anonymous or mapped to a fundamental
          // C++ type then we don't need to do anything. Note that if the
          // type is anonymous then it can't be derived from which makes it
          // impossible to substitute or dynamically-type with xsi:type.
          //
          SemanticGraph::Type& t (e.type ());


          Boolean fund (false);
          {
            IsFundamentalType traverser (fund);
            traverser.dispatch (t);
          }

          Boolean polymorphic (!fund
                               && options.value<CLI::generate_polymorphic> ()
                               && !t.context ().count ("anonymous"));

          os << "// " << e.name () << endl
             << "//" << endl;

          if (polymorphic)
          {
            // aCC cannot handle an inline call to type_factory_map_instance.
            //
            os << "{"
               << "::xsd::cxx::tree::type_factory_map< " << char_type <<
              " >& tfm (" << endl
               << "::xsd::cxx::tree::type_factory_map_instance< 0, " <<
              char_type << " > ());"
               << endl
               << "::std::auto_ptr< ::xsd::cxx::tree::type > tmp (" << endl
               << "tfm.create (" << endl
               << L << "\"" << e.name () << "\"," << endl
               << L << "\"" <<
              (e.qualified () ? e.namespace_ ().name () : L"") << "\"," << endl
               << "&::xsd::cxx::tree::factory_impl< " << type << " >," << endl
               << (e.global () ? "true" : "false") << ", " <<
              (e.qualified () ? "true" : "false") << ", " <<
              "i, n, f, this));"
               << endl
               << "if (tmp.get () != 0)"
               << "{";
          }
          else
          {
            if (e.qualified ())
            {
              os << "if (n.name () == " << L << "\"" << e.name () << "\" && " <<
                "n.namespace_ () == " << L << "\"" <<
                e.namespace_ ().name () << "\")"
                 << "{";
            }
            else
            {
              os << "if (n.name () == " << L << "\"" << e.name () << "\" && " <<
                "n.namespace_ ().empty ())"
                 << "{";
            }

            if (!fund)
            {
              os << "::std::auto_ptr< " << type << " > r (" << endl
                 << tr << "::create (i, f, this));"
                 << endl;
            }
          }


          // Checks. Disabled at the moment since they make it impossible to
          // parse valid instances where the same element is used in both
          // base and derived types. See the cxx/tree/name-clash/inheritance
          // test for details.
          //
          //
          if (max (e) != 1)
          {
            // sequence
            //
          }
          else if (min (e) == 0)
          {
            // optional
            //
            os << "if (!this->" << aname << " ())"
               << "{";
          }
          else
          {
            // one
            //
            os << "if (!" << emember (e) << ".present ())"
               << "{";
          }


          if (polymorphic || !fund)
          {
            if (polymorphic)
            {
              // Cast to static type.
              //
              os << "::std::auto_ptr< " << type << " > r (" << endl
                 << "dynamic_cast< " << type << "* > (tmp.get ()));"
                 << endl
                 << "if (r.get ())" << endl
                 << "tmp.release ();"
                 << "else" << endl
                 << "throw ::xsd::cxx::tree::not_derived< " << char_type <<
                " > ();"
                 << endl;
            }

            if (max (e) != 1)
            {
              // sequence
              //
              os << "this->" << aname << " ().push_back (r);";
            }
            else if (min (e) == 0)
            {
              // optional
              //
              os << "this->" << mname << " (r);";
            }
            else
            {
              // one
              //
              os << "this->" << mname << " (r);";
            }
          }
          else
          {
            if (max (e) != 1)
            {
              // sequence
              //
              os << "this->" << aname << " ().push_back (" << tr <<
                "::create (i, f, this));";
            }
            else if (min (e) == 0)
            {
              // optional
              //
              os << "this->" << mname << " (" << tr <<
                "::create (i, f, this));";
            }
            else
            {
              // one
              //
              os << "this->" << mname << " (" << tr <<
                "::create (i, f, this));";
            }
          }

          os << "continue;";

          // End of check block.
          //
          if (max (e) != 1)
          {
            // sequence
            //
          }
          else if (min (e) == 0)
          {
            // optional
            //
            os << "}";
          }
          else
          {
            // one
            //
            os << "}";
          }

          os << "}"; // if block

          if (polymorphic)
            os << "}";
        }
      };

      struct ElementTest : Traversal::Element, protected virtual Context
      {
        ElementTest (Context& c)
            : Context (c)
        {
        }

        virtual Void
        traverse (Type& e)
        {
          if (skip (e))
            return;

          if (max (e) == 1 && min (e) == 1)
          {
            // one
            //
            os << "if (!" << emember (e) << ".present ())"
               << "{"
               << "throw ::xsd::cxx::tree::expected_element< " <<
              char_type << " > (" << endl
               << L << "\"" << e.name () << "\"," << endl
               << L << "\"" <<
              (e.qualified () ? e.namespace_ ().name () : L"") << "\");"
               << "}";
          }
        }
      };

      struct Any: Traversal::Any, protected virtual Context
      {
        Any (Context& c)
            : Context (c)
        {
        }

        virtual Void
        traverse (Type& a)
        {
          String const& aname (eaname (a));
          String const& mname (emname (a));

          String const& ns (a.definition_namespace ().name ());
          String const& dom_doc (
            edom_document (
              dynamic_cast<SemanticGraph::Complex&> (a.scope ())));

          os << "// " << ename (a) << endl
             << "//" << endl
             << "if (";

          for (SemanticGraph::Any::NamespaceIterator i (a.namespace_begin ()),
                 e (a.namespace_end ()); i != e;)
          {
            if (*i == L"##any")
            {
              os << "true";
            }
            else if (*i == L"##other")
            {
              if (ns)
              {
                // Note that here I assume that ##other does not include
                // unqualified names in a schema with target namespace.
                // This is not what the spec says but that seems to be
                // the consensus.
                //
                os << "(!n.namespace_ ().empty () && " <<
                  "n.namespace_ () != " << L << "\"" << ns << "\")";
              }
              else
                os << "!n.namespace_ ().empty ()";
            }
            else if (*i == L"##local")
            {
              os << "n.namespace_ ().empty ()";
            }
            else if (*i == L"##targetNamespace")
            {
              os << "n.namespace_ () == " << L << "\"" << ns << "\"";
            }
            else
            {
              os << "n.namespace_ () == " << L << "\"" << *i << "\"";
            }

            if (++i != e)
              os << " ||" << endl;
          }

          os << ")"
             << "{";


          // Checks.
          //
          //
          if (max (a) != 1)
          {
            // sequence
            //
          }
          else if (min (a) == 0)
          {
            // optional
            //
            os << "if (!this->" << aname << " ())"
               << "{";
          }
          else
          {
            // one
            //
            os << "if (!" << emember (a) << ".present ())"
               << "{";
          }

          os << xerces_ns << "::DOMElement* r (" << endl
             << "static_cast< " << xerces_ns << "::DOMElement* > (" << endl
             << "this->" << dom_doc << " ().importNode (" << endl
             << "const_cast< " << xerces_ns << "::DOMElement* > (&i), true)));";

          if (max (a) != 1)
          {
            // sequence
            //
            os << "this->" << aname << " ().push_back (r);";
          }
          else if (min (a) == 0)
          {
            // optional
            //
            os << "this->" << mname << " (r);";
          }
          else
          {
            // one
            //
            os << "this->" << mname << " (r);";
          }

          os << "continue;";

          // End of check block.
          //
          if (max (a) != 1)
          {
            // sequence
            //
          }
          else if (min (a) == 0)
          {
            // optional
            //
            os << "}";
          }
          else
          {
            // one
            //
            os << "}";
          }

          os << "}"; // if block
        }
      };

      struct AnyTest : Traversal::Any, protected virtual Context
      {
        AnyTest (Context& c)
            : Context (c)
        {
        }

        virtual Void
        traverse (Type& a)
        {
          if (max (a) == 1 && min (a) == 1)
          {
            // one
            //
            os << "if (!" << emember (a) << ".present ())"
               << "{"
               << "throw ::xsd::cxx::tree::expected_element< " <<
              char_type << " > (" << endl
               << L << "\"*\"," << endl
               << L << "\"" << *a.namespace_begin () << "\");"
               << "}";
          }
        }
      };

      struct Attribute : Traversal::Attribute, protected virtual Context
      {
        Attribute (Context& c)
            : Context (c)
        {
        }

        virtual Void
        traverse (Type& a)
        {
          String const& mname (emname (a));

          String const& tr (etraits (a)); // traits type name

          if (a.qualified ())
          {
            os << "if (n.name () == " << L << "\"" << a.name () << "\" && " <<
              "n.namespace_ () == " << L << "\"" <<
              a.namespace_ ().name () << "\")"
               << "{";
          }
          else
          {
            os << "if (n.name () == " << L << "\"" << a.name () << "\" && " <<
              "n.namespace_ ().empty ())"
               << "{";
          }

          Boolean fund (false);
          {
            IsFundamentalType traverser (fund);
            traverser.dispatch (a.type ());
          }

          if (fund)
          {
            os << "this->" << mname << " (" << tr << "::create (i, f, this));";
          }
          else
          {
            String type (etype (a));

            os << "::std::auto_ptr< " << type << " > r (" << endl
               << tr << "::create (i, f, this));"
               << endl
               << "this->" << mname << " (r);";
          }

          os << "continue;"
             << "}";

        }
      };

      struct AnyAttribute: Traversal::AnyAttribute, protected virtual Context
      {
        AnyAttribute (Context& c)
            : Context (c)
        {
        }

        virtual Void
        traverse (Type& a)
        {
          String const& aname (eaname (a));

          String const& ns (a.definition_namespace ().name ());
          String const& dom_doc (
            edom_document (
              dynamic_cast<SemanticGraph::Complex&> (a.scope ())));

          os << "// " << ename (a) << endl
             << "//" << endl
             << "if (";

          // Note that we need to filter out namespaces for xmlns and
          // xsi.
          //
          for (SemanticGraph::Any::NamespaceIterator i (a.namespace_begin ()),
                 e (a.namespace_end ()); i != e;)
          {
            if (*i == L"##any")
            {
              os << "(n.namespace_ () != " <<
                "::xsd::cxx::xml::bits::xmlns_namespace< " << char_type <<
                " > () &&" << endl
                 << "n.namespace_ () != " <<
                "::xsd::cxx::xml::bits::xsi_namespace< " << char_type <<
                " > ())";
            }
            else if (*i == L"##other")
            {
              if (ns)
              {
                // Note that here I assume that ##other does not include
                // unqualified names in a schema with target namespace.
                // This is not what the spec says but that seems to be
                // the consensus.
                //
                os << "(!n.namespace_ ().empty () &&" << endl
                   << "n.namespace_ () != " << L << "\"" << ns << "\" &&" <<endl
                   << "n.namespace_ () != " <<
                  "::xsd::cxx::xml::bits::xmlns_namespace< " << char_type <<
                  " > () &&" << endl
                   << "n.namespace_ () != " <<
                  "::xsd::cxx::xml::bits::xsi_namespace< " << char_type <<
                  " > ())";
              }
              else
                os << "(!n.namespace_ ().empty () &&" << endl
                   << "n.namespace_ () != " <<
                  "::xsd::cxx::xml::bits::xmlns_namespace< " << char_type <<
                  " > () &&" << endl
                   << "n.namespace_ () != " <<
                  "::xsd::cxx::xml::bits::xsi_namespace< " << char_type <<
                  " > ())";
            }
            else if (*i == L"##local")
            {
              os << "n.namespace_ ().empty ()";
            }
            else if (*i == L"##targetNamespace")
            {
              os << "n.namespace_ () == " << L << "\"" << ns << "\"";
            }
            else
            {
              os << "n.namespace_ () == " << L << "\"" << *i << "\"";
            }

            if (++i != e)
              os << " ||" << endl;
          }

          os << ")"
             << "{"
             << xerces_ns << "::DOMAttr* r (" << endl
             << "static_cast< " << xerces_ns << "::DOMAttr* > (" << endl
             << "this->" << dom_doc << " ().importNode (" << endl
             << "const_cast< " << xerces_ns << "::DOMAttr* > (&i), true)));"
             << "this->" << aname << " ().insert (r);"
             << "continue;"
             << "}";
        }
      };


      struct AttributeTest : Traversal::Attribute, protected virtual Context
      {
        AttributeTest (Context& c)
            : Context (c)
        {
        }

        virtual Void
        traverse (Type& a)
        {
          String const& mname (emname (a));

          if (!a.optional () || a.default_ ())
          {
            os << "if (!" << emember (a) << ".present ())"
               << "{";

            if (a.default_ ())
            {
              Boolean fund (false);
              {
                IsFundamentalType traverser (fund);
                traverser.dispatch (a.type ());
              }

              String const& tr (etraits (a)); // traits type name

              if (fund)
              {
                os << "this->" << mname << " (" << endl
                   << tr << "::create (" << endl
                   << string_type << " (" << L << "\"" <<
                  escape_str (a.value ()) << "\")," << endl
                   << "&p.element (), f, this));";
              }
              else
              {
                os << "::std::auto_ptr< " << etype (a) << " > r (" << endl
                   << tr << "::create (" << endl
                   << string_type << " (" << L << "\"" <<
                  escape_str (a.value ()) << "\")," << endl
                   << "&p.element (), f, this));"
                   << endl
                   << "this->" << mname << " (r);";
              }
            }
            else
              os << "throw ::xsd::cxx::tree::expected_attribute< " <<
                char_type << " > (" << endl
                 << L << "\"" << a.name () << "\"," << endl
                 << L << "\"" <<
                (a.qualified () ? a.namespace_ ().name () : L"") << "\");";

            os << "}";
          }
        }
      };


      //
      //
      struct CtorBase : Traversal::Complex,
                        Traversal::Enumeration,
                        Traversal::Type,
                        protected virtual Context
      {
        // If base_arg is empty then no base argument is
        // generated.
        //
        CtorBase (Context& c, String const& base_arg)
            : Context (c), base_arg_ (base_arg)
        {
        }

        virtual Void
        traverse (SemanticGraph::Type&)
        {
          if (base_arg_)
            os << base_arg_;
        }

        virtual Void
        traverse (SemanticGraph::Enumeration&)
        {
          if (base_arg_)
            os << base_arg_;
        }

        Void
        traverse (SemanticGraph::Complex& c)
        {
          Args args (*this, base_arg_);
          args.traverse (c);
        }

      private:
        // No need to traverse AnyAttribute since it is always mapped
        // to a sequence.
        //
        struct Args : Traversal::Complex,
                      Traversal::Enumeration,
                      Traversal::Type,
                      Traversal::Any,
                      Traversal::Member,
                      protected virtual Context
        {
          Args (Context& c, String const& base_arg)
              : Context (c), base_arg_ (base_arg), first_ (true)
          {
            *this >> inherits_ >> *this;
            *this >> names_ >> *this;
          }

          virtual Void
          traverse (SemanticGraph::Type&)
          {
            if (base_arg_)
              os << comma () << base_arg_;
          }

          virtual Void
          traverse (SemanticGraph::Enumeration&)
          {
            if (base_arg_)
              os << comma () << base_arg_;
          }

          virtual Void
          traverse (SemanticGraph::Any& a)
          {
            if (!options.value<CLI::generate_wildcard> ())
              return;

            if (min (a) == 1 && max (a) == 1)
            {
              // one
              //
              os << comma () << ename (a);
            }
          }

          virtual Void
          traverse (SemanticGraph::Member& m)
          {
            if (skip (m))
              return;

            // Note that we are not including attributes with default
            // values here.
            if (min (m) == 1 && max (m) == 1)
            {
              // one
              //
              os << comma () << ename (m);
            }
          }

          using Complex::traverse;

        private:
          String
          comma ()
          {
            Boolean tmp (first_);
            first_ = false;
            return tmp ? "" : ",\n";
          }

        private:
          String base_arg_;
          Boolean first_;

          Traversal::Inherits inherits_;
          Traversal::Names names_;
        };

        String base_arg_;
      };


      struct CtorMember : Traversal::Member,
                          protected virtual Context
      {
        CtorMember (Context& c)
            : Context (c)
        {
        }

        virtual Void
        traverse (Type& m)
        {
          if (skip (m))
            return;

          String const& member (emember (m));

          if (max (m) != 1)
          {
            // sequence
            //
            os << "," << endl
               << "  " << member << " (" << flags_type << " (), this)";
          }
          else if (min (m) == 0 &&
                   (!m.is_a<SemanticGraph::Attribute> () || !m.default_ ()))
          {
            // optional
            //
            os << "," << endl
               << "  " << member << " (" << flags_type << " (), this)";
          }
          else
          {
            // one
            //

            if (min (m) == 0)
            {
              // This is an attribute with default value. We are going
              // to initialize it to its default value.
              //
              os << "," << endl
                 << "  " << member << " (" <<
                edefault_value (m) << " (), " << flags_type << " (), this)";
            }
            else
            {
              os << "," << endl
                 << "  " << member << " (" << ename (m) << ", " <<
                flags_type << " (), this)";
            }
          }
        }
      };

      struct CtorAny : Traversal::Any,
                       Traversal::AnyAttribute,
                       protected virtual Context
      {
        CtorAny (Context& c)
            : Context (c)
        {
        }

        virtual Void
        traverse (SemanticGraph::Any& a)
        {
          String const& member (emember (a));
          String const& dom_doc (
            edom_document (
              dynamic_cast<SemanticGraph::Complex&> (a.scope ())));

          if (max (a) != 1)
          {
            // sequence
            //
            os << "," << endl
               << "  " << member << " (this->" << dom_doc << " ())";
          }
          else if (min (a) == 0)
          {
            // optional
            //
            os << "," << endl
               << "  " << member << " (this->" << dom_doc << " ())";
          }
          else
          {
            // one
            //
            os << "," << endl
               << "  " << member << " (" << ename (a) << ", this->" <<
              dom_doc << " ())";
          }
        }

        virtual Void
        traverse (SemanticGraph::AnyAttribute& a)
        {
          String const& dom_doc (
            edom_document (
              dynamic_cast<SemanticGraph::Complex&> (a.scope ())));

          os << "," << endl
             << "  " << emember (a) << " (this->" << dom_doc << " ())";
        }
      };


      struct CopyMember : Traversal::Member,
                          protected virtual Context

      {
        CopyMember (Context& c, String const& arg_name_)
            : Context (c), arg_name (arg_name_)
        {
        }

        virtual Void
        traverse (SemanticGraph::Member& m)
        {
          if (skip (m))
            return;

          String const& member (emember (m));

          os << "," << endl
             << "  " << member << " (" <<
            arg_name << "." << member << ", f, this)";
        }

      private:
        String arg_name;
      };

      struct CopyAny : Traversal::Any,
                       Traversal::AnyAttribute,
                       protected virtual Context

      {
        CopyAny (Context& c, String const& arg_name_)
            : Context (c), arg_name (arg_name_)
        {
        }

        virtual Void
        traverse (SemanticGraph::Any& a)
        {
          String const& member (emember (a));
          String const& dom_doc (
            edom_document (
              dynamic_cast<SemanticGraph::Complex&> (a.scope ())));

          os << "," << endl
             << "  " << member << " (" <<
            arg_name << "." << member << ", this->" << dom_doc << " ())";
        }

        virtual Void
        traverse (SemanticGraph::AnyAttribute& a)
        {
          String const& member (emember (a));
          String const& dom_doc (
            edom_document (
              dynamic_cast<SemanticGraph::Complex&> (a.scope ())));

          os << "," << endl
             << "  " << member << " (" <<
            arg_name << "." << member << ", this->" << dom_doc << " ())";
        }

      private:
        String arg_name;
      };


      // C-tor initializer.
      //
      struct ElementCtorMember : Traversal::Member,
                                 protected virtual Context
      {
        ElementCtorMember (Context& c)
            : Context (c)
        {
        }

        virtual Void
        traverse (Type& m)
        {
          if (skip (m))
            return;

          String const& member (emember (m));

          os << "," << endl
             << "  " << member << " (f, this)";
        }
      };

      struct ElementCtorAny : Traversal::Any,
                              Traversal::AnyAttribute,
                              protected virtual Context
      {
        ElementCtorAny (Context& c)
            : Context (c)
        {
        }

        virtual Void
        traverse (SemanticGraph::Any& a)
        {
          String const& member (emember (a));
          String const& dom_doc (
            edom_document (
              dynamic_cast<SemanticGraph::Complex&> (a.scope ())));

          os << "," << endl
             << "  " << member << " (this->" << dom_doc << " ())";
        }

        virtual Void
        traverse (SemanticGraph::AnyAttribute& a)
        {
          String const& member (emember (a));
          String const& dom_doc (
            edom_document (
              dynamic_cast<SemanticGraph::Complex&> (a.scope ())));

          os << "," << endl
             << "  " << member << " (this->" << dom_doc << " ())";
        }
      };


      // Default c-tor member initializer.
      //
      struct DefaultCtorMemberInit : Traversal::Member,
                                     protected virtual Context
      {
        DefaultCtorMemberInit (Context& c)
            : Context (c)
        {
        }

        virtual Void
        traverse (Type& m)
        {
          if (skip (m))
            return;

          String const& member (emember (m));

          os << "," << endl
             << "  " << member << " (" << flags_type << " (), this)";
        }
      };

      struct DefaultCtorAnyInit : Traversal::Any,
                                  Traversal::AnyAttribute,
                                  protected virtual Context
      {
        DefaultCtorAnyInit (Context& c)
            : Context (c)
        {
        }

        virtual Void
        traverse (SemanticGraph::Any& a)
        {
          String const& member (emember (a));
          String const& dom_doc (
            edom_document (
              dynamic_cast<SemanticGraph::Complex&> (a.scope ())));

          os << "," << endl
             << "  " << member << " (this->" << dom_doc << " ())";
        }

        virtual Void
        traverse (SemanticGraph::AnyAttribute& a)
        {
          String const& member (emember (a));
          String const& dom_doc (
            edom_document (
              dynamic_cast<SemanticGraph::Complex&> (a.scope ())));

          os << "," << endl
             << "  " << member << " (this->" << dom_doc << " ())";
        }
      };

      // Test whether the base has comparison operators.
      //
      struct HasComparisonOperator: Traversal::Fundamental::Type,
                                    Traversal::List,
                                    Traversal::Union,
                                    Traversal::Complex,
                                    Traversal::Enumeration,
                                    Traversal::Member,
                                    Traversal::Any,
                                    Traversal::AnyAttribute,
                                    protected virtual Context
      {
        // generate should initially be false.
        //
        HasComparisonOperator (Context& c, Boolean& generate)
            : Context (c), generate_ (generate)
        {
          *this >> inherits_ >> *this;
          *this >> names_ >> *this;
        }

        virtual Void
        traverse (SemanticGraph::Fundamental::Type&)
        {
          // All built-in types are comparable.
          generate_ = true;
        }

        virtual Void
        traverse (SemanticGraph::List&)
        {
          generate_ = true;
        }

        virtual Void
        traverse (SemanticGraph::Union&)
        {
          generate_ = true;
        }

        virtual Void
        traverse (SemanticGraph::Enumeration& e)
        {
          Traversal::Enumeration::inherits (e);
        }

        virtual Void
        traverse (SemanticGraph::Complex& c)
        {
          Complex::names (c);

          if (!generate_)
            Complex::inherits (c);
        }

        virtual Void
        traverse (SemanticGraph::Member& m)
        {
          if (!skip (m))
            generate_ = true;
        }

        virtual Void
        traverse (SemanticGraph::Any&)
        {
          if (options.value<CLI::generate_wildcard> ())
            generate_ = true;
        }

        virtual Void
        traverse (SemanticGraph::AnyAttribute&)
        {
          if (options.value<CLI::generate_wildcard> ())
            generate_ = true;
        }

      private:
        Boolean& generate_;

        Traversal::Inherits inherits_;
        Traversal::Names names_;
      };

      //
      //
      struct MemberComparison : Traversal::Member,
                                protected virtual Context
      {
        MemberComparison (Context& c, Boolean& first)
            : Context (c), first_ (first)
        {
        }

        virtual Void
        traverse (Type& m)
        {
          if (skip (m))
            return;

          String const& aname (eaname (m));

          if (!first_)
            os << " &&" << endl;
          else
            first_ = false;

          os << "x." << aname << " () == y." << aname << " ()";
        }

      private:
        Boolean& first_;
      };

      struct AnyComparison : Traversal::Any,
                             Traversal::AnyAttribute,
                             protected virtual Context
      {
        AnyComparison (Context& c, Boolean& first)
            : Context (c), first_ (first)
        {
        }

        virtual Void
        traverse (SemanticGraph::Any& a)
        {
          String const& aname (eaname (a));

          if (!first_)
            os << " &&" << endl;
          else
            first_ = false;

          if (max (a) == 1 && min (a) == 1)
            os << "x." << aname << " ().isEqualNode (&y." << aname << " ())";
          else
            os << "x." << aname << " () == y." << aname << " ()";
        }

        virtual Void
        traverse (SemanticGraph::AnyAttribute& a)
        {
          String const& aname (eaname (a));

          if (!first_)
            os << " &&" << endl;
          else
            first_ = false;

          os << "x." << aname << " () == y." << aname << " ()";
        }

      private:
        Boolean& first_;
      };

      // Check whether a type has a parse() function (i.e., has any
      // members, recursively).
      //
      struct HasParseFunction: Traversal::Complex,
                               Traversal::Member,
                               Traversal::Any,
                               Traversal::AnyAttribute,
                               protected virtual Context
      {
        HasParseFunction (Context& c, Boolean& has)
            : Context (c), has_ (has)
        {
          *this >> names_ >> *this;
          *this >> inherits_ >> *this;
        }

        virtual Void
        traverse (SemanticGraph::Complex& c)
        {
          names (c);

          if (!has_)
            inherits (c);
        }

        virtual Void
        traverse (SemanticGraph::Member&)
        {
          has_ = true;
        }

        virtual Void
        traverse (SemanticGraph::Any&)
        {
          has_ = true;
        }

        virtual Void
        traverse (SemanticGraph::AnyAttribute&)
        {
          if (options.value<CLI::generate_wildcard> ())
            has_ = true;
        }

      private:
        Boolean& has_;

        Traversal::Names names_;
        Traversal::Inherits inherits_;
      };


      //
      //
      struct Complex : Traversal::Complex, protected virtual Context
      {
        Complex (Context& c)
            : Context (c),
              member_name_ (c),
              any_ (c),
              element_ (c),
              any_test_ (c),
              element_test_ (c),
              attribute_ (c),
              attribute_test_ (c),
              any_attribute_ (c),
              default_ctor_any_init_ (c),
              default_ctor_member_init_ (c),
              from_base_ctor_arg_ (c, true),
              ctor_any_ (c),
              ctor_member_ (c),
              element_ctor_any_ (c),
              element_ctor_member_ (c)
        {
          inherits_member_ >> member_name_;

          names_element_ >> element_;
          if (options.value<CLI::generate_wildcard> ())
            names_element_ >> any_;

          names_element_test_ >> element_test_;
          if (options.value<CLI::generate_wildcard> ())
            names_element_test_ >> any_test_;

          names_attribute_ >> attribute_;
          names_attribute_test_ >> attribute_test_;
          names_any_attribute_ >> any_attribute_;

          default_ctor_init_names_ >> default_ctor_member_init_;
          if (options.value<CLI::generate_wildcard> ())
            default_ctor_init_names_ >> default_ctor_any_init_;

          from_base_ctor_arg_names_ >> from_base_ctor_arg_;

          ctor_names_ >> ctor_member_;
          if (options.value<CLI::generate_wildcard> ())
            ctor_names_ >> ctor_any_;

          element_ctor_names_ >> element_ctor_member_;
          if (options.value<CLI::generate_wildcard> ())
            element_ctor_names_ >> element_ctor_any_;
        }


        virtual Void
        traverse (Type& c)
        {
          String name (ename (c));

          // If renamed name is empty then we do not need to generate
          // anything for this type.
          //
          if (renamed_type (c, name) && !name)
            return;

          Boolean string_based (false);
          {
            IsStringBasedType t (string_based);
            t.dispatch (c);
          }

          SemanticGraph::Enumeration* enum_base (0);
          {
            IsEnumBasedType t (enum_base);
            t.dispatch (c);
          }

          String base; // base type name
          if (c.inherits_p ())
          {
            std::wostringstream o;

            BaseTypeName base_type (*this, o);
            Traversal::Inherits inherits_type (base_type);

            // Cannot use inherits_none here because need the
            // result in a string.
            //
            inherits (c, inherits_type);

            base = o.str ();
          }
          else
            base = any_type;

          os << "// " << name << endl
             << "//" << endl
             << endl;

          //
          //
          {
            Member member (*this, name);
            Traversal::Names names_member (member);
            names (c, names_member);
          }

          Boolean generate_no_base_ctor (false);
          {
            GenerateWithoutBaseCtor t (generate_no_base_ctor);
            t.traverse (c);
          }

          // default c-tor
          //
          if (options.value<CLI::generate_default_ctor> ())
          {
            Boolean generate (false);
            {
              GenerateDefaultCtor t (*this, generate, generate_no_base_ctor);
              t.traverse (c);
            }

            if (generate)
            {
              os << name << "::" << endl
                 << name << " ()" << endl
                 << ": " << base << " ()";

              if (edom_document_member_p (c))
              {
                os << "," << endl
                   << "  " << edom_document_member (c) << " (" <<
                  "::xsd::cxx::xml::dom::create_document< " << char_type <<
                  " > ())";
              }

              names (c, default_ctor_init_names_);

              os << "{"
                 << "}";
            }
          }

          // c-tor (base, all-non-optional-members)
          //
          if (options.value<CLI::generate_from_base_ctor> ())
          {
            Boolean generate (false);
            {
              GenerateFromBaseCtor t (*this, generate);
              t.traverse (c);
            }

            if (generate)
            {
              String const& base_arg (ename (c.inherits ().base ()));

              os << name << "::" << endl
                 << name << " (const ";

              inherits (c, inherits_member_);

              os << "& " << base_arg;

              names (c, from_base_ctor_arg_names_);

              os << ")" << endl
                 << ": " << base << " (" << base_arg << ")";

              if (edom_document_member_p (c))
              {
                os << "," << endl
                   << "  " << edom_document_member (c) << " (" <<
                  "::xsd::cxx::xml::dom::create_document< " << char_type <<
                  " > ())";
              }

              names (c, ctor_names_);

              os << "{"
                 << "}";
            }
          }

          // c-tor (all-non-optional-members)
          //
          if (generate_no_base_ctor)
          {
            os << name << "::" << endl
               << name << " (";

            {
              CtorArgsWithoutBase ctor_args (*this, true, true);
              ctor_args.dispatch (c);
            }

            os << ")" << endl
               << ": " << base << " (";

            {
              CtorBase base (*this, "");
              Traversal::Inherits inherits_base (base);

              inherits (c, inherits_base);
            }

            os << ")";

            if (edom_document_member_p (c))
            {
              os << "," << endl
                 << "  " << edom_document_member (c) << " (" <<
                "::xsd::cxx::xml::dom::create_document< " << char_type <<
                " > ())";
            }

            names (c, ctor_names_);

            os << "{"
               << "}";
          }

          if (string_based)
          {
            if (enum_base != 0)
            {
              // c-tor (enum-value, all-non-optional-members)
              //
              String const& base_arg (ename (*enum_base));

              os << name << "::" << endl
                 << name << " (" << fq_name (*enum_base) << "::" <<
                evalue (*enum_base) << " " << base_arg;

              {
                CtorArgsWithoutBase ctor_args (*this, true, false);
                ctor_args.dispatch (c);
              }

              os << ")" << endl
                 << ": " << base << " (";

              {
                CtorBase base (*this, base_arg);
                Traversal::Inherits inherits_base (base);

                inherits (c, inherits_base);
              }

              os << ")";

              if (edom_document_member_p (c))
              {
                os << "," << endl
                   << "  " << edom_document_member (c) << " (" <<
                  "::xsd::cxx::xml::dom::create_document< " << char_type <<
                  " > ())";
              }

              names (c, ctor_names_);

              os << "{"
                 << "}";
            }

            String const& base_arg (ename (ultimate_base (c)));

            // c-tor (const char*, all-non-optional-members)
            //
            os << name << "::" << endl
               << name << " (const " << char_type << "* " << base_arg;

            {
              CtorArgsWithoutBase ctor_args (*this, true, false);
              ctor_args.dispatch (c);
            }

            os << ")" << endl
               << ": " << base << " (";

            {
              CtorBase base (*this, base_arg);
              Traversal::Inherits inherits_base (base);

              inherits (c, inherits_base);
            }

            os << ")";

            if (edom_document_member_p (c))
            {
              os << "," << endl
                 << "  " << edom_document_member (c) << " (" <<
                "::xsd::cxx::xml::dom::create_document< " << char_type <<
                " > ())";
            }

            names (c, ctor_names_);

            os << "{"
               << "}";


            // c-tor (const std::string&, all-non-optional-members)
            //
            os << name << "::" << endl
               << name << " (const " << string_type << "& " << base_arg;

            {
              CtorArgsWithoutBase ctor_args (*this, true, false);
              ctor_args.dispatch (c);
            }

            os << ")" << endl
               << ": " << base << " (";

            {
              CtorBase base (*this, base_arg);
              Traversal::Inherits inherits_base (base);

              inherits (c, inherits_base);
            }

            os << ")";

            if (edom_document_member_p (c))
            {
              os << "," << endl
                 << "  " << edom_document_member (c) << " (" <<
                "::xsd::cxx::xml::dom::create_document< " << char_type <<
                " > ())";
            }

            names (c, ctor_names_);

            os << "{"
               << "}";
          }

          // c-tor (ultimate-base, all-non-optional-members)
          //
          os << name << "::" << endl
             << name << " (";

          String base_arg;

          {
            CtorArgs ctor_args (*this, base_arg);
            ctor_args.dispatch (c);
          }

          os << ")" << endl
             << ": " << base << " (";

          {
            CtorBase base (*this, base_arg);
	    Traversal::Inherits inherits_base (base);

            inherits (c, inherits_base);
          }

          os << ")";

          if (edom_document_member_p (c))
          {
            os << "," << endl
               << "  " << edom_document_member (c) << " (" <<
              "::xsd::cxx::xml::dom::create_document< " << char_type <<
              " > ())";
          }

          names (c, ctor_names_);

          os << "{"
             << "}";


          // copy c-tor
          //
          os << name << "::" << endl
             << name << " (const " << name << "& x," << endl
             << flags_type << " f," << endl
             << container << "* c)" << endl
             << ": " << base << " (x, f, c)";

          if (edom_document_member_p (c))
          {
            os << "," << endl
               << "  " << edom_document_member (c) << " (" <<
              "::xsd::cxx::xml::dom::create_document< " << char_type <<
              " > ())";
          }

          {
            CopyAny copy_any (*this, "x");
            CopyMember copy_member (*this, "x");
            Traversal::Names names;

            names >> copy_member;

            if (options.value<CLI::generate_wildcard> ())
              names >> copy_any;

            Complex::names (c, names);
          }

          os << "{"
             << "}";

          //
          //
          Boolean he (has<Traversal::Element> (c));
          Boolean hae (has<Traversal::Any> (c));

          Boolean ha (has<Traversal::Attribute> (c));
          Boolean haa (has<Traversal::AnyAttribute> (c));

          Boolean gen_wildcard (options.value<CLI::generate_wildcard> ());

          //
          //
          if (!options.value<CLI::suppress_parsing> ())
          {
            // c-tor (xercesc::DOMElement)
            //
            os << name << "::" << endl
               << name << " (const " << xerces_ns << "::DOMElement& e," << endl
               << flags_type << " f," << endl
               << container << "* c)" << endl
               << ": " << base << " (e, f";

            if (he || ha || hae || (haa && gen_wildcard))
              os << " | " << flags_type << "::base";

            os << ", c)";

            if (edom_document_member_p (c))
            {
              os << "," << endl
                 << "  " << edom_document_member (c) << " (" <<
                "::xsd::cxx::xml::dom::create_document< " << char_type <<
                " > ())";
            }

            names (c, element_ctor_names_);

            os << "{";

            //@@ throw if p is no exhausted at the end.
            //
            if (he || ha || hae || (haa && gen_wildcard))
              os << "if ((f & " << flags_type << "::base) == 0)"
                 << "{"
                 << parser_type << " p (e);"
                 << "this->" << unclash (name, "parse") << " (p, f);"
                 << "}";

            os << "}";

            Boolean simple (true);
            {
              IsSimpleType t (simple);
              t.dispatch (c);
            }

            if (simple)
            {
              // c-tor (xercesc::DOMAttr)
              //
              os << name << "::" << endl
                 << name << " (const " << xerces_ns << "::DOMAttr& a," << endl
                 << flags_type << " f," << endl
                 << container << "* c)" << endl
                 << ": " << base << " (a, f, c)"
                 << "{"
                 << "}";

              // c-tor (string const&, xercesc::DOMElement)
              //
              os << name << "::" << endl
                 << name << " (const " << string_type << "& s," << endl
                 << "const " << xerces_ns << "::DOMElement* e," << endl
                 << flags_type << " f," << endl
                 << container << "* c)" << endl
                 << ": " << base << " (s, e, f, c)"
                 << "{"
                 << "}";
            }

            if (he || ha || hae || (haa && gen_wildcard))
            {
              Boolean has_base_parse (false);
              if (c.inherits_p ())
              {
                HasParseFunction test (*this, has_base_parse);
                test.dispatch (c.inherits ().base ());
              }

              os << "void " << name << "::" << endl
                 << unclash (name, "parse") << " (" <<
                parser_type << "& p," << endl
                 << flags_type <<
                (he || ha || has_base_parse ? " f" : "") << ")"
                 << "{";

              // Allow the base to parse its part.
              //
              if (has_base_parse)
                os << "this->" << base << "::parse (p, f);"
                   << endl;

              if (he || hae)
              {
                os << "for (; p.more_elements (); p.next_element ())"
                   << "{"
                   << "const " << xerces_ns << "::DOMElement& i (" <<
                  "p.cur_element ());"
                   << "const " << qname_type << " n (" << endl
                   << "::xsd::cxx::xml::dom::name< " << char_type << " > (i));"
                   << endl;

                names (c, names_element_);

                os << "break;"
                   << "}";

                // Make sure all non-optional elements are set.
                //
                names (c, names_element_test_);
              }

              if (ha || (haa && gen_wildcard))
              {
                if (has_base_parse)
                  os << "p.reset_attributes ();"
                     << endl;

                os << "while (p.more_attributes ())"
                   << "{"
                   << "const " << xerces_ns << "::DOMAttr& i (" <<
                  "p.next_attribute ());"
                   << "const " << qname_type << " n (" << endl
                   << "::xsd::cxx::xml::dom::name< " << char_type << " > (i));"
                   << endl;

                names (c, names_attribute_);

                // Generate anyAttribute code after all the attributes.
                //
                if (gen_wildcard)
                  names (c, names_any_attribute_);


                // os << "{"  // else block
                // @@
                // This doesn't play well with inheritance because we
                // don't expect base's attributes. Also there are other
                // "special" attributes such as xmlns, etc.
                //
                // << "throw ::xsd::cxx::tree::unexpected_attribute < " <<
                // char_type << " > (n, " << L << "\"" << "\");"
                // << "}";

                os << "}"; // while loop

                // Make sure all non-optional attributes are set.
                //
                names (c, names_attribute_test_);
              }

              os << "}";
            }
          }

          // _clone
          //
          os << name << "* " << name << "::" << endl
             << "_clone (" << flags_type << " f," << endl
             << container << "* c) const"
             << "{"
             << "return new " << name << " (*this, f, c);"
             << "}";

          // d-tor
          //
          os << name << "::" << endl
             << "~" << name << " ()"
             << "{"
             << "}";

          // Register with type factory map.
          //
          if (options.value<CLI::generate_polymorphic> () &&
              !c.context ().count ("anonymous"))
          {
            // Note that we are using the original type name.
            //
            os << "static" << endl
               << "::xsd::cxx::tree::type_factory_initializer< 0, " <<
              char_type << ", " << ename (c) << " >" << endl
               << "_xsd_" << ename (c) << "_type_factory_init (" << endl
               << L << "\"" << c.name () << "\"," << endl
               << L << "\"" << xml_ns_name (c) << "\");"
               << endl
               << endl;
          }

          // Comparison operators.
          //
          if (options.value<CLI::generate_comparison> () &&
              (he || ha || !c.inherits_p () ||
               ((hae || haa) && gen_wildcard)))
          {
            Boolean base_comp (false);

            if (c.inherits_p ())
            {
              HasComparisonOperator test (*this, base_comp);
              test.dispatch (c.inherits ().base ());
            }

            Boolean has_body (he || ha || base_comp ||
                              ((hae || haa) && gen_wildcard));

            os << "bool" << endl
               << "operator== (const " << name << "&" <<
              (has_body ? " x" : "") << ", " <<
              "const " << name << "&" << (has_body ? " y" : "") << ")"
               << "{"
               << "return ";

            if (base_comp)
              os << "static_cast< const " << base << "& > (x) ==" << endl
                 << "static_cast< const " << base << "& > (y)";

            {
              Boolean first (!base_comp);
              AnyComparison any (*this, first);
              MemberComparison member (*this, first);
              Traversal::Names names;

              names >> member;

              if (gen_wildcard)
                names >> any;

              Complex::names (c, names);
            }

            if (!has_body)
              os << "true";

            os << ";"
               << "}";

            os << "bool" << endl
               << "operator!= (const " << name << "& x, " <<
              "const " << name << "& y)"
               << "{"
               << "return !(x == y);"
               << "}";
          }
        }

      private:
        Traversal::Inherits inherits_member_;
        MemberTypeName member_name_;

        Traversal::Names names_element_;
        Traversal::Names names_element_test_;

        Traversal::Names names_attribute_;
        Traversal::Names names_any_attribute_;
        Traversal::Names names_attribute_test_;

        Any any_;
        Element element_;

        AnyTest any_test_;
        ElementTest element_test_;

        Attribute attribute_;
        AttributeTest attribute_test_;

        AnyAttribute any_attribute_;

        DefaultCtorAnyInit default_ctor_any_init_;
        DefaultCtorMemberInit default_ctor_member_init_;
        Traversal::Names default_ctor_init_names_;

        FromBaseCtorArg from_base_ctor_arg_;
        Traversal::Names from_base_ctor_arg_names_;

        CtorAny ctor_any_;
        CtorMember ctor_member_;
        Traversal::Names ctor_names_;

        ElementCtorAny element_ctor_any_;
        ElementCtorMember element_ctor_member_;
        Traversal::Names element_ctor_names_;
      };


      // Generate substitution group map entries.
      //
      struct GlobalMember : Traversal::Element,
                            GlobalElementBase,
                            protected virtual Context
      {
        GlobalMember (Context& c)
            : Context (c), GlobalElementBase (c), type_name_ (c)
        {
          belongs_ >> type_name_;
        }

        virtual Void
        traverse (Type& e)
        {
          if (!generate_p (e))
            return;

          if (e.substitutes_p () &&
              options.value<CLI::generate_polymorphic> ())
          {
            String const& name (ename (e));
            Type& r (e.substitutes ().root ());

            os << "static" << endl
               << "::xsd::cxx::tree::type_factory_initializer< 0, " <<
              char_type << ", ";

            belongs (e, belongs_);

            os << " >" << endl
               << "_xsd_" << name << "_element_factory_init (" << endl
               << L << "\"" << r.name () << "\"," << endl
               << L << "\"" << r.namespace_ ().name () << "\"," << endl
               << L << "\"" << e.name () << "\"," << endl
               << L << "\"" << e.namespace_ ().name () << "\");"
               << endl
               << endl;
          }
        }

      private:
        Traversal::Belongs belongs_;
        MemberTypeName type_name_;
      };
    }

    Void
    generate_tree_source (Context& ctx,
                          UnsignedLong first,
                          UnsignedLong last)
    {
      if (ctx.options.value<CLI::generate_wildcard> ())
      {
        ctx.os << "#include <xsd/cxx/xml/dom/wildcard-source.hxx>" << endl
               << endl;
      }

      if (!ctx.options.value<CLI::suppress_parsing> ())
        ctx.os << "#include <xsd/cxx/xml/dom/parsing-source.hxx>" << endl
               << endl;

      if (ctx.options.value<CLI::generate_polymorphic> ())
      {
        ctx.os << "#include <xsd/cxx/tree/type-factory-map.hxx>" << endl
               << endl;

        ctx.os << "namespace _xsd"
               << "{"
               << "static" << endl
               << "::xsd::cxx::tree::type_factory_plate< 0, " <<
          ctx.char_type << " >" << endl
               << "type_factory_plate_init;"
               << "}";
      }

      Traversal::Schema schema;
      Traversal::Sources sources;
      Traversal::Names names_ns, names;
      Namespace ns (ctx, first, last);

      List list (ctx);
      Union union_ (ctx);
      Complex complex (ctx);
      Enumeration enumeration (ctx);
      GlobalMember member (ctx);

      schema >> sources >> schema;
      schema >> names_ns >> ns >> names;

      names >> list;
      names >> union_;
      names >> complex;
      names >> enumeration;
      names >> member;

      schema.dispatch (ctx.schema_root);
    }
  }
}
