/*
			  Y Sound Systems

		 Client to Server Library Functions


	All Y client programs needs to #include this file and link to
	the libY2 library.

	For contact and programming information, see:

	http://wolfpack.twu.net/YIFF

 */

#ifndef YLIB_H
#define YLIB_H

#include <stdio.h>
#include <sys/types.h>

#ifdef __cplusplus
extern "C" {
#endif

#include "Y.h"


/*
 *	Volume structure:
 */
typedef struct {

	/* Coefficient = (val / ((u_int16_t)-1)) */

	u_int16_t	left,
			right,
			back_left,
			back_right;

} YVolumeStruct;

/*
 *	Audio CD track structure:
 */
typedef struct {

	int		number;		/* Track number, starts from 1
					 * (0 indicates end of track list
					 * when getting track list) */
	unsigned long	start_time;	/* In seconds */
	unsigned long	length;		/* In seconds */
	char		name[YAudioCDTrackNameMax];

} YAudioCDTrackStruct;


/*
 *	Event structures:
 */

/* Change mode event structure */
typedef struct {

	Boolean		preset;		/* Is preset if True */

	/* If preset is True, these members contain correct info */
	char		mode_name[YAudioNameMax];

	/* If preset is False, these members contain correct info */
	int		sample_size;
	int		channels;
	int		sample_rate;
	int		direction;
	int		allow_fragmenting;
	int		num_fragments;
	int		fragment_size_bytes;

} YEventAudioChange;

/* Y server stats event structure */
typedef struct {

	int		protocol_version_major;
	int		protocol_version_minor;
	Coefficient	cycle_load;
	unsigned long	start_time;

	char		vendor_name[YVendorNameMax];	/* Name of Y server vendor */
	char		server_name[YServerNameMax];	/* Name of Y server */
	char		sound_name[YSoundNameMax];	/* Name of Sound device */

	/* More items to be added in future */

} YEventServerStats;

/* Change cycle event structure */
typedef struct {

	long		cycle_us;

} YEventCycleChange;

/* Disconnect event structure */
typedef struct {

	int		reason;		/* Why you were disconnected */

} YEventDisconnect;

/* Host add or remove event structure */
typedef struct {

	int		op;		/* 0 for remove, 1 for add */
	YIPUnion	ip;

} YEventHost;

/* Sound kill event structure */
typedef struct {

	YID		yid;

} YEventSoundKill;

/* Mixer event structure */
typedef struct {

	int		code;		/* Mixer device code */

	Coefficient	value[YMixerValues];	/* Argument values */

} YEventMixer;

/* Sound play event structure */
typedef struct {

/* These flag definations need to match those of the server */
#define YPlayValuesFlagYID              (1 << 1)
#define YPlayValuesFlagPosition         (1 << 2)
#define YPlayValuesFlagLength           (1 << 3)
#define YPlayValuesFlagRepeats          (1 << 4)
#define YPlayValuesFlagTotalRepeats     (1 << 5)
#define YPlayValuesFlagVolume           (1 << 6)
#define YPlayValuesFlagSampleRate       (1 << 7)

	unsigned long	flags;

	YID		yid;

	YDataPosition	position;	/* Current play position in bytes */
	YDataLength	length;		/* Length of audio data in bytes */

	int		repeats,	/* Number of repeats so far */
			total_repeats;	/* Total number of repeats */

	Coefficient     left_volume,    /* Volume from 0.0 to 1.0 */
			right_volume,
			back_left_volume,
			back_right_volume;

	int		sample_rate;	/* Applied sample rate in Hz */

} YEventSoundPlay;

/* Sound object attributes event structure */
typedef struct {

	int		format;		/* One of SndObjType* */
	int		sample_size;
	int		channels;
	int		sample_rate;
	YDataLength	length;		/* Length of audio data in bytes */

	char		path[YPathMax];

} YEventSoundObjectAttributes;

/* Shutdown event structure */
typedef struct {

	int		reason;		/* Why the server shut down */

} YEventShutdown;

/* Sync event structure */
typedef struct {

	long		cycle_ahead_us;

} YEventSync;

/* Audio stats structure */ 
typedef struct { 

	int		cycle_set;

	long		cycle_us;
	long		compensated_cycle_us;

	long		write_ahead_us;
	long		cumulative_latency_us;

	int		sample_size;
	int		channels;
	int		sample_rate;
	int		bytes_per_sec;

	Boolean		allow_fragments;
	int		num_fragments;
	int		fragment_size;

	Boolean		flip_stereo;
	int		direction;

	char		mode_name[YAudioNameMax];	/* Can be "" if Audio
							 * values were not set
							 * by a preset Audio
							 * mode */

} YEventAudioStats;

/* Client message structure */
typedef struct {

	int		format;		/* One of YClientMessageFormat* */
	int		type;		/* One of YClientMessageType*, note
					 * that clients can also define their
					 * own types so unexpected values
					 * for type should be ignored
					 */
	char		message[YClientMessageMessageMax];
	int		length;		/* Length of message in bytes */

} YEventClientMessage;

/* YSHM sound structure */
typedef struct {

	int		minor_op_code;
	int		shm_id;
	void		*ptr;		/* Pointer to shared sound buffer */
	int		length;		/* Length of shared sound buffer
					 * in bytes */

} YEventYSHMSound;

/* Y Audio CD stats structure */
typedef struct {

	int		minor_op_code;
	int		track_number;
	unsigned long	track_start_time;
	unsigned long	track_length;
	char		track_name[YAudioCDTrackNameMax];

} YEventAudioCDStats;

/* Main YEvent structure */
typedef struct {

	int		type;	/* Event type code (aka the major op code) */

	YEventAudioChange	audio;
	YEventServerStats	serverstats;
	YEventCycleChange	cycle;
	YEventDisconnect	disconnect;
	YEventHost		host;
	YEventSoundKill		kill;
	YEventMixer		mixer;
	YEventSoundPlay		play;
	YEventSoundObjectAttributes	sndobjattributes;
	YEventShutdown		shutdown;
	YEventSync		sync;
	YEventAudioStats	audiostats;

	/* New in version 2.14 */
	YEventClientMessage	clientmessage;
	YEventYSHMSound		yshmsound;
	YEventAudioCDStats	audiocdstats;

} YEvent;


/*
 *      Connection to YIFF server structure:
 *
 *	All members here are to be cosidered private.
 */
typedef struct {

	int		fd;

	int		we_started_server;	/* 1 if we started the Y server */

	/* Queued events */
	int		total_queued_events;
	YEvent		*queued_event;

	YID		prev_generated_yid;

	/* Receive buffer */
	u_int8_t	*buf;
	YDataLength     buf_len,        /* Allocated length of buf */
			buf_cont;       /* Contents in buf */

} YConnection;

/*
 *	Audio mode list structure:
 */
typedef struct {

	char		name[YAudioNameMax];

	int		sample_rate;
	int		channels;
	int		sample_size;
	int		fragment_size_bytes;
	char		direction;
	char		allow_fragmenting;
	int		num_fragments;

} YAudioModeValuesStruct;



/*
 *      Attempts to connect to the Y server at the address and port
 *      specified by con_arg.
 *
 *      con_arg is a null terminated string of the format
 *      "<address>:<port>", ie "127.0.0.1:9433".
 *
 *      If the connect failed and start_arg is not NULL, then
 *      start_arg will be executed as the command to start the
 *      server.  After which a connect attempt will be made again.
 *
 *      Returns a pointer to a YConnection structure or NULL on
 *      error.  To disconnect and deallocate the YConnection structure,
 *      pass it to YIFFDisconnect().
 */
extern YConnection *YOpenConnection(
	const char *start_arg,	/* Can be NULL for don't start if needed */
	const char *con_arg	/* Can be NULL for default address and port */
);

/*
 *      Changes the Y server's Audio values.
 *
 *      Note that not all combinations ov values may work, since the
 *      Y server may not support certain combinations of values.  It is
 *      recommended that you use YSetModePreset() whenever possible
 *      since it uses preconfigured values that are known to work.
 *
 *      Returns -1 on failure, and 0 on success.
 */
extern int YSetAudioModeValues(
	YConnection *con,
	int sample_size,
	int channels,
	int sample_rate,
	int direction,		/* 0 = play, only 0 is supported */
	int allow_fragmenting,
	int num_fragments,
	int fragment_size
);

/*
 *      Changes the Y server's Audio mode to the values specified by
 *      the preset Audio mode who matches the given name.
 *
 *      Returns -1 on failure, and 0 on success.
 */
extern int YChangeAudioModePreset(YConnection *con, const char *name);

/*
 *	Gets Audio device statistics and stores them in the
 *	YEventAudioStats buffer.
 *
 *      Returns -1 on failure, and 0 on success.
 */
extern int YGetAudioStats(YConnection *con, YEventAudioStats *buf);


/*
 *      Gets list of preset Audio modes and their values.
 *
 *      Can return NULL on error.
 */
extern YAudioModeValuesStruct **YGetAudioModes(
	YConnection *con,
	int *count
);

/*
 *      Deletes the list of Audio modes.
 */
extern void YFreeAudioModesList(YAudioModeValuesStruct **list, int count);

/*
 *      Matches an Audio mode with the given values.
 *
 *      If an Audio mode is matched then buf will be set with the
 *      values of the matched Audio mode and True is returned,
 *      otherwise False is returned.
 */
extern Boolean YMatchAudioModeValues(
	YConnection *con,
	YAudioModeValuesStruct *buf,
	int sample_rate,
	int sample_size,
	int channels,
	int direction	/* 0 = play, 1 = record */
);


/*
 *      Gets Y server statistics and stores them in the given
 *      YEventServerStats buffer.
 *
 *      Returns -1 on error and 0 on success.
 */
extern int YGetServerStats(YConnection *con, YEventServerStats *buf);


/*
 *      Calculates the theoretical cycle (in microseconds) for
 *      the Y server given the Audio values.  The Audio values need
 *      not be the current Audio values of the recorder.
 *
 *      If the connection pointer con is NULL, then a general
 *      recommendation answer is returned. This value may not
 *      work with the recorder that you are connected to.
 */
extern long YCalculateCycle(
	YConnection *con,
	int sample_rate, int channels,
	int sample_size, int fragment_size
);

/*
 *      Set cycle in microseconds.
 *
 *      Returns -1 on error and 0 on success.
 */
extern int YSetCycle(YConnection *con, long us);

/*
 *      Syncs all of the Y server's resources, both the Y server and the
 *      client will also be synced.
 *
 *      If block is True then this operation will block (in finite time)
 *      to ensure that the Y server is up to sync.
 */
extern void YSyncAll(YConnection *con, Boolean block);


/*
 *      Adds the IP address specified by ip to the Y server's access
 *	list.
 *
 *      Returns -1 on failure, and 0 on success.
 */
extern int YAddHost(YConnection *con, YIPUnion *ip);

/*
 *	Removes the IP address specified by ip from the Y server's
 *	access list. If the specified ip does not exist in the Y server's
 *	access list then no operation will be performed.
 *
 *	Returns -1 on failure, and 0 on success.
 */
extern int YRemoveHost(YConnection *con, YIPUnion *ip);

/*
 *      Sets specified mixer channel values.
 *
 *      Returns -1 on failure, and 0 on success.
 */
extern int YSetMixerChannel(
	YConnection *con,
	int mixer_channel_code,		/* One of YMixerCode* */
	Coefficient value1, Coefficient value2	/* 0.0 to 1.0 */
);
extern int YSetMixerChannelQuad(
	YConnection *con,
	int mixer_channel_code,
	Coefficient value1, Coefficient value2,	/* 0.0 to 1.0 */
	Coefficient value3, Coefficient value4	/* 0.0 to 1.0 */
);


/*
 *      Gets specified mixer channel values.
 *
 *      Returns -1 on failure, and 0 on success.
 */
extern int YGetMixerChannel(
	YConnection *con,
	int mixer_channel_code,		/* One of YMixerCode* */
	Coefficient *value1, Coefficient *value2	/* 0.0 to 1.0 */
);
extern int YGetMixerChannelQuad(
	YConnection *con,
	int mixer_channel_code,
	Coefficient *value1, Coefficient *value2,	/* 0.0 to 1.0 */
	Coefficient *value3, Coefficient *value4	/* 0.0 to 1.0 */
);


/*
 *      Starts playing the sound object specified by path, returns
 *      the YID identifying the sound object being played or YIDNULL on
 *      failure.
 */
extern YID YStartPlaySoundObjectSimple(
	YConnection *con,
	const char *path	/* Path on disk on server's computer */
);

/*
 *      Starts playing the sound object specified by path, returns
 *      the YID indentifying the sound object being played or YIDNULL on
 *      failure.
 *
 *      The only criteria for a success here is that the sound object
 *      exist.
 */
extern YID YStartPlaySoundObject(
	YConnection *con,
	const char *path,	/* Path on disk on server's computer */
	YEventSoundPlay *value
);

/*
 *      Gets sound object attributes, returns -1 on error or
 *      0 on success.
 *
 *      Note, the sound object need not be playing in order to
 *      fetch its attributes.
 */
extern int YGetSoundObjectAttributes(
	YConnection *con, const char *path,
	YEventSoundObjectAttributes *buf
);

/*
 *      Modifies the play values of the sound object specified by yid
 *      that is already playing.
 */
extern void YSetPlaySoundObjectValues(
	YConnection *con, YID yid, YEventSoundPlay *value
);

/*
 *	Gets the play values of the sound object specified by yid
 *      that is already playing.
 *
 *	Returns 0 on success and non-zero on error.
 */
extern int YGetPlaySoundObjectValues(
	YConnection *con, YID yid, YEventSoundPlay *value
);

/*
 *      Destroys (stops/kills/terminates) the sound object
 *      being played, indetified by yid. If the yid does not
 *      match any sound object currently being played, no
 *      operation will be performed.
 */
extern void YDestroyPlaySoundObject(YConnection *con, YID yid);


/*
 *	Sends a client message.
 *
 *      Returns -1 on failure, and 0 on success.
 */
extern int YSendClientMessage(
	YConnection *con,
	Boolean notify_self,	/* Send ourself this YClientMessage event too */ 	
	int format,		/* One of YClientMessageFormat* */
	int type,		/* One of YClientMessageType*, note that
				 * clients can also define their own types.
				 * so unexpected values may appear.
				 */
	const char *message,	/* Message */
	int length		/* Length of message in bytes */
);


/*
 *      Opens the shared memory segment of the sound buffer on the Y
 *      server.
 *
 *      Returns the pointer to the memory segment, the length of the
 *      buffer in bytes, and the shared memory segment id.
 *
 *      The returned pointer must be closed by calling YSHMClose().
 *
 *      Can return NULL on error.
 */
extern void *YSHMOpenSound(YConnection *con, int *length, int *id);

/*
 *      Closes the shared memory segment of the sound buffer that was
 *      previously opened by a call to YSHMOpenSound().
 */
extern void YSHMCloseSound(YConnection *con, void *ptr);


/*
 *	Plays audio CD track specified by track_number.
 *
 *      Returns -1 on failure, and 0 on success.
 */
extern int YPlayAudioCDTrack(YConnection *con, int track_number);

/*
 *	Stops audio CD playing.
 *
 *      Returns -1 on failure, and 0 on success.
 */
extern int YStopAudioCD(YConnection *con);

/*
 *	Ejects audio CD.
 *
 *      Returns -1 on failure, and 0 on success.
 */
extern int YEjectAudioCD(YConnection *con);


/*
 *      Gets list of Audio CD tracks.
 *
 *      Can return NULL on error.
 */
extern YAudioCDTrackStruct **YGetAudioCDTracks(
	YConnection *con, int *count
);

/*
 *      Deallocates the list of Audio CD track structures and the
 *      pointer array.
 */
extern void YFreeAudioCDTracksList(YAudioCDTrackStruct **list, int count);


/*
 *      Disconnects from YIFF server and frees all allocated
 *      resources.
 *
 *      The pointer con may no longer be used after a call to this
 *      function.
 */
extern void YCloseConnection(YConnection *con, Boolean no_shutdown);

/*
 *      Shuts down the YIFF server, thus disconnecting and
 *      freeing all allocated resources.
 *
 *      The pointer con may no longer be used after a call to this
 *      function.
 */
extern void YShutdownServer(YConnection *con);


/*
 *      Get next (oldest) event from the server and stores it on
 *      the pointer to the YEvent event.
 *
 *      If block is set to True, then execution will be
 *      blocked untill next event is recieved.
 *
 *      If the recieved event is of type YDisconnect or YShutdown then
 *      the given con structure's connection descriptor will be closed
 *      and reset to -1.
 *
 *	Returns positive if an event is fetched.
 */
extern int YGetNextEvent(
	YConnection *con,
	YEvent *event,
	Boolean block
);

/*
 *      Puts back the event onto the queue as the newest (highest
 *      index) event.
 *
 *      So all other older events (if any) will be returned before this
 *      one when YGetNextEvent() is called.
 */
extern void YPutBackEvent(YConnection *con, YEvent *event);


#ifdef __cplusplus
}
#endif

#endif	/* YLIB_H */
