# -*- coding: utf-8 -*-
## 
## Copyright (C)2005 Ingeniweb

## This program is free software; you can redistribute it and/or modify
## it under the terms of the GNU General Public License as published by
## the Free Software Foundation; either version 2 of the License, or
## (at your option) any later version.

## This program is distributed in the hope that it will be useful,
## but WITHOUT ANY WARRANTY; without even the implied warranty of
## MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
## GNU General Public License for more details.

## You should have received a copy of the GNU General Public License
## along with this program; see the file COPYING. If not, write to the
## Free Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

"""
Lock Mixin core class
"""
__docformat__ = 'restructuredtext'



# Zope imports
from AccessControl import ClassSecurityInfo
from DateTime import DateTime

# CMF imports
from Products.CMFCore import CMFCorePermissions
from Products.CMFCore.utils import getToolByName

# Archetypes imports
from Products.Archetypes.public import Schema, BooleanField

from Products.PloneExFile.Permissions import LockRole
from Products.PloneExFile.LockWidget import LockWidget
from Products.PloneExFile.config import Log, LOG_DEBUG

schema = Schema((
    # Locked Mixin
    BooleanField(
        'is_locked',
        accessor='isLocked',
        mutator='lock',
        default=False,
        widget=LockWidget(
            label='lock ?',
            label_msgid='label_lock',
            visible={'view' : 'hidden', 'edit' : 'visible'},
            ),
        ),
    ))

class LockMixin:

    security = ClassSecurityInfo()

    LockOwnerRole = LockRole
    ToggleLockPermission = CMFCorePermissions.ModifyPortalContent
    ForceUnlockPermission = CMFCorePermissions.ModifyPortalContent

    schema = schema

    _locked_by = ''
    _locked_date = None
    _unlocked_by = ''
    _unlocked_date = None
    _lock_history = None

    # Init method
    security.declarePrivate('__init__')
    def __init__(self):
        """__init__(self)"""
        self._locked_by = ''
        self._locked_date = None
        self._unlocked_by = ''
        self._unlocked_date = None
        self._lock_history = []

    #                                                                          #
    #                             LOCKING SUPPORT                              #
    #                                                                          #

    security.declareProtected(CMFCorePermissions.View, 'hasLockSupport')
    def hasLockSupport(self):
        return True

    security.declareProtected(CMFCorePermissions.View, 'isLocked')
    def isLocked(self):
        """Is the content locked ?
        """
        # Return lock support
        if not self.hasLockSupport():
            return False

        return self.getField('is_locked').get(self)

    security.declareProtected(ToggleLockPermission, 'lock')
    def lock(self, value, **kwargs):
        """
        lock or unlock
        False -> unlock
        True -> lock

        Returns islocked
        """

        # Return lock support
        if not self.canToggleLock():
            return self.isLocked()

        # Test permission
        mtool = getToolByName(self, 'portal_membership')
        member = mtool.getAuthenticatedMember()
        # Get member information
        username = member.getUserName()
        userid = member.getMemberId()

        # Remove all LockOwner local roles
        for user, roles in self.get_local_roles():
            if self.LockOwnerRole in roles:
                new_roles = filter(lambda x: x != self.LockOwnerRole, list(roles))

                if new_roles:
                    self.manage_setLocalRoles(user, new_roles)
                else:
                    self.manage_delLocalRoles([user,],)

        if not value or value == '0' or value == 'False':
            value = False
        else:
            value = True

        # Apply locking
        if value:
            self.setLockUser(username)
            self.manage_addLocalRoles(userid, roles = [self.LockOwnerRole,])
        else:
            self.setUnlockUser(username)

        field = self.getField('is_locked').set(self, value, **kwargs)

        # Return status information
        return self.isLocked()

    security.declareProtected(ToggleLockPermission, 'toggleLock')
    def toggleLock(self):
        """
        toggleLock(self)

        Returns is locked
        """

        if self.isLocked():
            self.lock(False)
        else:
            self.lock(True)

        return self.isLocked()

    security.declareProtected(CMFCorePermissions.View, 'getLockUser')
    def getLockUser(self):
        """Returns the username of the member which has locked the content"""
        return getattr(self, '_locked_by', '')

    security.declareProtected(CMFCorePermissions.View, 'getLockDate')
    def getLockDate(self):
        """Returns the date when the content was locked"""
        return getattr(self, '_locked_date', None)

    security.declareProtected(CMFCorePermissions.View, 'getUnlockUser')
    def getUnlockUser(self):
        """Returns the username of the member which has unlocked the content"""
        return getattr(self, '_unlocked_by', '')

    security.declareProtected(CMFCorePermissions.View, 'getUnlockDate')
    def getUnlockDate(self):
        """Returns the date when the content was unlocked"""
        return getattr(self, '_unlocked_date', None)

    security.declarePrivate('setLockUser')
    def setLockUser(self, value):
        """Set the username of the member which has locked the content
        """
        now = DateTime()
        setattr(self, '_locked_by', value)
        setattr(self, '_unlocked_by', '')
        setattr(self, '_locked_date', now)
        setattr(self, '_unlocked_date',  None)
        if not getattr(self, '_lock_history', None):
            setattr(self, '_lock_history', [])
        self._lock_history.insert(0, ('lock', value, now))

    security.declarePrivate('setUnlockUser')
    def setUnlockUser(self, value):
        """Set the username of the member which has unlocked the content
        """
        now = DateTime()
        setattr(self, '_unlocked_by', value)
        setattr(self, '_locked_by', '')
        setattr(self, '_unlocked_date',  now)
        setattr(self, '_locked_date',  None)
        if not getattr(self, '_lock_history', None):
            setattr(self, '_lock_history', [])
        self._lock_history.insert(0, ('unlock', value, now))

    security.declareProtected(CMFCorePermissions.View, 'canToggleLock')
    def canToggleLock(self):
        """Convenience method that is used to decide whether the edit form is shown.

        If the content is locked, it can only be edited by the a person that
        can unlock it.
        If the content is not locked, the edit permission is sufficient.
        """
        # Return lock support
        if not self.hasLockSupport():
            return False

        mtool = getToolByName(self, 'portal_membership')
        member = mtool.getAuthenticatedMember()

        can_lock = mtool.checkPermission(self.ToggleLockPermission, self) or \
                 mtool.checkPermission(CMFCorePermissions.ManagePortal, self)

        can_unlock = member.getMemberId() == self.getLockUser() or \
                   mtool.checkPermission(self.ForceUnlockPermission, self) or \
                   mtool.checkPermission(CMFCorePermissions.ManagePortal, self)


        if not self.isLocked() and not (mtool.checkPermission(self.ToggleLockPermission, self) or \
                                        mtool.checkPermission(CMFCorePermissions.ManagePortal, self)) or \
           self.isLocked() and not (member.getMemberId() == self.getLockUser() or \
                                    mtool.checkPermission(self.ForceUnlockPermission, self) or \
                                    mtool.checkPermission(CMFCorePermissions.ManagePortal, self)):
            return False
        else:
            return True
