/*
 * Copyright (c) 2007-2010 by The Broad Institute, Inc. and the Massachusetts Institute of Technology.
 * All Rights Reserved.
 *
 * This software is licensed under the terms of the GNU Lesser General Public License (LGPL), Version 2.1 which
 * is available at http://www.opensource.org/licenses/lgpl-2.1.php.
 *
 * THE SOFTWARE IS PROVIDED "AS IS." THE BROAD AND MIT MAKE NO REPRESENTATIONS OR WARRANTIES OF
 * ANY KIND CONCERNING THE SOFTWARE, EXPRESS OR IMPLIED, INCLUDING, WITHOUT LIMITATION, WARRANTIES
 * OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE, NONINFRINGEMENT, OR THE ABSENCE OF LATENT
 * OR OTHER DEFECTS, WHETHER OR NOT DISCOVERABLE.  IN NO EVENT SHALL THE BROAD OR MIT, OR THEIR
 * RESPECTIVE TRUSTEES, DIRECTORS, OFFICERS, EMPLOYEES, AND AFFILIATES BE LIABLE FOR ANY DAMAGES OF
 * ANY KIND, INCLUDING, WITHOUT LIMITATION, INCIDENTAL OR CONSEQUENTIAL DAMAGES, ECONOMIC
 * DAMAGES OR INJURY TO PROPERTY AND LOST PROFITS, REGARDLESS OF WHETHER THE BROAD OR MIT SHALL
 * BE ADVISED, SHALL HAVE OTHER REASON TO KNOW, OR IN FACT SHALL KNOW OF THE POSSIBILITY OF THE
 * FOREGOING.
 */

/*
 * TrackPanel.java
 *
 * Created on Sep 5, 2007, 4:09:39 PM
 *
 * To change this template, choose Tools | Templates
 * and open the template in the editor.
 */

package org.broad.igv.ui.panel;


import org.broad.igv.ui.util.Packable;

import javax.swing.*;
import java.awt.*;


/**
 * @author jrobinso
 */
public class HeaderPanel extends JPanel {

    public HeaderPanel() {
        setDoubleBuffered(true);
    }

    @Override
    public void setBounds(int x, int y, int width, int height) {

        int adjustedX = getAdjustedX();
        int adjustedWidth = getAdjustedWidth();
        super.setBounds(adjustedX, y, adjustedWidth, height);
    }

    @Override
    public void setBounds(Rectangle r) {

        int adjustedX = getAdjustedX();
        int adjustedWidth = getAdjustedWidth();
        super.setBounds(new Rectangle(adjustedX, r.y, adjustedWidth, r.height));
    }

    @Override
    public void setSize(int width, int height) {

        int adjustedWidth = getAdjustedWidth();
        super.setSize(adjustedWidth, height);
    }

    @Override
    public void setSize(Dimension d) {

        int adjustedWidth = getAdjustedWidth();
        super.setSize(new Dimension(adjustedWidth, d.height));
    }

    /**
     * Calculate a new X location for this component
     *
     * @return The new X location.
     */
    private int getAdjustedX() {

        int adjustedX = 0;
        Integer attributePanelEnd = null;
        int width = 0;

        // Pack the attribute panels
        Component[] children = getParent().getComponents();
        for (final Component child : children) {
            if (child instanceof Packable) {

                if (attributePanelEnd == null) {
                    int x = child.getX();
                    adjustedX = x;
                    width = child.getWidth();
                    attributePanelEnd = new Integer(x + width);
                }
                break;
            }
        }

        // New X location
        if (attributePanelEnd != null) {
            adjustedX = attributePanelEnd.intValue();

            if (width > 0)
                adjustedX += IGVPanel.X_SPACING_BETWEEN_COMPONENTS;
        }

        return adjustedX;
    }

    /**
     * Calculate a new width location for this component
     *
     * @return The new width.
     */
    private int getAdjustedWidth() {

        int width = getWidth();

        Container parent = (Container) getParent();
        if (parent != null) {
            int viewportWidth = getViewportWidth();

            int parentVisibleWidth = ((JPanel) parent).getVisibleRect().width;
            if (parentVisibleWidth < viewportWidth)
                viewportWidth = parentVisibleWidth;

            int delta = viewportWidth - (getX() + width);
            width = getWidth() + delta;
        }

        // Width cannot be less than minimum width
        int minimumWidth = getMinimumSize().width;
        if (width < minimumWidth)
            width = minimumWidth;

        return width;
    }

    private int getViewportWidth() {

        int width = 0;
        Container parent = getParent();

        if (parent instanceof JViewport) {
            width = (int) ((JViewport) parent).getViewSize().getWidth();

            int parentVisibleWidth = ((JPanel) parent).getVisibleRect().width;
            if (parentVisibleWidth < width)
                width = parentVisibleWidth;
        } else {
            width = parent.getWidth();
        }

        return width;
    }


    @Override
    protected void paintComponent(Graphics g) {

        super.paintComponent(g);

        // Draw border manually - using setBorder shifts the layout x by 
        // the thickness of the border. This cause the DataPanel and Header
        // panel to not align exactly. In the current case ROI shows they are 
        // off by 1 pixel along the x axis.
        // Manually draw border because setBorder() shifts the layout
        g.setColor(Color.BLACK);
        int x = getX();
        int y = getY();
        int width = getWidth();
        int height = getHeight();
        g.drawLine(0, 0, width - 1, 0); // top border
        g.drawLine(0, 0, 0, height); // left border    
        g.drawLine(width - 1, 0, width - 1, height); // right border
    }
}
