/****************************************************************************
 **
 ** Copyright (C) 2002-2004 Tilo Riemer <riemer@lincvs.org>,
 **                         Frank Hemer <frank@hemer.org>
 **                         Jose Hernandez <joseh@tesco.net>
 **
 **
 **----------------------------------------------------------------------------
 **
 **----------------------------------------------------------------------------
 **
 ** LinCVS is available under two different licenses:
 **
 ** If LinCVS is linked against the GPLed version of Qt 
 ** LinCVS is released under the terms of GPL also.
 **
 ** If LinCVS is linked against a nonGPLed version of Qt 
 ** LinCVS is released under the terms of the 
 ** LinCVS License for non-Unix platforms (LLNU)
 **
 **
 ** LinCVS License for non-Unix platforms (LLNU):
 **
 ** Redistribution and use in binary form, without modification, 
 ** are permitted provided that the following conditions are met:
 **
 ** 1. Redistributions in binary form must reproduce the above copyright
 **    notice, this list of conditions and the following disclaimer in the
 **    documentation and/or other materials provided with the distribution.
 ** 2. It is not permitted to distribute the binary package under a name
 **    different than LinCVS.
 ** 3. The name of the authors may not be used to endorse or promote
 **    products derived from this software without specific prior written
 **    permission.
 ** 4. The source code is the creative property of the authors.
 **    Extensions and development under the terms of the Gnu Public License
 **    are limited to the Unix platform. Any distribution or compilation of 
 **    the source code against libraries licensed other than gpl requires 
 **    the written permission of the authors.
 **
 **
 ** THIS SOFTWARE IS PROVIDED BY THE AUTHOR "AS IS" AND ANY EXPRESS OR 
 ** IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED 
 ** WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE 
 ** ARE DISCLAIMED. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY 
 ** DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL 
 ** DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE 
 ** GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS 
 ** INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, 
 ** WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING 
 ** NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS 
 ** SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 **
 **
 **
 ** LinCVS License for Unix platforms:
 **
 ** This program is free software; you can redistribute it and/or modify 
 ** it under the terms of the GNU General Public License as published by 
 ** the Free Software Foundation; either version 2 of the License, or 
 ** (at your  option) any later version.  This program is distributed in 
 ** the hope that it will be useful, but WITHOUT ANY WARRANTY; without 
 ** even the implied warranty of MERCHANTABILITY or FITNESS FOR A 
 ** PARTICULAR PURPOSE.
 **
 ** See the GNU General Public License for more details.
 **
 ** You should have received a copy of the GNU General Public License
 ** along with this program; if not, write to the Free Software Foundation,
 ** Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 **
 *****************************************************************************/

#include "config.h"


#include <qapplication.h>
#include <qgroupbox.h>
#include <qlistbox.h>
#include <qpushbutton.h>
#include <qcombobox.h>
#include <qradiobutton.h>
#include <qfiledialog.h>
#include <qregexp.h>
#include <qtimer.h>
#include <qmessagebox.h>
#include <qlabel.h>
#include <qwhatsthis.h>

#include "globals.h"
#include "ImportDialogImpl.h"
#include "WrappersIgnoreDialogImpl.h"
#include "getCvsMod.h"
#include "Validators.h"

ImportDialogImpl::ImportDialogImpl( QString* defaultProfile,
      QStringList* ProfileList,
      QStringList* UserList,
      QStringList* ServerList,
      QStringList* RepositoryList,
      QStringList* WorkdirList,
      QStringList* vendor,
      QStringList* release,
      QString* comment,
      const QIconSet &whatsThisIconSet,
      QWidget* parent,
      const char* name,
      bool modal,
      WFlags fl )
  : ImportDialog( modal ? parent : (LookAndFeel::g_b0AsParent ? 0 : parent),
		  name, modal, fl ),
     m_parent(parent),
     m_defBrowseModButtonName(BrowseModulesButton->name()),
     m_defBrowseModButtonText(BrowseModulesButton->text()),
     m_ProfileList(*ProfileList),
     m_UserList(*UserList),
     m_ServerList(*ServerList),
     m_RepositoryList(*RepositoryList),
     m_WorkdirList(*WorkdirList),
     m_Vendor(*vendor),
     m_Release(*release),
     m_Comment(*comment),
     m_running(false)

{
   m_pWhatsThis->setIconSet(whatsThisIconSet);
   m_pWhatsThis->setMaximumWidth(m_pWhatsThis->height());
   
   m_VendorTag->setValidator(new TagValidator(m_VendorTag));
   m_ReleaseTag->setValidator(new TagValidator(m_ReleaseTag));

   m_defaultProfile = defaultProfile;

   //for now this emulates the enum used in ProfilesDialogImpl
   //this should be replaced by a common usage
   m_AccessDict.insert(tr("Password"),&PSERVER);
   m_AccessDict.insert(tr("Remote Shell"),&RSH);
   m_AccessDict.insert(tr("Local/NFS"),&LOCAL);
   m_AccessList.append(tr("Password"));
   m_AccessList.append(tr("Remote Shell"));
   m_AccessList.append(tr("Local/NFS"));

   if (m_Vendor.empty()) {
      m_Vendor.append(tr("INITIAL_IMPORT_VENDOR_TAG"));
   }
   if (m_Release.empty()) {
      m_Release.append(tr("INITIAL_IMPORT_RELEASE_TAG"));
   }

   m_ProfileList.sort();
   m_ProfileList.prepend(tr("No profile selected"));
   m_ProfileBox->insertStringList(m_ProfileList);
   m_UserList.sort();
   m_UserBox->insertStringList(m_UserList);
   m_ServerList.sort();
   m_ServerBox->insertStringList(m_ServerList);
   m_RepositoryList.sort();
   m_RepositoryBox->insertStringList(m_RepositoryList);
   m_AccessBox->insertStringList(m_AccessList);
   insertSortAndSelectFirst(m_ImportDir,m_WorkdirList);
   m_Vendor.prepend(tr("INITIAL_IMPORT_VENDOR_TAG"));
   simplifyStringList(&m_Vendor);
   insertSortAndSelectFirst(m_VendorTag,m_Vendor);
   m_Release.prepend(tr("INITIAL_IMPORT_RELEASE_TAG"));
   simplifyStringList(&m_Release);
   insertSortAndSelectFirst(m_ReleaseTag,m_Release);

   //read environment(CVSROOT) and set default values
   QString connectMethod;
   QString userName;
   QString passwd;
   QString host;
   int port;
   QString rootDir;
   if (analyzeCVSROOT(connectMethod, userName, passwd, host, port, rootDir)) {
      if (!host.isNull()) {
	 host += ":" + QString::number(port);
      }
      m_UserBox->setCurrentText(userName);
      m_ServerBox->setCurrentText(host);
      m_RepositoryBox->setCurrentText(rootDir);
      if (connectMethod.startsWith("local") || connectMethod.startsWith("fork")) {
	 m_AccessBox->setCurrentItem(2);
      } else if (connectMethod.startsWith("pserver")) {
	 m_AccessBox->setCurrentItem(0);
      } else {
	 m_AccessBox->setCurrentItem(1);
      }
   } else {
      m_UserBox->setCurrentText("");
      m_ServerBox->setCurrentText("");
      m_RepositoryBox->setCurrentText("");
      m_AccessBox->setCurrentItem(2);
   }

   m_CommitInfo->setText("");

   m_SshPresetList.append(tr("no ssh"));
   m_SshPresetList.append(tr("use ssh"));
   m_SshPresetList.append(tr("use own ssh-agent"));
   m_SshPresetList.append(tr("use running ssh-agent"));
   m_SshPreset->insertStringList(m_SshPresetList);

   m_ExternalRsh->setChecked(true);
   m_Rsh->setText(ExtApps::g_cvsRsh.path);
   accessActivated(m_AccessBox->currentText());//calls resetSize()

   //build ignorelist from cvs default + homedir/.cvsignore settings
   //per dir is ignored because these are appended to the -I option anyway
   m_ignoreList = QStringList::split(" ",CVSIGNOREFILES);
   QFile cvsignore(QDir::homeDirPath() + "/.cvsignore");
   if(cvsignore.open(IO_ReadOnly)) {
      QTextStream ignoreStream(&cvsignore); 
      while(!ignoreStream.eof()) {
	 QString line = ignoreStream.readLine().simplifyWhiteSpace();
	 if (!line.isEmpty()) {
	    m_ignoreList.append(line);
	 }
      }
      cvsignore.close();
   }
   m_ignoreString = m_ignoreList.join(" ");
   if (m_ignoreString.find("!") > -1) m_ignoreString = m_ignoreString.mid(m_ignoreString.findRev("!", -1) + 1);

   m_binaryList = BINARYFILESPECLIST;

   if (!m_defaultProfile->isEmpty()) {
      setDefaultProfile(*m_defaultProfile);
   }
}

/*  
 *  Destroys the object and frees any allocated resources
 */
ImportDialogImpl::~ImportDialogImpl()
{
   // no need to delete child widgets, Qt does it all for us
}

int ImportDialogImpl::findlistIndex(QStringList *list, const QString item)
{
   int idx;
   idx = 0;
  
   for ( QStringList::Iterator it = list->begin(); it != list->end(); ++it )
      {
	 if ( (*it).stripWhiteSpace()==item.stripWhiteSpace() )
	    {
	       return idx;
	    }
	 idx++;  
      }
   return 0;
}

void ImportDialogImpl::insertSortAndSelectFirst(QComboBox * box,QStringList & list) {
   if (list.isEmpty()) return;
   QString first = list.first();
   list.sort();
   box->insertStringList(list);
   box->setCurrentItem(findlistIndex(&list,first));
}

void ImportDialogImpl::setDefaultProfile(const QString& s) {

   /*! search item in combobox profile */
   m_ProfileBox->setCurrentItem( findlistIndex(&m_ProfileList, s) );

   /*! set the profile */  
   profileActivated(s);
}

void ImportDialogImpl::profileActivated(const QString& s) {
   /*! search index in qlist */
   QValueList<CLincvsProfile>::const_iterator it;
   for (it = cvsProfileContentList.begin(); it != cvsProfileContentList.end(); it++) {
      if ((*it).name().stripWhiteSpace() == s.stripWhiteSpace() )  {
	 /*! search item in combobox user */
	 m_UserBox->setCurrentItem( findlistIndex(&m_UserList, (*it).user()) );

	 /*! search item in combobox server */
	 m_ServerBox->setCurrentItem( findlistIndex(&m_ServerList, (*it).server()) );
	  
	 /*! search item in combobox repository */
	 m_RepositoryBox->setCurrentItem( findlistIndex(&m_RepositoryList, (*it).repository()) );
	  
	 /*! set item in combobox method */
	 m_AccessBox->setCurrentItem( (*it).method());
	 accessActivated( m_AccessBox->currentText());
	  
	 m_SshPreset->setCurrentItem( (*it).sshClientPreset());

	 /*! set default profile */
	 *m_defaultProfile = s.stripWhiteSpace();

	 //and go away
	 return;
      }
   }

   *m_defaultProfile = "";
   m_UserBox->setCurrentText("");
   m_ServerBox->setCurrentText("");
   m_RepositoryBox->setCurrentText("");
   m_AccessBox->setCurrentItem(0);
   m_SshPreset->setCurrentItem(0);
   accessActivated( m_AccessBox->currentText());
}

void ImportDialogImpl::extRshClicked() {
   m_InternalRsh->setChecked(false);
   m_Rsh->setEnabled(true);
}

void ImportDialogImpl::intRshClicked() {
   m_ExternalRsh->setChecked(false);
   m_Rsh->setEnabled(false);
}

void ImportDialogImpl::accessActivated(const QString& s) {
   if ((*m_AccessDict.find(s))==RSH) {
      RshFrame->show();
   } else {
      RshFrame->hide();
   }

   if ((*m_AccessDict.find(s))==LOCAL) {
      m_UserBox->setEnabled(false);
      m_ServerBox->setEnabled(false);
   } else {
      m_UserBox->setEnabled(true);
      m_ServerBox->setEnabled(true);
   }

   resetSize();
}

void ImportDialogImpl::browseDirClicked() {
   QString fn = QFileDialog::getExistingDirectory (m_ImportDir->currentText(), this, NULL,
	 tr("Choose source directory"),
	 true);

   if (!fn.isEmpty ()) {
      m_ImportDir->setCurrentText(fn);
   }
}

void ImportDialogImpl::browseModulesClicked() {

   if (m_running) {
    
      QTimer::singleShot(0,m_parent,SLOT(stopCurAction()));
      BrowseModulesButton->setName(m_defBrowseModButtonName);
      BrowseModulesButton->setText(m_defBrowseModButtonText);

   } else {

      QString connectMethod;
      QString userName;
      QString passwd;
      QString host;
      int port;
      QString rootDir;

      analyzeCVSROOT(connectMethod, userName, passwd, host, port, rootDir);
      if ( !m_RepositoryBox->currentText().isEmpty()) {
	 rootDir = m_RepositoryBox->currentText();
      }

      if ( (*m_AccessDict.find(m_AccessBox->currentText()))==LOCAL) {
	 getCvsModule *mBrowser = new getCvsModule(((QString)rootDir)+"/", this);
	 connect (mBrowser, SIGNAL (cvsModuleSelected( const QString & )), 
	       SLOT (insertModule(const QString &)));
    
	 mBrowser->resize( 500, 400 );
	 mBrowser->setCaption(tr("CVS Module Browser"));
	 mBrowser->exec();
      } else {

	 QString connMethod;
	 if( (*m_AccessDict.find(m_AccessBox->currentText()))==PSERVER) {
	    connMethod = "pserver";
	 } else if((*m_AccessDict.find(m_AccessBox->currentText()))==RSH) {
	    if(m_ExternalRsh->isChecked()) {
	       connMethod = "ext";
	    } else {
	       connMethod = "server";
	    }
	 }

	 QString dir = "";
	 QString server = m_ServerBox->currentText();
	 if (server.find(':') == -1) server += ":";
	 m_TmpBrowseModulesRoot =
	    ":"+connMethod+
	    ":"+m_UserBox->currentText()+
	    "@"+server+
	    m_RepositoryBox->currentText();

	 int sshAccess = getSshPreset();

	 //check ssh access settings
	 bUseSsh = false;
	 bUseSshAgent = false;
	 bUseSshAgentVars = false;
	 switch( sshAccess) {
	    case USESSH: {
	       bUseSsh = true;
	       break;
	    }
	    case USESSHAGENT: {
	       bUseSshAgent = true;
	       break;
	    }
	    case USESSHAGENTVARS: {
	       bUseSshAgentVars = true;
	       break;
	    }
	 }

	 QString strCmd = "checkout -p -n CVSROOT/modules";
	 QString topModule = QString::null;
	 BrowseModulesButton->setName("stop action");
	 BrowseModulesButton->setText(tr("Stop"));
	 callInteractive( topModule, dir, m_TmpBrowseModulesRoot,
	       strCmd, CVS_MBROWSEMODULES_CMD,
	       ExtApps::g_cvsRsh.path,  //additional options of cvsRsh not supported yet
	       false);
      }
   }
}

void ImportDialogImpl::editIgnoreClicked() {
   WrappersIgnoreDialogImpl* dlg = new WrappersIgnoreDialogImpl(&m_ignoreList,
	 *(m_pWhatsThis->iconSet()),
	 tr("Files to ignore"), this,
	 0, true, LookAndFeel::g_modalF);
   dlg->exec();
   m_ignoreString = m_ignoreList.join(" ");
   if (m_ignoreString.find("!") > -1) m_ignoreString = m_ignoreString.mid(m_ignoreString.findRev("!", -1) + 1);
   delete dlg;
}

void ImportDialogImpl::editBinaryClicked() {
   WrappersIgnoreDialogImpl* dlg = new WrappersIgnoreDialogImpl(&m_binaryList,
	 *(m_pWhatsThis->iconSet()),
	 tr("Import files as binary"), this,
	 0, true, LookAndFeel::g_modalF);
   dlg->exec();
   delete dlg;
}

void ImportDialogImpl::previewClicked() {

   m_bScanInterrupted = false;

   //previewDlg = new ImportPreview();
   QApplication::setOverrideCursor(Qt::waitCursor);

   progressDlg = new ProgressDialog(this,
	 NULL,true, Qt::WStyle_Customize | Qt::WStyle_NoBorder);
   connect(progressDlg->CancelButton,SIGNAL(clicked()),this,SLOT(cancelPreviewScan()));
   progressDlg->m_InfoMessage->setText("Scanning dir:");
   progressDlg->show();
   progressDlg->update();

   previewDlg = new ImportPreview(this, NULL, true,
	 LookAndFeel::g_modalF);
   previewDlg->m_pWhatsThis->setIconSet(*(m_pWhatsThis->iconSet()));
   previewDlg->m_pWhatsThis->setMaximumWidth(previewDlg->m_pWhatsThis->height());

   addToPreview(m_ImportDir->currentText().replace(QRegExp ("/+$"),""));

   progressDlg->hide();
   delete progressDlg;
   progressDlg = 0;

   QApplication::restoreOverrideCursor();
   qApp->processEvents();
  
   if (!m_bScanInterrupted) {
      previewDlg->exec();
   }
  
   delete previewDlg;
   previewDlg = 0;
}

void ImportDialogImpl::addToPreview( QString& path) {

   progressDlg->m_InfoText->setText(path);
   qApp->processEvents();
   if (m_bScanInterrupted) {
      return;
   }

   QDir D(path);
   D.setFilter( QDir::Files | QDir::Dirs | QDir::Hidden);
   QStringList allDirEntrieslist = D.entryList();
   allDirEntrieslist.remove( "." );
   allDirEntrieslist.remove( ".." );
   allDirEntrieslist.remove( "CVS" );

   //collect ignore info
   QString filesToIgnore = m_ignoreString;
   QFile cvsignore(path + "/.cvsignore");
   if(cvsignore.open(IO_ReadOnly)) {
      QTextStream ignoreStream(&cvsignore); 
      while(!ignoreStream.eof()) {
	 filesToIgnore += " " + ignoreStream.readLine().simplifyWhiteSpace();
      }
      cvsignore.close();
   }
   filesToIgnore.replace(QRegExp("\\."),"\\.");
   filesToIgnore.replace(QRegExp("\\?"),".");
   filesToIgnore.replace(QRegExp("\\*"),".*");
   filesToIgnore.replace(QRegExp("\\$"), "\\$");
   if (filesToIgnore.find("!") > -1) filesToIgnore = filesToIgnore.mid(filesToIgnore.findRev("!", -1) + 1);

   filesToIgnore.replace(QRegExp(" "), "|");
   filesToIgnore = "^(" + filesToIgnore + ")$";

   //collect binary info
   QString binaryFiles = getBinaryWildcards().join(" ");
  
   binaryFiles.replace(QRegExp("\\."),"\\.");
   binaryFiles.replace(QRegExp("\\?"),".");
   binaryFiles.replace(QRegExp("\\*"),".*");
   binaryFiles.replace(QRegExp("\\$"), "\\$");
  
   binaryFiles.replace(QRegExp(" "), "|");
   binaryFiles = "^(" + binaryFiles + ")$";

   //now insert entries to appropriate view
   QStringList::Iterator it;
   for ( it = allDirEntrieslist.begin(); it != allDirEntrieslist.end(); it++ ) {

      QString file = path + "/" + (*it);
      QFileInfo fileInfo( file);

      if(fileInfo.isDir() && fileInfo.isReadable() && fileInfo.isExecutable() && ((*it).find(QRegExp(filesToIgnore)) == -1) ) {// dir

	 addToPreview(file.replace(QRegExp ("/+$"),""));

      } else if(fileInfo.isFile() && fileInfo.isReadable()) {// file

	 if ((*it).find(QRegExp(filesToIgnore)) != -1) {// ignore

	    previewDlg->m_IgnoreList->insertItem(new QListViewItem(previewDlg->m_IgnoreList,file));

	 } else {// don't ignore

	    if ((*it).find(QRegExp(binaryFiles)) != -1) {// binary
	       previewDlg->m_ImportList->insertItem(new QListViewItem(previewDlg->m_ImportList,file,"-kb"));
	    } else {// not binary
	       previewDlg->m_ImportList->insertItem(new QListViewItem(previewDlg->m_ImportList,file));
	    }
	 }
      } else {

	 previewDlg->m_RestList->insertItem(new QListViewItem(previewDlg->m_RestList,file));
      
      }
   }

}

void ImportDialogImpl::resetSize() {
   QTimer::singleShot(0,this,SLOT(reduceHeight()));
}

void ImportDialogImpl::reduceHeight() {
   adjustSize();
}

QSize ImportDialogImpl::sizeHint () const {
   QSize s = ImportDialog::sizeHint();
   s.setWidth(ImportDialog::width());
   return s;
}

void ImportDialogImpl::accept() {

   if (m_BrowseModules->currentText().isEmpty()) {
      QMessageBox::warning(this, tr("Warning"), tr("You have to enter a module name"), QMessageBox::Ok, QMessageBox::NoButton);
      return;
   } else if (m_CommitInfo->text().isEmpty()) {
      QMessageBox::warning(this, tr("Warning"), tr("You have to enter an import message"), QMessageBox::Ok, QMessageBox::NoButton);
      return;
   } else if (m_VendorTag->currentText().isEmpty()) {
      QMessageBox::warning(this, tr("Warning"), tr("You have to enter a vendor tag"), QMessageBox::Ok, QMessageBox::NoButton);
      return;
   } else if (m_ReleaseTag->currentText().isEmpty()) {
      QMessageBox::warning(this, tr("Warning"), tr("You have to enter a release tag"), QMessageBox::Ok, QMessageBox::NoButton);
      return;
   }
   ImportDialog::accept();
}

QStringList ImportDialogImpl::getIgnoreWildcards() {
   return QStringList::split(" ",m_ignoreString);
}

QStringList ImportDialogImpl::getBinaryWildcards() {
   return m_binaryList;
}

QString ImportDialogImpl::user() {
   return m_UserBox->currentText();
}

QString ImportDialogImpl::server() {
   return m_ServerBox->currentText();
}

QString ImportDialogImpl::tagV() {
   return m_VendorTag->currentText();
}

QString ImportDialogImpl::tagR() {
   return m_ReleaseTag->currentText();
}

QString ImportDialogImpl::comment() {
   return m_CommitInfo->text();
}

QString ImportDialogImpl::localDir() {
   return m_ImportDir->currentText();
}

QString ImportDialogImpl::module() {
   return m_BrowseModules->currentText();
}

QString ImportDialogImpl::repository() {
   return m_RepositoryBox->currentText().replace (QRegExp ("/+$"), "");
}

QString ImportDialogImpl::getCvsRsh() {
   return m_Rsh->displayText();
}

int ImportDialogImpl::getSshPreset() {
   switch (m_SshPreset->currentItem()) {
      case 0: return NOSSH;
      case 1: return USESSH;
      case 2: return USESSHAGENT;
      case 3: return USESSHAGENTVARS;
      default: return -1;
   }
}

int ImportDialogImpl::mode() {
   return (*m_AccessDict.find(m_AccessBox->currentText()));
}

int ImportDialogImpl::rshMode() {
   if(m_ExternalRsh->isChecked()) {
      return RSH_EXT;
   } else {
      return RSH_SERVER;
   }
}

void ImportDialogImpl::getAndSetModule() {
   m_BrowseModules->setCurrentText(m_ModuleBrowser->getModule());
}

void ImportDialogImpl::insertModule(const QString& module) {
   m_BrowseModules->setCurrentText(module);
}

void ImportDialogImpl::cancelPreviewScan() {
   m_bScanInterrupted = true;
}

void ImportDialogImpl::setEnabled(bool state) {
   m_ProfileBox->setEnabled(state);
   m_UserBox->setEnabled(state);
   m_ServerBox->setEnabled(state);
   m_RepositoryBox->setEnabled(state);
   m_AccessBox->setEnabled(state);
   RshFrame->setEnabled(state);
   EditSpecFrame->setEnabled(state);
   TagFrame->setEnabled(state);
   GroupBox1->setEnabled(state);
   m_pWhatsThis->setEnabled(state);
   PreviewButton->setEnabled(state);
   ImportButton->setEnabled(state);
   CancelButton->setEnabled(state);
   m_ImportDir->setEnabled(state);
   m_BrowseModules->setEnabled(state);
   BrowseDirButton->setEnabled(state);
}

void ImportDialogImpl::cvsCallStarted() {
   m_running = true;
   setEnabled(FALSE);
   QApplication::setOverrideCursor(Qt::waitCursor);
}

void ImportDialogImpl::cvsCallFinished() {
   QApplication::restoreOverrideCursor();
   setEnabled(TRUE);
   m_running = false;
}

void ImportDialogImpl::afterCall( int cmd, CvsBuffer* output, bool failed) {
   cvsCallFinished();

   switch (cmd) {
      case CVS_MBROWSEMODULES_CMD: {
	 if (failed) {
	    BrowseModulesButton->setName(m_defBrowseModButtonName);
	    BrowseModulesButton->setText(m_defBrowseModButtonText);
	    return;
	 }
	 bool fileOk = false;
	 QString line;
	 unsigned int len = (*output).numLines();
	 unsigned int skipLines = outputLineOffset + 6;
	 for ( unsigned int i = skipLines; i<len;i++) {//only two lines if no CVSROOT/modules file
	    line = (*output).textLine(i);
	    if ( (line.at(0)!='#') && ( !(line.isEmpty()) ) ) {
	       fileOk = true;
	       break;
	    }
	 }
	 if (!fileOk) {
	    int sshAccess = getSshPreset();
	    //check ssh access settings
	    bUseSsh = false;
	    bUseSshAgent = false;
	    bUseSshAgentVars = false;
	    switch( sshAccess) {
	       case USESSH: {
		  bUseSsh = true;
		  break;
	       }
	       case USESSHAGENT: {
		  bUseSshAgent = true;
		  break;
	       }
	       case USESSHAGENTVARS: {
		  bUseSshAgentVars = true;
		  break;
	       }
	    }
	    QString dir = "";
	    QString files = "history -xAMR -a -l";
	    QString topModule = QString::null;
	    callInteractive( topModule, dir, m_TmpBrowseModulesRoot,
		  files, CVS_HBROWSEMODULES_CMD,
		  ExtApps::g_cvsRsh.path,  //additional options of cvsRsh not supported yet
		  false);
	    return;
	 } else {
	    BrowseModulesButton->setName(m_defBrowseModButtonName);
	    BrowseModulesButton->setText(m_defBrowseModButtonText);
	    m_ModuleBrowser = new ModuleBrowserImpl(*(m_pWhatsThis->iconSet()),
		  NULL,
		  m_TmpBrowseModulesRoot,
		  m_parent,
		  this,
		  "Browse Modules",
		  true,
		  LookAndFeel::g_modalF | WDestructiveClose);
	    connect(m_ModuleBrowser,SIGNAL(OkClicked()),this,SLOT(getAndSetModule()));
	    m_ModuleBrowser->parseModules (output);
	    m_ModuleBrowser->show();
	 }
	 break;
      }
      case CVS_HBROWSEMODULES_CMD: {
	 if (failed) {
	    BrowseModulesButton->setName(m_defBrowseModButtonName);
	    BrowseModulesButton->setText(m_defBrowseModButtonText);
	    return;
	 }
	 BrowseModulesButton->setName(m_defBrowseModButtonName);
	 BrowseModulesButton->setText(m_defBrowseModButtonText);
	 m_ModuleBrowser = new ModuleBrowserImpl(*(m_pWhatsThis->iconSet()),
	       NULL,
	       m_TmpBrowseModulesRoot,
	       m_parent,
	       this,
	       "Browse Modules",
	       true,
	       LookAndFeel::g_modalF | WDestructiveClose);
	 connect(m_ModuleBrowser,SIGNAL(OkClicked()),this,SLOT(getAndSetModule()));
	 m_ModuleBrowser->parseHistory (output);
	 m_ModuleBrowser->show();
	 break;
      }
   }
}

void ImportDialogImpl::enterWhatsThisMode()
{
   QWhatsThis::enterWhatsThisMode();
}
