/*
 *   meioqry.c -- MEIO Manager query functions
 *
 *  Written By: Mike Sullivan IBM Corporation
 *
 *  Copyright (C) 1999 IBM Corporation
 *
 * This program is free software; you can redistribute it and/or modify      
 * it under the terms of the GNU General Public License as published by      
 * the Free Software Foundation; either version 2 of the License, or         
 * (at your option) any later version.                                       
 *                                                                           
 * This program is distributed in the hope that it will be useful,           
 * but WITHOUT ANY WARRANTY; without even the implied warranty of            
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the             
 * GNU General Public License for more details.                              
 *                                                                           
 * NO WARRANTY                                                               
 * THE PROGRAM IS PROVIDED ON AN "AS IS" BASIS, WITHOUT WARRANTIES OR        
 * CONDITIONS OF ANY KIND, EITHER EXPRESS OR IMPLIED INCLUDING, WITHOUT      
 * LIMITATION, ANY WARRANTIES OR CONDITIONS OF TITLE, NON-INFRINGEMENT,      
 * MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE. Each Recipient is    
 * solely responsible for determining the appropriateness of using and       
 * distributing the Program and assumes all risks associated with its        
 * exercise of rights under this Agreement, including but not limited to     
 * the risks and costs of program errors, damage to or loss of data,         
 * programs or equipment, and unavailability or interruption of operations.  
 *                                                                           
 * DISCLAIMER OF LIABILITY                                                   
 * NEITHER RECIPIENT NOR ANY CONTRIBUTORS SHALL HAVE ANY LIABILITY FOR ANY   
 * DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL        
 * DAMAGES (INCLUDING WITHOUT LIMITATION LOST PROFITS), HOWEVER CAUSED AND   
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR     
 * TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE    
 * USE OR DISTRIBUTION OF THE PROGRAM OR THE EXERCISE OF ANY RIGHTS GRANTED  
 * HEREUNDER, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGES             
 *                                                                           
 * You should have received a copy of the GNU General Public License         
 * along with this program; if not, write to the Free Software               
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA 
 *                                                                           
 * 
 *  10/23/2000 - Alpha Release 0.1.0
 *            First release to the public
 *
 */
#include <port_types.h> 
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <pthread.h>

#include <meio_api.h>      
#include "meiomacs.h"      
#include <meiogtwy.h>

extern unsigned short MEIO_DriverVersion;

enum {                     
  AOPORTS
  ,AUPORTS
  ,DUPORTS
  ,DOPORTS
  ,ASWITCH
  ,AINTERNAL
  ,BIOSTYPE
};

#define LASTTYPE BIOSTYPE

static struct handlestat
{
  PRSET  Start;   
  USHORT Count;   
} HandleStats[ BIOSTYPE+1 ] = { 0 };

/*  GetHandleInfo() - local function to retrieve handle statistics */
static USHORT GetHandleInfo(int typeidx, USHORT type)
{
  int        i;
  PRSET      Prs;
  
  MW_SYSLOG_3(TRACE_MEIO_CORE,"meioqry::GetHandleInfo entry typeidx %x type %x\n",typeidx,type);
  
  if (0==HandleStats[typeidx].Start) {
    Prs = (PRSET)&AllHandles[0];
    
    for (i=0;  *Prs; i++, Prs++) {
      if ((*Prs)->usType == type ) {
	HandleStats[typeidx].Start = Prs;
	HandleStats[typeidx].Count = 1;
	
	for (Prs++; *Prs; i++, Prs++)
	  if ((*Prs)->usType == type )
	    HandleStats[typeidx].Count++;
	  else break;      
	
	break;   
      } 
    } 
   } 
  
  MW_SYSLOG_2(TRACE_MEIO_CORE,"meioqry::GetHandleInfo exit HandleStats[typeidx].Count %x\n",HandleStats[typeidx].Count);
  return (HandleStats[typeidx].Count);
}

/*  CopyHandleInfo() - copy port handles to user space */
static ULONG CopyHandleInfo(int     typeidx,
			    USHORT  type,
			    PULONG  fpBufferSize,
			    PUSHORT pU) 
{
  ULONG   ulRC = MEIO_NOERROR;
  USHORT  count;
  PRSET   Prs;
  
  MW_SYSLOG_3(TRACE_MEIO_CORE,"meioqry::CopyHandleInfo entry typeidx %x type %x\n",typeidx,type);
  count = GetHandleInfo(typeidx, type);   // get the count of ports
  
  if (((count+1)*sizeof(USHORT)) > *fpBufferSize) {
    *fpBufferSize = (count+1)*sizeof(USHORT);
    ulRC = MEIO_INSUFFICIENT_BUFFER;
  } else {
    Prs = HandleStats[typeidx].Start;
    
    while (count--) {
      *pU++ = (*Prs++)->usHandle;
    }
    
    *pU = 0;  
  }

   MW_SYSLOG_2(TRACE_MEIO_CORE,"meioqry::CopyHandleInfo exit ulRC %lx\n",ulRC);
   return(ulRC);
}


/*   CopyConnectionData() - scan resourceset array looking for types of
                          connections.  When found, return port data iff
                          usage matches UsedFlag.  Return all if UsedFlag
                          is 0.
*/
static ULONG CopyConnectionData(USHORT type, USHORT UsedFlag,
				PULONG fpBufferSize, PUSHORT pU)             
{
  ULONG        ulRC = MEIO_NOERROR;
  unsigned int count = 0;
  int          i;
  PRS          Prs;
  PRSET        ppR;
  
  MW_SYSLOG_3(TRACE_MEIO_CORE,"meioqry::CopyConnectionData entry type %x UsedFlag %x\n",type,UsedFlag);
  for (i=0; AllResources[i]; i++) {
    Prs = AllResources[i];
    if (((Prs->Resource.usHandle & TYPEMASK) == type) &&
	( UsedFlag == 0 || Prs->Resource.usOwnerCount > 0 ) ) {
      count++;
    }
   } 

  count = (count * 2 * sizeof(USHORT)) + sizeof(USHORT);
  
  if (count > *fpBufferSize) {
    *fpBufferSize = count;
    ulRC = MEIO_INSUFFICIENT_BUFFER;
  } else {
    for (i=0; AllResources[i]; i++) {
      Prs = AllResources[i];
      
      if (((Prs->Resource.usHandle & TYPEMASK) == type) &&
	  ( UsedFlag == 0 || Prs->Resource.usOwnerCount > 0 ) ) {
	ppR = Prs->lpResourceSet;
	*pU++ = ppR[0]->usHandle; 
	*pU++ = ppR[1]->usHandle; 
      }
    }
    
    *pU = 0; 
  }  

  MW_SYSLOG_2(TRACE_MEIO_CORE,"meioqry::CopyConnectionData exit ulRC %lx\n",ulRC);
  return(ulRC);
}

/*   MEIO_Query_Abilities() - function to return info about MEIO Capabilities  
                            and info for this adapter.
*/
ULONG     FAR MEIO_Query_Abilities(PMEIO_DSP      pMeioDsp,
                                   MEIO_QUERYTYPE Request,
                                   PULONG         fpBufferSize,
                                   PVOID          fpBuffer)
{
  ULONG   ulRC = MEIO_NOERROR;
  PUSHORT pU;                                                       
  
  (void)pMeioDsp; 

  MW_SYSLOG_3(TRACE_MEIO_CORE,"meioqry::MEIO_Query_Abilities entry pMeioDsp %p Request %x\n",pMeioDsp,Request);
  pU = (PUSHORT)fpBuffer;     

  switch (Request)  {
  case QAPIV:             
    if (sizeof(USHORT) > *fpBufferSize) {
      *fpBufferSize = sizeof(USHORT);
      ulRC = MEIO_INSUFFICIENT_BUFFER;
    } else {
      *pU = MEIO_DriverVersion;
    } 
    break;
    
  case QAOWNERS:
    ulRC = CopyHandleInfo(AOPORTS,ANALOG_OWNER_PORT, fpBufferSize,pU);
    break;
    
  case QAUSERS:    
    ulRC = CopyHandleInfo(AUPORTS,ANALOG_USER_PORT, fpBufferSize,pU); 
    break;
    
  case QDOWNERS:
    ulRC = CopyHandleInfo(DOPORTS,DIGITAL_OWNER_PORT, fpBufferSize,pU);
    break;
    
  case QDUSERS:   
    ulRC = CopyHandleInfo(DUPORTS,DIGITAL_USER_PORT, fpBufferSize,pU);
    break;
    
  case QACONNECTIONS:   
    ulRC = CopyConnectionData(ARESOURCESET, (USHORT)1, fpBufferSize,pU);
    break;
    
  case QDCONNECTIONS:   
    ulRC = CopyConnectionData(DRESOURCESET, (USHORT)1, fpBufferSize,pU);    
    break;
    
  case QALLAC:
    ulRC = CopyConnectionData(ARESOURCESET, (USHORT)0, fpBufferSize,pU); 
    break;

  case QALLDC:  
    ulRC = CopyConnectionData(DRESOURCESET, (USHORT)0, fpBufferSize,pU); 
    break;
    
  case QADAPTER:
    if (sizeof(ULONG) > *fpBufferSize) {
      *fpBufferSize = sizeof(ULONG);
      ulRC = MEIO_INSUFFICIENT_BUFFER;
    } else {
      *(PULONG)pU = GetMeioAdapterCaps(0);   
    } 
    break;

  default:
    break;
  }
  MW_SYSLOG_2(TRACE_MEIO_CORE,"meioqry::MEIO_Query_Abilities exit ulRC %lx\n",ulRC);
  return(ulRC);
}
