/*
 * Copyright (c) 1999 Apple Computer, Inc. All rights reserved.
 *
 * @APPLE_LICENSE_HEADER_START@
 * 
 * Portions Copyright (c) 1999 Apple Computer, Inc.  All Rights
 * Reserved.  This file contains Original Code and/or Modifications of
 * Original Code as defined in and that are subject to the Apple Public
 * Source License Version 1.1 (the "License").  You may not use this file
 * except in compliance with the License.  Please obtain a copy of the
 * License at http://www.apple.com/publicsource and read it before using
 * this file.
 * 
 * The Original Code and all software distributed under the License are
 * distributed on an "AS IS" basis, WITHOUT WARRANTY OF ANY KIND, EITHER
 * EXPRESS OR IMPLIED, AND APPLE HEREBY DISCLAIMS ALL SUCH WARRANTIES,
 * INCLUDING WITHOUT LIMITATION, ANY WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE OR NON- INFRINGEMENT.  Please see the
 * License for the specific language governing rights and limitations
 * under the License.
 * 
 * @APPLE_LICENSE_HEADER_END@
 */
/*
	File:		RTSPRequestInterface.cp

	Contains:	Implementation of class defined in RTSPRequestInterface.h
	

	$Log: RTSPRequestInterface.cpp,v $
	Revision 1.2  1999/02/19 23:08:32  ds
	Created
	

*/


//INCLUDES:
#ifndef __MW_
#include <sys/types.h>
#include <sys/uio.h>
#endif

#include "RTSPRequestInterface.h"

#include "RTSPSessionInterface.h"
#include "RTSPRequestStream.h"
#include "RTSPServerInterface.h"
#include "StringParser.h"
#include "OS.h"

char 		RTSPRequestInterface::sPremadeHeader[kStaticHeaderSizeInBytes];
StrPtrLen	RTSPRequestInterface::sPremadeHeaderPtr(sPremadeHeader, kStaticHeaderSizeInBytes);
char		RTSPRequestInterface::sServerHeader[kStaticHeaderSizeInBytes];
StrPtrLen	RTSPRequestInterface::sServerHeaderPtr(sServerHeader, kStaticHeaderSizeInBytes);
StrPtrLen	RTSPRequestInterface::sColonSpace(": ", 2);


void	RTSPResponseFormatter::BufferIsFull(char* inBuffer, UInt32 inBufferLen)
{
	//allocate a buffer twice as big as the old one, and copy over the contents
	UInt32 theNewBufferSize = this->GetTotalBufferSize() * 2;
	char* theNewBuffer = new ('RESB') char[theNewBufferSize];
	::memcpy(theNewBuffer, inBuffer, inBufferLen);

	//if the old buffer was dynamically allocated also, we'd better delete it.
	if (inBuffer != &fOutputBuf[0])
		delete [] inBuffer;
	
	this->SetBuffer(theNewBuffer, theNewBufferSize, inBufferLen);
}

void RTSPResponseFormatter::SetBuffer(char* inBuffer, UInt32 inBufferLen, UInt32 currentOffset)
{
	fStartPut = inBuffer;
	fCurrentPut = inBuffer + currentOffset;
	fEndPut = inBuffer + inBufferLen;
}


void  RTSPRequestInterface::Initialize()
{
	//Write out a premade server header
	StringFormatter serverFormatter(sServerHeaderPtr.Ptr, kStaticHeaderSizeInBytes);
	serverFormatter.Put(RTSPProtocol::GetHeaderString(RTSPProtocol::kServerHeader));
	serverFormatter.Put(sColonSpace);
	serverFormatter.Put(RTSPServerInterface::GetRTSPPrefs()->GetServerName());
	serverFormatter.PutChar('/');
	serverFormatter.Put(RTSPServerInterface::GetRTSPPrefs()->GetServerVersion());
	sServerHeaderPtr.Len = serverFormatter.GetCurrentOffset();
	Assert(sServerHeaderPtr.Len < kStaticHeaderSizeInBytes);
	
	//make a partially complete header
	StringFormatter headerFormatter(sPremadeHeaderPtr.Ptr, kStaticHeaderSizeInBytes);
	PutStatusLine(&headerFormatter, RTSPProtocol::kSuccessOK, RTSPProtocol::k10Version);
	headerFormatter.Put(sServerHeaderPtr);
	headerFormatter.PutEOL();
	headerFormatter.Put(RTSPProtocol::GetHeaderString(RTSPProtocol::kCSeqHeader));
	headerFormatter.Put(sColonSpace);
	sPremadeHeaderPtr.Len = headerFormatter.GetCurrentOffset();
	Assert(sPremadeHeaderPtr.Len < kStaticHeaderSizeInBytes);
}

//CONSTRUCTOR / DESTRUCTOR: very simple stuff
RTSPRequestInterface::RTSPRequestInterface(RTSPRequestStream *stream, RTSPSessionInterface *session)
:	//fResponseKeepAlive(true), //parameter need not be set
	fStatus(RTSPProtocol::kSuccessOK), //parameter need not be set
	//fMethod(0), //parameter need not be set
	fVersion(RTSPProtocol::k10Version),
	fStartTime(-1),
	fStopTime(-1),
	fClientPortA(0),
	fClientPortB(0),
	fRequestKeepAlive(true),
	fSession(session),
	fStream(stream),
	fStandardHeadersWritten(false),
	fBytesSent(0)
{
	//Setup QTSS parameters that can be setup now
	fQTSSConnParams[qtssFullRequestParam] = *stream->GetRequestBuffer();
	
#ifdef DUMPREQUESTS
	{
	StrPtrLen* requestBuff = stream->GetRequestBuffer();
	char saveChar = requestBuff->Ptr[requestBuff->Len];
	requestBuff->Ptr[requestBuff->Len] = '\0';
	printf ("%s\n\n", requestBuff->Ptr);
	requestBuff->Ptr[requestBuff->Len] = saveChar;
	}
#endif
}

void RTSPRequestInterface::AppendHeader(RTSPProtocol::RTSPHeader inHeader, StrPtrLen* inValue)
{
	if (!fStandardHeadersWritten)
		this->WriteStandardHeaders();
		
	fOutputFormatter.Put(RTSPProtocol::GetHeaderString(inHeader));
	fOutputFormatter.Put(sColonSpace);
	fOutputFormatter.Put(*inValue);
	fOutputFormatter.PutEOL();
}

void RTSPRequestInterface::PutStatusLine(StringFormatter* putStream, RTSPProtocol::RTSPStatusCode status,
										RTSPProtocol::RTSPVersion version)
{
	putStream->Put(RTSPProtocol::GetVersionString(version));
	putStream->PutSpace();
	putStream->Put(RTSPProtocol::GetStatusCodeAsString(status));
	putStream->PutSpace();
	putStream->Put(RTSPProtocol::GetStatusCodeString(status));
	putStream->PutEOL();	
}


void RTSPRequestInterface::AppendTransportHeader(StrPtrLen* serverIPAddr,
													StrPtrLen* serverPortA,
													StrPtrLen* serverPortB)
{
	static StrPtrLen	sServerPortString(";server_port=");
	static StrPtrLen	sClientPortString(";client_port=");
	static StrPtrLen	sSourceString(";source=");
	static StrPtrLen	sPremadeTransportHeader("rtp/avp");

	if (!fStandardHeadersWritten)
		this->WriteStandardHeaders();

	fOutputFormatter.Put(RTSPProtocol::GetHeaderString(RTSPProtocol::kTransportHeader));
	fOutputFormatter.Put(sColonSpace);
	fOutputFormatter.Put(sPremadeTransportHeader);
	
	//The source IP addr is optional, only append it if it is provided
	if (serverIPAddr != NULL)
	{
		fOutputFormatter.Put(sSourceString);
		fOutputFormatter.Put(*serverIPAddr);
	}
	
	//Always append the port pairs. This info MUST be provided
	fOutputFormatter.Put(sServerPortString);
	fOutputFormatter.Put(*serverPortA);
	fOutputFormatter.PutChar('-');
	fOutputFormatter.Put(*serverPortB);
	fOutputFormatter.Put(sClientPortString);
	fOutputFormatter.Put(fQTSSConnParams[qtssClientPortA]);
	fOutputFormatter.PutChar('-');
	fOutputFormatter.Put(fQTSSConnParams[qtssClientPortB]);
	fOutputFormatter.PutEOL();
}

void RTSPRequestInterface::AppendSessionHeader(StrPtrLen* inSessionID, StrPtrLen* inTimeout)
{
	static StrPtrLen	sTimeoutString(";timeout=");
	
	if (!fStandardHeadersWritten)
		this->WriteStandardHeaders();

	fOutputFormatter.Put(RTSPProtocol::GetHeaderString(RTSPProtocol::kSessionHeader));
	fOutputFormatter.Put(sColonSpace);
	fOutputFormatter.Put(*inSessionID);
	fOutputFormatter.Put(sTimeoutString);
	fOutputFormatter.Put(*inTimeout);
	fOutputFormatter.PutEOL();
}

void RTSPRequestInterface::AppendContentBaseHeader(StrPtrLen* theURL)
{
	if (!fStandardHeadersWritten)
		this->WriteStandardHeaders();

	fOutputFormatter.Put(RTSPProtocol::GetHeaderString(RTSPProtocol::kContentBaseHeader));
	fOutputFormatter.Put(sColonSpace);
	fOutputFormatter.Put(*theURL);
	fOutputFormatter.PutChar('/');
	fOutputFormatter.PutEOL();
}

void RTSPRequestInterface::AppendRTPInfoHeader(RTSPProtocol::RTSPHeader inHeader,
												StrPtrLen* url, StrPtrLen* seqNumber,
												StrPtrLen* ssrc, StrPtrLen* rtpTime)
{
	static StrPtrLen sURL("url=", 4);
	static StrPtrLen sSeq(";seq=", 5);
	static StrPtrLen sSsrc(";ssrc=", 6);
	static StrPtrLen sRTPTime(";rtptime=", 9);

	if (!fStandardHeadersWritten)
		this->WriteStandardHeaders();

	fOutputFormatter.Put(RTSPProtocol::GetHeaderString(inHeader));
	if (inHeader != RTSPProtocol::kSameAsLastHeader)
		fOutputFormatter.Put(sColonSpace);
		
	//Only append the various bits of RTP information if they actually have been
	//providied
	if ((url != NULL) && (url->Len))
	{
		fOutputFormatter.Put(sURL);
		fOutputFormatter.Put(*url);
	}
	if ((seqNumber != NULL) && (seqNumber->Len > 0))
	{
		fOutputFormatter.Put(sSeq);
		fOutputFormatter.Put(*seqNumber);
	}
	if ((ssrc != NULL) && (ssrc->Len > 0))
	{
		fOutputFormatter.Put(sSsrc);
		fOutputFormatter.Put(*ssrc);
	}
	if ((rtpTime != NULL) && (rtpTime->Len > 0))
	{
		fOutputFormatter.Put(sRTPTime);
		fOutputFormatter.Put(*rtpTime);
	}
	fOutputFormatter.PutEOL();
}

void RTSPRequestInterface::AppendContentLength(SInt32 inLength)
{
	fOutputFormatter.Put(RTSPProtocol::GetHeaderString(RTSPProtocol::kContentLengthHeader));
	fOutputFormatter.Put(sColonSpace);
	fOutputFormatter.Put(inLength);
	fOutputFormatter.PutEOL();	
}

void RTSPRequestInterface::WriteStandardHeaders()
{
	static StrPtrLen	sCloseString("Close", 5);

	Assert(sPremadeHeader != NULL);
	fStandardHeadersWritten = true; //must be done here to prevent recursive calls
	
	//if this is a "200 OK" response (most HTTP responses), we have some special
	//optmizations here
	if (fStatus == RTSPProtocol::kSuccessOK)
	{
		fOutputFormatter.Put(sPremadeHeaderPtr);
		StrPtrLen* cSeq = GetHeaderValue(RTSPProtocol::kCSeqHeader);
		Assert(cSeq != NULL);
		if (cSeq->Len > 1)
			fOutputFormatter.Put(*cSeq);
		else if (cSeq->Len == 1)
			fOutputFormatter.PutChar(*cSeq->Ptr);
		fOutputFormatter.PutEOL();
	}
	else
	{
		//other status codes just get built on the fly
		PutStatusLine(&fOutputFormatter, fStatus, RTSPProtocol::k10Version);
		fOutputFormatter.Put(sServerHeaderPtr);
		fOutputFormatter.PutEOL();
		AppendHeader(RTSPProtocol::kCSeqHeader, GetHeaderValue(RTSPProtocol::kCSeqHeader));
	}
	//append sessionID header
	StrPtrLen* incomingID = GetHeaderValue(RTSPProtocol::kSessionHeader);
	if ((incomingID != NULL) && (incomingID->Len > 0))
		AppendHeader(RTSPProtocol::kSessionHeader, incomingID);

	//follows the HTTP/1.1 convention: if server wants to close the connection, it
	//tags the response with the Connection: close header
	if (!fResponseKeepAlive)
		AppendHeader(RTSPProtocol::kConnectionHeader, &sCloseString);
}

void RTSPRequestInterface::SendHeader()
{
	if (!fStandardHeadersWritten)
		this->WriteStandardHeaders();
	fOutputFormatter.PutEOL();
}

void RTSPRequestInterface::ResetHeaders()
{
	fStandardHeadersWritten = false;
	fOutputFormatter.Reset();
}

QTSS_ErrorCode
RTSPRequestInterface::Send(iovec* inVec, UInt32 inNumVectors, UInt32 inTotalLength)
{
	UInt32 theCurrentOffset = fOutputFormatter.GetCurrentOffset();
	
	if (theCurrentOffset > 0)
	{
		inVec[0].iov_base = fOutputFormatter.GetBufPtr();
		inVec[0].iov_len = theCurrentOffset;
		(void)fSession->GetSocket()->WriteV(inVec, inNumVectors);
		fOutputFormatter.Reset();
		fBytesSent += theCurrentOffset;
	}
	else
		(void)fSession->GetSocket()->WriteV(&inVec[1], inNumVectors - 1);
	
	//update fBytesSent to reflect the length of the vectors
	fBytesSent += inTotalLength;
	return QTSS_NoErr;	
}

RTSPProtocol::RTSPStatusCode RTSPRequestInterface::SendErrorResponse(RTSPProtocol::RTSPStatusCode inStatusCode,
																	RTSPMessages::Message inMessageNum,
																	StrPtrLen* inStringArg)
{
	Assert(this != NULL);
	
	//call this in case a module already starting writing some headers
	this->ResetHeaders();
	
	//set RTSP headers necessary for this error response message
	this->SetStatus(inStatusCode);
	this->SetKeepAlive(false);//we want to go away!
	
	//send the response header. In all situations where errors could happen, we
	//don't really care, cause there's nothing we can do anyway!
	(void)this->SendHeader();
	
	//put the error number
	char theNumberString[50];
	::sprintf(theNumberString, "%ld ", inMessageNum);
	(void)this->Write(theNumberString, ::strlen(theNumberString));
	
	//put the message, make sure to look for a "%s" in the string, and if it exists,
	//also put the string argument
	StrPtrLen* message = RTSPServerInterface::GetRTSPMessages()->GetMessage(inMessageNum);
	Assert(message != NULL);
	
	//we can safely assume that message is in fact NULL terminated
	char* stringLocation = ::strstr(message->Ptr, "%s");
	if (stringLocation != NULL)
	{
		//write first chunk
		(void)this->Write(message->Ptr, stringLocation - message->Ptr);
		if (inStringArg != NULL && inStringArg->Len > 0)
		{
			//write string arg if it exists
			(void)this->Write(inStringArg->Ptr, inStringArg->Len);
			stringLocation += 2;
		}
		//write last chunk
		(void)this->Write(stringLocation, (message->Ptr + message->Len) - stringLocation);
	}
	else
		(void)this->Write(message->Ptr, message->Len);
		
	//send response and close up shop
	(void)this->Close();

	return inStatusCode;
}


QTSS_ErrorCode
RTSPRequestInterface::Write(char* inBuffer, UInt32 inLength)
{
	//now just write whatever remains into the output buffer
	fOutputFormatter.Put(inBuffer, inLength);
	return QTSS_NoErr;
}

QTSS_ErrorCode RTSPRequestInterface::Close()
{
	if (fOutputFormatter.GetCurrentOffset() > 0)
	{
		fBytesSent += fOutputFormatter.GetCurrentOffset();
		(void)fSession->GetSocket()->Send(	fOutputFormatter.GetBufPtr(),
											fOutputFormatter.GetCurrentOffset());
		fOutputFormatter.Reset();
	}
	fOutputFormatter.ShrinkBuffer();
	return QTSS_NoErr;
}

//This is the new model of accessing parameters one simple call based on
//the QTSS enumerated type for request parameters. Integrating QTSS directly
//into this object reduces redundancy and speeds up parameter access for QTSS
//plug-ins.
StrPtrLen*
RTSPRequestInterface::GetQTSSParameter(QTSS_ParamKeywords whichParam)
{
	if ((whichParam >= kQTSSMinRequestParamIndex) && (whichParam <= kQTSSMaxRequestParamIndex))
	{
		StrPtrLen* retval = &fQTSSConnParams[whichParam];
		
		//If the parameter currently has no value, check to see if there is a param
		//retrieval function lying around to setup the parameter
		if ((retval->Len == 0) && (sParamRetrievalFunctions[whichParam] != NULL))
			return (*sParamRetrievalFunctions[whichParam])(this);
		else
			return retval;
	}
	else
		return NULL;
}

StrPtrLen*
RTSPRequestInterface::GetHeaderValue(RTSPProtocol::RTSPHeader inHeader)
{
	if (inHeader < RTSPProtocol::kIllegalHeader)
		return &fHeaders[inHeader];
	else
		return NULL;
}

#if !__MacOSX__
int FindUsersHome(const StrPtrLen &user, char *path)
{
	int		err = 0, got = 0;
	char	buf[256];

	FILE* fd = ::fopen( "/etc/passwd", "r" );
	if (errno == 0 || fd != NULL)
	{
		while (fgets(buf, sizeof(buf)-1, fd) != NULL)
		{
			if (strlen(buf) < user.Len)
				continue;
			if (strncmp(user.Ptr, buf, user.Len) == 0)
			{
				int i;
				char *p = buf;
				// user:pw:uid:gid:name:home:shell
				for (i=0; i<5; i++)
					{
					p = strchr(p, ':');
					if (!p)
						goto skip;
					p += 1;
					}
				while (*p && *p != ':')
					*path++ = *p++;
				*path = '\0';
				got = 1;
				goto done;
			}
skip:
			;
		}
done:
		err = ::fclose(fd);
	}
	return got;
}
#endif

char* RTSPRequestInterface::GetFullPath(QTSS_ParamKeywords whichFileType, StrPtrLen* suffix)
{
	//sometimes, this parameter is generated on the fly, so we need to make sure that happens
	(void)this->GetQTSSParameter(whichFileType);

#if !__MacOSX__
	// check for user paths
	if (fQTSSConnParams[whichFileType].Ptr[1] == '~')
	{
		UInt32			len;
		StrPtrLen		user;
		StringParser	p(&fQTSSConnParams[whichFileType]);
		p.ConsumeLength(NULL, 2);		// get rid of the /~
		p.ConsumeUntil(&user, '/');		// up to next delimiter
		char*	movieDirPath = RTSPServerInterface::GetRTSPPrefs()->
					GetUsersMovieFolderPath(0, &len);
		if (len)
		{
			char*	home = malloc(1024);
			if (FindUsersHome(user, home))
			{
				strcat(home, movieDirPath);
				if (home[strlen(home)-1] != '/')
					strcat(home, "/");
				strncat(home, p.GetCurrentPosition(), p.GetDataRemaining());
				if (suffix != NULL)
					strncat(home, suffix->Ptr, suffix->Len);
				free(movieDirPath);
				return home;
			}
			else
			{
				free(home);
			}
		}
	}
#endif

	//construct a full path out of the root dir path for this request,
	//and the url path.
	UInt32 theFullPathLen = fQTSSConnParams[whichFileType].Len;
	if (suffix != NULL)
		theFullPathLen += suffix->Len;
	
	//This function allocates some memory and places the path to the root
	//directory at the beginning of the buffer. It will also change
	//theFullPathLen to be the length of the root directory path.
	UInt32 appendingLength = theFullPathLen;
	char* theFullPath = RTSPServerInterface::GetRTSPPrefs()->
							GetMovieFolderPath(theFullPathLen, &theFullPathLen);
	
	//write the rest of the path into this new buffer
	StringFormatter thePathFormatter(theFullPath + theFullPathLen, appendingLength + 1);
	thePathFormatter.Put(fQTSSConnParams[whichFileType]);
	if (suffix != NULL)
		thePathFormatter.Put(*suffix);
	thePathFormatter.PutTerminator();

	return theFullPath;
}

StrPtrLen* 	RTSPRequestInterface::GetTruncatedAbsoluteURL(RTSPRequestInterface *req)
{
	req->fQTSSConnParams[qtssTruncAbsoluteURLParam] = req->fQTSSConnParams[qtssAbsoluteURLParam];
	StrPtrLen* temp = &req->fQTSSConnParams[qtssTruncAbsoluteURLParam];
	temp->Len--;
	while (temp->Ptr[temp->Len] != '/')
		temp->Len--;
	return temp;		
}


#pragma mark __PARAM_RETRIEVAL_FUNCTIONS__

//param retrieval functions described in .h file
StrPtrLen* RTSPRequestInterface::GetTruncatedPath(RTSPRequestInterface *req)
{
	req->fQTSSConnParams[qtssFilePathTruncParam] = req->fQTSSConnParams[qtssFilePathParam];
	StrPtrLen* temp = &req->fQTSSConnParams[qtssFilePathTruncParam];
	temp->Len--;
	while (temp->Ptr[temp->Len] != '/')
		temp->Len--;
	return temp;		
}

StrPtrLen* 	RTSPRequestInterface::GetFileName(RTSPRequestInterface *req)
{
	req->fQTSSConnParams[qtssFileNameParam] = req->fQTSSConnParams[qtssFilePathParam];
	StrPtrLen* temp = &req->fQTSSConnParams[qtssFileNameParam];

	//paranoid check
	if (temp->Len == 0)
		return temp;
		
	//walk back in the file name until we hit a /
	SInt32 x = temp->Len - 1;
	for (; x > 0; x--)
		if (temp->Ptr[x] == '/')
			break;
	//once we do, make the tempPtr point to the next character after the slash,
	//and adjust the length accordingly
	if (temp->Ptr[x] == '/')
	{
		temp->Ptr = (&temp->Ptr[x]) + 1;
		temp->Len -= (x + 1);
	}
	return temp;		
}

StrPtrLen* 	RTSPRequestInterface::GetFileDigit(RTSPRequestInterface *req)
{
	req->fQTSSConnParams[qtssFileDigitParam] = req->fQTSSConnParams[qtssFilePathParam];
	StrPtrLen* temp = &req->fQTSSConnParams[qtssFileDigitParam];
	temp->Ptr += req->fQTSSConnParams[qtssFilePathParam].Len - 1;
	temp->Len = 0;
	while ((StringParser::sDigitMask[*temp->Ptr] != 0) &&
			(temp->Len <= req->fQTSSConnParams[qtssFilePathParam].Len))
	{
		temp->Ptr--;
		temp->Len++;
	}
	//termination condition means that we aren't actually on a digit right now.
	//Move pointer back onto the digit
	temp->Ptr++;
	
	return temp;
}


RTSPRequestInterface::ParamRetrievalFunction
RTSPRequestInterface::sParamRetrievalFunctions[] =
{
	NULL,						//qtssMethodParam = 0
	NULL,						//qtssFilePathParam = 1
	NULL,						//qtssURLParam = 2
	GetTruncatedPath, 			//qtssFilePathTruncParam = 3,
	GetFileName,		 		//qtssFileName = 4,
	NULL,						//qtssClientPortA = 5,
	NULL,						//qtssClientPortB = 6,
	GetFileDigit,				//qtssFileDigitParam = 7
	NULL,						//qtssAbsoluteURLParam = 8
	NULL,						//qtssFullRequestParam = 9
	NULL,						//qtssFirstLineParam = 10
	GetTruncatedAbsoluteURL		//qtssTruncAbsoluteURLParam = 11
};

