/*
 * Copyright (c) 2005-2008 Substance Kirill Grouchnikov. All Rights Reserved.
 *
 * Redistribution and use in source and binary forms, with or without 
 * modification, are permitted provided that the following conditions are met:
 * 
 *  o Redistributions of source code must retain the above copyright notice, 
 *    this list of conditions and the following disclaimer. 
 *     
 *  o Redistributions in binary form must reproduce the above copyright notice, 
 *    this list of conditions and the following disclaimer in the documentation 
 *    and/or other materials provided with the distribution. 
 *     
 *  o Neither the name of Substance Kirill Grouchnikov nor the names of 
 *    its contributors may be used to endorse or promote products derived 
 *    from this software without specific prior written permission. 
 *     
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" 
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, 
 * THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR 
 * PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR 
 * CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, 
 * EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, 
 * PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; 
 * OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, 
 * WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE 
 * OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, 
 * EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE. 
 */
package test.check;

import java.awt.*;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.lang.reflect.Method;
import java.util.Calendar;
import java.util.Date;

import javax.swing.*;
import javax.swing.event.ChangeEvent;
import javax.swing.event.ChangeListener;
import javax.swing.table.AbstractTableModel;
import javax.swing.table.JTableHeader;

import org.jvnet.lafwidget.animation.*;
import org.jvnet.lafwidget.utils.LookUtils;
import org.jvnet.lafwidget.utils.LafConstants.AnimationKind;
import org.jvnet.substance.*;
import org.jvnet.substance.theme.SubstanceTheme;
import org.jvnet.substance.theme.ThemeInfo;
import org.jvnet.substance.utils.SubstanceColorUtilities;

import com.jgoodies.forms.builder.DefaultFormBuilder;
import com.jgoodies.forms.layout.FormLayout;

/**
 * Test application panel for testing {@link JTable} component.
 * 
 * @author Kirill Grouchnikov
 */
public class TablePanel extends ControllablePanel {
	/**
	 * The table.
	 */
	private JTable table;

	/**
	 * Custom renderer for columns that contain {@link Color} data.
	 * 
	 * @author Kirill Grouchnikov
	 */
	private static class MyColorTableRenderer extends
			SubstanceDefaultTableCellRenderer {
		/*
		 * (non-Javadoc)
		 * 
		 * @see javax.swing.table.TableCellRenderer#getTableCellRendererComponent(javax.swing.JTable,
		 *      java.lang.Object, boolean, boolean, int, int)
		 */
		@Override
		public Component getTableCellRendererComponent(JTable table,
				Object value, boolean isSelected, boolean hasFocus, int row,
				int column) {
			Color color = (Color) value;
			this.setForeground(color);
			this.setBackground(SubstanceColorUtilities.invertColor(color));
			this.setText("row " + row);
			return this;
		}
	}

	/**
	 * Custom renderer for the columns that contain {@link Float} data.
	 * 
	 * @author Kirill Grouchnikov
	 */
	private static class MyFloatTableRenderer extends
			SubstanceDefaultTableCellRenderer {
		/*
		 * (non-Javadoc)
		 * 
		 * @see javax.swing.table.TableCellRenderer#getTableCellRendererComponent(javax.swing.JTable,
		 *      java.lang.Object, boolean, boolean, int, int)
		 */
		@Override
		public Component getTableCellRendererComponent(JTable table,
				Object value, boolean isSelected, boolean hasFocus, int row,
				int column) {
			int c = (10 * row) % 255;
			Color color = new Color(c, c, c);
			this.setForeground(new Color(255 - c, 0, 0));
			this.setBackground(color);
			this.setText(value.toString());
			return this;
		}
	}

	/**
	 * Custom table model.
	 * 
	 * @author Kirill Grouchnikov
	 */
	private static class MyTableModel extends AbstractTableModel {
		/**
		 * The current row count.
		 */
		private int rows;

		/**
		 * The column count.
		 */
		private int cols = 10;

		/**
		 * The table data.
		 */
		private Object[][] data;

		/**
		 * The table column classes.
		 */
		private Class<?>[] columns = new Class<?>[] { String.class,
				JComboBox.class, Boolean.class, Byte.class, Float.class,
				Double.class, String.class, Date.class, ImageIcon.class,
				Color.class };

		/**
		 * Creates the custom table model.
		 * 
		 * @param rows
		 *            Initial number of rows.
		 */
		public MyTableModel(int rows) {
			this.rows = rows;
			this.data = new Object[rows][this.cols];
			for (int i = 0; i < rows; i++) {
				this.data[i][0] = "cell " + i + ":" + 0;
				this.data[i][1] = "predef";
				this.data[i][2] = new Boolean(i % 2 == 0);
				this.data[i][3] = new Byte((byte) i);
				this.data[i][4] = new Float(i);
				this.data[i][5] = new Double(i);
				this.data[i][6] = "cell " + i + ":" + 6;
				Calendar cal = Calendar.getInstance();
				cal.set(2000 + i, 1 + i, 1 + i);
				this.data[i][7] = cal.getTime();
				int count = 0;
				if (UIManager.getLookAndFeel() instanceof SubstanceLookAndFeel) {
					for (ThemeInfo themeInfo : SubstanceLookAndFeel
							.getAllThemes().values()) {
						if (count++ == i) {
							try {
								this.data[i][8] = SubstanceImageCreator
										.getHexaMarker(
												6,
												(SubstanceTheme) Class
														.forName(
																themeInfo
																		.getClassName())
														.newInstance());
							} catch (Exception exc) {
							}
						}
					}
				}

				int comp = i * 20;
				int red = (comp / 3) % 255;
				int green = (comp / 2) % 255;
				int blue = comp % 255;
				this.data[i][9] = new Color(red, green, blue);
			}
		}

		/*
		 * (non-Javadoc)
		 * 
		 * @see javax.swing.table.AbstractTableModel#getColumnClass(int)
		 */
		@Override
		public Class<?> getColumnClass(int columnIndex) {
			return this.columns[columnIndex];
		}

		/*
		 * (non-Javadoc)
		 * 
		 * @see javax.swing.table.TableModel#getColumnCount()
		 */
		public int getColumnCount() {
			return this.cols;
		}

		/*
		 * (non-Javadoc)
		 * 
		 * @see javax.swing.table.TableModel#getRowCount()
		 */
		public int getRowCount() {
			return this.rows;
		}

		/*
		 * (non-Javadoc)
		 * 
		 * @see javax.swing.table.TableModel#getValueAt(int, int)
		 */
		public Object getValueAt(int row, int col) {
			return this.data[row][col];
		}

		/*
		 * (non-Javadoc)
		 * 
		 * @see javax.swing.table.AbstractTableModel#getColumnName(int)
		 */
		@Override
		public String getColumnName(int column) {
			return this.getColumnClass(column).getSimpleName();
		}

		/*
		 * (non-Javadoc)
		 * 
		 * @see javax.swing.table.AbstractTableModel#isCellEditable(int, int)
		 */
		@Override
		public boolean isCellEditable(int rowIndex, int columnIndex) {
			return (rowIndex % 2 == 0);
		}

		/*
		 * (non-Javadoc)
		 * 
		 * @see javax.swing.table.AbstractTableModel#setValueAt(java.lang.Object,
		 *      int, int)
		 */
		@Override
		public void setValueAt(Object value, int row, int col) {
			this.data[row][col] = value;
			this.fireTableCellUpdated(row, col);
		}
	}

	/**
	 * Creates a test panel with table.
	 */
	public TablePanel() {
		// TableColumnModel columnModel = new DefaultTableColumnModel() {
		// @Override public int getColumnCount() {
		// return 10;
		// }
		//
		// @Override public TableColumn getColumn(int columnIndex) {
		// return new TableColumn(
		// }
		// };
		this.table = new JTable(new MyTableModel(20));
		this.table.setDefaultRenderer(Color.class, new MyColorTableRenderer());
		this.table.setDefaultRenderer(Float.class, new MyFloatTableRenderer());
		final JScrollPane tableScrollpane = new JScrollPane(this.table);
		tableScrollpane.setName("Main table in table panel");

		JComboBox combo = new JComboBox(new Object[] { "aa", "bb", "cc" });
		combo.setBorder(null);
		this.table.getColumnModel().getColumn(1).setCellEditor(
				new DefaultCellEditor(combo));

		this.table
				.setSelectionMode(ListSelectionModel.MULTIPLE_INTERVAL_SELECTION);
		// We allow row selection as the default
		this.table.setCellSelectionEnabled(true);
		this.table.setRowSelectionAllowed(true);
		this.table.setColumnSelectionAllowed(false);

		this.table.setShowGrid(false);
		this.table.setDragEnabled(false);
		this.table
				.setTableHeader(new JTableHeader(this.table.getColumnModel()));

		this.setLayout(new BorderLayout());
		this.add(tableScrollpane, BorderLayout.CENTER);

		final JLabel instructional = new JLabel("Every odd row is editable");
		this.add(instructional, BorderLayout.NORTH);
		// create a looping animation to change the label foreground
		// from black to blue and back to draw some attention.
		FadeKind custom = new FadeKind("substance.testapp.table.instructional");
		FadeTracker.getInstance().trackFadeLooping(custom,
				AnimationKind.DEBUG_FAST, instructional, 0, false,
				new FadeTrackerAdapter() {
					@Override
					public void fadePerformed(FadeKind fadeKind,
							float fadeCycle10) {
						instructional.setForeground(SubstanceColorUtilities
								.getInterpolatedColor(Color.black, Color.blue,
										fadeCycle10 / 10.0));
					}
				}, -1, true);

		FormLayout lm = new FormLayout("right:pref, 4dlu, fill:pref:grow", "");
		DefaultFormBuilder builder = new DefaultFormBuilder(lm,
				new ScrollablePanel());

		builder.appendSeparator("Table settings");
		final JCheckBox isEnabled = new JCheckBox("is enabled");
		isEnabled.setSelected(table.isEnabled());
		isEnabled.addActionListener(new ActionListener() {
			public void actionPerformed(ActionEvent e) {
				table.setEnabled(isEnabled.isSelected());
				// the table header is not repainted on disabling / enabling :(
				table.getTableHeader().repaint();
			}
		});
		builder.append("Enabled", isEnabled);

		JButton changeFirstColumn = new JButton("change 1st column");
		changeFirstColumn.addActionListener(new ActionListener() {
			public void actionPerformed(ActionEvent e) {
				new Thread(new Runnable() {
					public void run() {
						for (int i = 0; i < table.getModel().getRowCount(); i++) {
							table.getModel().setValueAt(
									Thread.currentThread().getName() + " " + i,
									i, 0);
							try {
								Thread.sleep(200);
							} catch (InterruptedException e) {
							}
						}
					}
				}).start();
			}
		});
		builder.append("Change values", changeFirstColumn);

		final JSlider rowCountSlider = new JSlider(20, 1000, this.table
				.getModel().getRowCount());
		rowCountSlider.setPaintLabels(false);
		rowCountSlider.setPaintTicks(false);
		rowCountSlider.addChangeListener(new ChangeListener() {
			public void stateChanged(ChangeEvent e) {
				if (rowCountSlider.getValueIsAdjusting())
					return;
				TablePanel.this.table.setModel(new MyTableModel(rowCountSlider
						.getValue()));
			}
		});
		builder.append("Row count", rowCountSlider);

		final JCheckBox areRowsSelectable = new JCheckBox("Rows selectable");
		areRowsSelectable.setSelected(this.table.getRowSelectionAllowed());
		areRowsSelectable.addActionListener(new ActionListener() {
			public void actionPerformed(ActionEvent e) {
				TablePanel.this.table.setRowSelectionAllowed(areRowsSelectable
						.isSelected());
			}
		});
		builder.append("Selectable", areRowsSelectable);

		final JCheckBox areColsSelectable = new JCheckBox("Cols selectable");
		areColsSelectable.setSelected(this.table.getColumnSelectionAllowed());
		areColsSelectable.addActionListener(new ActionListener() {
			public void actionPerformed(ActionEvent e) {
				TablePanel.this.table
						.setColumnSelectionAllowed(areColsSelectable
								.isSelected());
			}
		});
		builder.append("", areColsSelectable);

		// TODO JDK 6 replace with regular code
		if (LookUtils.IS_JAVA_6_OR_LATER) {
			try {
				final JCheckBox isSorted = new JCheckBox("Sorted");
				Method isAutoSorter = JTable.class.getDeclaredMethod(
						"getAutoCreateRowSorter", new Class[0]);
				isSorted.setSelected((Boolean) isAutoSorter.invoke(this.table,
						new Object[0]));
				isSorted.addActionListener(new ActionListener() {
					public void actionPerformed(ActionEvent e) {
						try {
							Method setAutoSorter = JTable.class
									.getDeclaredMethod(
											"setAutoCreateRowSorter",
											new Class[] { boolean.class });
							setAutoSorter.invoke(table, new Object[] { isSorted
									.isSelected() });
							table.repaint();
							table.getTableHeader().repaint();
						} catch (Exception exc) {
						}
					}
				});
				builder.append("Sorted", isSorted);
			} catch (Exception exc) {
			}
		}

		final JCheckBox watermarkBleed = new JCheckBox("Watermark bleed");
		watermarkBleed.addActionListener(new ActionListener() {
			public void actionPerformed(ActionEvent e) {
				TablePanel.this.table.putClientProperty(
						SubstanceLookAndFeel.WATERMARK_TO_BLEED, Boolean
								.valueOf(watermarkBleed.isSelected()));
				tableScrollpane.putClientProperty(
						SubstanceLookAndFeel.WATERMARK_TO_BLEED, Boolean
								.valueOf(watermarkBleed.isSelected()));
				tableScrollpane.repaint();
			}
		});
		builder.append("Watermark", watermarkBleed);

		final JCheckBox linesVertical = new JCheckBox("Vertical visible");
		linesVertical.setSelected(this.table.getShowVerticalLines());
		linesVertical.addActionListener(new ActionListener() {
			public void actionPerformed(ActionEvent e) {
				TablePanel.this.table.setShowVerticalLines(linesVertical
						.isSelected());
			}
		});
		builder.append("Lines", linesVertical);
		final JCheckBox linesHorizontal = new JCheckBox("Horizontal visible");
		linesHorizontal.setSelected(this.table.getShowHorizontalLines());
		linesHorizontal.addActionListener(new ActionListener() {
			public void actionPerformed(ActionEvent e) {
				TablePanel.this.table.setShowHorizontalLines(linesHorizontal
						.isSelected());
			}
		});
		builder.append("", linesHorizontal);

		final JComboBox resizeModeCombo = new JComboBox(
				new Object[] { JTable.AUTO_RESIZE_OFF,
						JTable.AUTO_RESIZE_NEXT_COLUMN,
						JTable.AUTO_RESIZE_SUBSEQUENT_COLUMNS,
						JTable.AUTO_RESIZE_LAST_COLUMN,
						JTable.AUTO_RESIZE_ALL_COLUMNS });
		resizeModeCombo.setSelectedItem(this.table.getAutoResizeMode());
		resizeModeCombo.setRenderer(new SubstanceDefaultComboBoxRenderer(
				resizeModeCombo) {
			@Override
			public Component getListCellRendererComponent(JList list,
					Object value, int index, boolean isSelected,
					boolean cellHasFocus) {
				JLabel result = (JLabel) super.getListCellRendererComponent(
						list, value, index, isSelected, cellHasFocus);

				int iv = (Integer) value;
				switch (iv) {
				case JTable.AUTO_RESIZE_OFF:
					result.setText("off");
					break;
				case JTable.AUTO_RESIZE_NEXT_COLUMN:
					result.setText("next");
					break;
				case JTable.AUTO_RESIZE_SUBSEQUENT_COLUMNS:
					result.setText("subsequent");
					break;
				case JTable.AUTO_RESIZE_LAST_COLUMN:
					result.setText("last");
					break;
				case JTable.AUTO_RESIZE_ALL_COLUMNS:
					result.setText("all");
					break;
				}

				return result;
			}
		});

		resizeModeCombo.addActionListener(new ActionListener() {
			public void actionPerformed(ActionEvent e) {
				int selected = (Integer) resizeModeCombo.getSelectedItem();
				TablePanel.this.table.setAutoResizeMode(selected);
			}
		});

		builder.append("Resize mode", resizeModeCombo);

		final JCheckBox hasRollovers = new JCheckBox("Has rollover effect");
		hasRollovers.setSelected(FadeConfigurationManager.getInstance()
				.fadeAllowed(FadeKind.ROLLOVER, this.table));
		hasRollovers.addActionListener(new ActionListener() {
			public void actionPerformed(ActionEvent e) {
				if (hasRollovers.isSelected()) {
					FadeConfigurationManager.getInstance().allowFades(
							FadeKind.ROLLOVER, TablePanel.this.table);
				} else {
					FadeConfigurationManager.getInstance().disallowFades(
							FadeKind.ROLLOVER, TablePanel.this.table);
				}
			}
		});
		builder.append("Rollovers", hasRollovers);

		final JCheckBox hasSelectionAnimations = new JCheckBox(
				"Has selection effect");
		hasSelectionAnimations.setSelected(FadeConfigurationManager
				.getInstance().fadeAllowed(FadeKind.SELECTION, this.table));
		hasSelectionAnimations.addActionListener(new ActionListener() {
			public void actionPerformed(ActionEvent e) {
				if (hasSelectionAnimations.isSelected()) {
					FadeConfigurationManager.getInstance().allowFades(
							FadeKind.SELECTION, TablePanel.this.table);
				} else {
					FadeConfigurationManager.getInstance().disallowFades(
							FadeKind.SELECTION, TablePanel.this.table);
				}
			}
		});
		builder.append("Selections", hasSelectionAnimations);

		builder.appendSeparator("Header settings");
		final JCheckBox useCurrentTheme = new JCheckBox("Global theme");
		useCurrentTheme.setSelected(true);
		useCurrentTheme.addActionListener(new ActionListener() {
			public void actionPerformed(ActionEvent e) {
				TablePanel.this.table.getTableHeader().putClientProperty(
						SubstanceLookAndFeel.THEME_PROPERTY,
						(useCurrentTheme.isSelected() ? null : "Bottle Green"));
				TablePanel.this.table.getTableHeader().repaint();
			}
		});
		builder.append("Theme", useCurrentTheme);

		final JCheckBox isAlwaysActive = new JCheckBox("Is always active");
		isAlwaysActive.setSelected(false);
		isAlwaysActive.addActionListener(new ActionListener() {
			public void actionPerformed(ActionEvent e) {
				TablePanel.this.table.getTableHeader().putClientProperty(
						SubstanceLookAndFeel.PAINT_ACTIVE_PROPERTY,
						(isAlwaysActive.isSelected() ? Boolean.TRUE : null));
				TablePanel.this.table.getTableHeader().repaint();
			}
		});
		builder.append("Active", isAlwaysActive);

		builder.appendSeparator("Font settings");
		JButton tahoma12 = new JButton("Tahoma 12");
		tahoma12.addActionListener(new ActionListener() {
			public void actionPerformed(ActionEvent e) {
				table.setFont(new Font("Tahoma", Font.PLAIN, 12));
			}
		});
		builder.append("Set font", tahoma12);

		JButton tahoma13 = new JButton("Tahoma 13");
		tahoma13.addActionListener(new ActionListener() {
			public void actionPerformed(ActionEvent e) {
				table.setFont(new Font("Tahoma", Font.PLAIN, 13));
			}
		});
		builder.append("Set font", tahoma13);

		this.controlPanel = builder.getPanel();
	}
}