/* Copyright (C) 2001, 2007 United States Government as represented by
   the Administrator of the National Aeronautics and Space Administration.
   All Rights Reserved.
*/
package gov.nasa.worldwind.servers.wms;

import org.w3c.dom.Document;
import org.w3c.dom.Node;
import org.w3c.dom.NodeList;

import javax.xml.parsers.DocumentBuilder;
import javax.xml.parsers.DocumentBuilderFactory;
import javax.xml.xpath.XPath;
import javax.xml.xpath.XPathConstants;
import javax.xml.xpath.XPathFactory;
import java.io.InputStream;
import java.io.File;
import java.util.HashSet;
import java.util.Set;
import java.util.Properties;

/**
 *
 * @author brownrigg
 * @version $Id: Configuration.java 5011 2008-04-10 16:53:54Z rick $
 */
public class Configuration {
    
    public Configuration(InputStream configFile) {
        try {
            // NOTE: our tedious use of try-catch blocks around each configuration element
            // is an attempt to provide as much information about an improper
            // configuration as possible. We use private worker methods to hide some
            // of the tedium.
            
            DocumentBuilderFactory docfac = DocumentBuilderFactory.newInstance();
            DocumentBuilder builder = docfac.newDocumentBuilder();
            Document doc = builder.parse(configFile);
            
            XPathFactory xpfac = XPathFactory.newInstance();
            XPath xpath = xpfac.newXPath();
            
            // get the server listener port...
            getListenerPort(xpath, doc);
            
            // GDAL's path and work directory for temp files...
            getGDALInfo(xpath, doc);

            // get optional size of thread pool...
            getThreadPoolSize(xpath, doc);
            
            // extract the map sources...
            getMapSources(xpath, doc);
            
        } catch (Exception ex) {
            logMessage("Unable to load/complete configuration: " + ex.toString());
            throw new Error(ex);
        }
    }

    // Gettors...
    public int getListenerPort() { return listenerPort; }
    public String getGDALPath() { return gdalPath; }
    public String getWorkDirectory() { return workDir; }
    public int getThreadPoolSize() { return threadPoolSize; }
    public Set<MapSource> getMapSources() { return mapSources; }

    //
    private void getListenerPort(XPath xpath, Document doc) {
        try {
            String str = xpath.evaluate(PORT, doc);
            try {
                listenerPort = Integer.parseInt(str);
            } catch (NumberFormatException ex) {
                logMessage("Invalid or missing listener-port: " + ex.toString());
                logMessage("    using default of " + listenerPort);
            }
        } catch (Exception ex) {
            logMessage("Error parsing ListenerPort configuration: " + ex.toString());
            throw new Error(ex);
        }
    }
    
    private void getGDALInfo(XPath xpath, Document doc) {
        try {
            // get the pathname with which to locate the GDAL executables...
            String str = xpath.evaluate(GDALPATH, doc);
            if ("".equals(str))
                    logMessage("WARNING: Missing GDAL path; GDAL-based map generators will not function correctly");
            else
                gdalPath = str;

            // get the working directory for temp files created by GDAL...
            str = xpath.evaluate(WORKDIR, doc);
            if ("".equals(str))
                logMessage("Missing working-directory, using default of \"" + workDir + "\"");
            else
                workDir = str;

            // make sure working directory exists and we can write to it...
            File workDirFile = new File(workDir);
            if ( (workDirFile.exists() && !workDirFile.canWrite()) ||
                    (!workDirFile.exists()) && !workDirFile.mkdirs())
                    logMessage("WARNING: GDAL working-directory \"" + workDir + "\" can not be written to; " +
                            "GDAL-based map generators will not function correctly");


        } catch (Exception ex) {
            logMessage("Error parsing Work-Directory configuration: " + ex.toString());
            throw new Error(ex);
        }
    }

    private void getThreadPoolSize(XPath xpath, Document doc) {
        try {
            // this is an optional configuration point...
            String str = xpath.evaluate(THREADPOOL, doc);
            if (!"".equals(str)) {
                try {
                    int size = Integer.parseInt(str);
                    if (size <= 0)
                        throw new NumberFormatException("non-positive integer");
                    threadPoolSize = size;
                } catch (NumberFormatException ex) {
                    logMessage("Invalid size given for thread pool: " + ex.toString());
                    logMessage("    using default of " + threadPoolSize);
                }
            }

        } catch (Exception ex) {
            logMessage("Error parsing Work-Directory configuration: " + ex.toString());
            throw new Error(ex);
        }
    }

    private void getMapSources(XPath xpath, Document doc) {
        try {
            // These can be hierarchical, so we get all the top level MapSources, 
            // and let our helper method descend any nested definitions...
            Node root = (Node) xpath.evaluate(ROOT_NODE, doc, XPathConstants.NODE);
            NodeList nlist = (NodeList) xpath.evaluate(MAPSOURCE, root, XPathConstants.NODESET);
            for (int i = 0; i < nlist.getLength(); i++) {
                try {
                    Node n = nlist.item(i);
                    MapSource map = parseMapSource(xpath, n, null);
                    mapSources.add(map);
                } catch (Exception ex) {
                    logMessage("Error parsing MapSource #" + i + ": " + ex.toString());
                }
            }
            if (mapSources.size() == 0) {
                logMessage("No Map-Sources specified -- all GetMap requests will be rejected!");
            }
        } catch (Exception ex) {
            logMessage("Error parsing Map-Sources configuration: " + ex.toString());
            throw new Error(ex);
        }
    }
   
    private MapSource parseMapSource(XPath xpath, Node n, MapSource parent) throws Exception {
        String name = xpath.evaluate(MAPSOURCE_NAME, n);
        String dir = xpath.evaluate(MAPSOURCE_ROOTDIR, n);
        String title = xpath.evaluate(MAPSOURCE_TITLE, n);
        String className = xpath.evaluate(MAPSOURCE_CLASS, n);

        /*** TODO -- need to re-evaluate how to determine properly configured MapSources, in the context
         * of our new support for hierarchical named and un-named sources.
         */
        if (!"".equals(name) && ("".equals(title) || "".equals(dir) || "".equals(className))) {
            throw new Exception("MapSource is incompletely specified; ignored");
        }
        
        Class serviceClass = null;
        if (!"".equals(name)) {
            try {
                serviceClass = Class.forName(className);
                if (!MapGenerator.class.isAssignableFrom(serviceClass)) {
                    throw new Exception();
                }
            } catch (Exception ex) {
                throw new Exception(className + " is not found or is not of type MapService");
            }
        }
        
        if ("".equals(name) && "".equals(title))
            throw new Exception("Unnamed MapSources must have a title string");

        // Get optional description...
        String description = xpath.evaluate(MAPSOURCE_DESCRIPTION, n);
        String keywords = xpath.evaluate(MAPSOURCE_KEYWORDS, n);

        // Get any optional MapSource-specific properties...
        Properties properties = new Properties();
        NodeList props = (NodeList) xpath.evaluate(MAPSOURCE_PROPERTIES, n, XPathConstants.NODESET);
        if (props.getLength() > 0) {
            for (int j = 0; j < props.getLength(); j++) {
                Node p = props.item(j);
                String propName = xpath.evaluate(MAPSOURCE_PROPNAME, p);
                String propValue = xpath.evaluate(MAPSOURCE_PROPVAL, p);
                if ("".equals(propName) || "".equals(propValue)) {
                    SysLog.inst().warning("Missing name/value for <property> in MapSource " + name);
                } else {
                    properties.put(propName, propValue);
                }
            }
        }
        
        MapSource map = new MapSource(parent, name, title, dir, serviceClass, description, keywords, properties);

        // handle any nested definitions...
        NodeList nlist = (NodeList) xpath.evaluate(MAPSOURCE, n, XPathConstants.NODESET);
        for (int i = 0; i < nlist.getLength(); i++) {
                Node node = nlist.item(i);
                MapSource childMap = parseMapSource(xpath, node, map);
                map.addChild(childMap);
        }
        
        return map;
    }

    private void logMessage(String msg) {
        SysLog.inst().warning("CONFIG: " + msg);
    }
    
    private int listenerPort = 8080;
    private int threadPoolSize = 20;
    private HashSet<MapSource> mapSources = new HashSet<MapSource>();
    private String gdalPath;
    private String workDir;
    {
        workDir = System.getProperty("java.io.tmpdir");
    }
    
    private static final String ROOT_NODE = "/wms-config";
    private static final String SERVER = "//server";
    private static final String PORT = SERVER + "/port";
    private static final String GDALPATH = SERVER + "/gdal-path";
    private static final String WORKDIR = SERVER + "/work-directory";
    private static final String THREADPOOL = SERVER + "/thread-pool-size";
    private static final String MAPSOURCE = "./mapsource";
    private static final String MAPSOURCE_NAME = "@name";
    private static final String MAPSOURCE_TITLE = "@title";
    private static final String MAPSOURCE_ROOTDIR = "./root-dir";
    private static final String MAPSOURCE_CLASS = "./class";
    private static final String MAPSOURCE_DESCRIPTION = "./description";
    private static final String MAPSOURCE_KEYWORDS = "@keywords";
    private static final String MAPSOURCE_PROPERTIES = "./property";
    private static final String MAPSOURCE_PROPNAME = "@name";
    private static final String MAPSOURCE_PROPVAL = "@value";
}
