/***************************************************************************

	Taito Qix hardware

	driver by John Butler, Ed Mueller, Aaron Giles

	Games supported:
		* Qix (3 sets)
		* Qix II Tournament
		* Zookeeper (3 sets)
		* Space Dungeon
		* Kram (2 sets)
		* The Electric Yo-Yo (2 sets)
		* Slither (2 sets)
		* Complex X

	Known bugs:
		* none at this time

	Notes:
		* the encrypted version of Kram doesn't have the protection MCU

****************************************************************************

	Memory map

****************************************************************************

	========================================================================
	CPU #1 (all but Zookeeper)
	========================================================================
	8000-83FF   R/W   xxxxxxxx    Shared RAM with CPU #2
	8400-87FF   R/W   xxxxxxxx    Local RAM
	8800        R/W   xxxxxxxx    ACIA base address
	8C00        R/W   --------    Video FIRQ assert
	8C01        R/W   --------    Data FIRQ acknowledge
	9000-93FF   R/W   xxxxxxxx    Sound control/data IRQ PIA (pia_3)
	9400-97FF   R/W   xxxxxxxx    Coin/P1 inputs PIA (pia_0)
	9800-9BFF   R/W   xxxxxxxx    Spare/P2 inputs PIA (pia_1)
	9C00-9FFF   R/W   xxxxxxxx    P2 inputs/coin control PIA (pia_2)
	A000-FFFF   R     xxxxxxxx    Program ROM
	========================================================================
	Interrupts:
		NMI not connected
		IRQ generated by pia_3
		FIRQ generated by CPU #2
	========================================================================

	========================================================================
	CPU #1 (Zookeeper)
	========================================================================
	0000-03FF   R/W   xxxxxxxx    Shared RAM with CPU #2
	0400-07FF   R/W   xxxxxxxx    Local RAM
	0800        R/W   xxxxxxxx    ACIA base address
	0C00        R/W   --------    Video FIRQ assert
	0C01        R/W   --------    Data FIRQ acknowledge
	1000-13FF   R/W   xxxxxxxx    Sound control/data IRQ PIA (pia_3)
	1400-17FF   R/W   xxxxxxxx    Coin/P1 inputs PIA (pia_0)
	1800-1BFF   R/W   xxxxxxxx    Spare/P2 inputs PIA (pia_1)
	1C00-1FFF   R/W   xxxxxxxx    P2 inputs/coin control PIA (pia_2)
	8000-FFFF   R     xxxxxxxx    Program ROM
	========================================================================
	Interrupts:
		NMI not connected
		IRQ generated by pia_3
		FIRQ generated by CPU #2
	========================================================================

	========================================================================
	CPU #2 (all)
	========================================================================
	0000-7FFF   R/W   xxxxxxxx    Direct video RAM access
	8000-83FF   R/W   xxxxxxxx    Shared RAM with CPU #1
	8400-87FF   R/W   xxxxxxxx    CMOS NVRAM
	8800          W   xxxxxxxx    LED/Palette banking
	                  xxxxxx--       (LED control)
	                  ------xx       (palette bank select)
	8801          W   ------xx    Banked ROM select (Zookeeper only)
	8C00        R/W   --------    Data FIRQ assert
	8C01        R/W   --------    Video FIRQ acknowledge
	9000-93FF   R/W   xxxxxxxx    Palette RAM
	                  xx------       (red component)
	                  --xx----       (green component)
	                  ----xx--       (blue component)
	                  ------xx       (intensity)
	9400        R/W   xxxxxxxx    Video RAM latched access
	9401        R/W   xxxxxxxx    Video RAM access mask (Slither only)
	9402-9403     W   xxxxxxxx    Video RAM address latch
	9800        R     xxxxxxxx    Current scanline
	9C00-9FFF   R/W   xxxxxxxx    68A45 video controller
	A000-BFFF   R     xxxxxxxx    Banked ROM (Zookeeper only)
	C000-FFFF   R     xxxxxxxx    Program ROM
	========================================================================
	Interrupts:
		NMI not connected
		IRQ not connected
		FIRQ generated by CPU #1
	========================================================================

	========================================================================
	CPU #3 (all but Slither)
	========================================================================
	0000-007F   R/W   xxxxxxxx    Program RAM
	2000-2003   R/W   xxxxxxxx    TMS5200 control PIA (pia_5; not used)
	4000-4003   R/W   xxxxxxxx    Sound control/DAC PIA (pia_4)
	D000-FFFF   R     xxxxxxxx    Program ROM
	========================================================================
	Interrupts:
		NMI not connected
		IRQ generated by pia_4
	========================================================================

****************************************************************************

	Qix uses two 6809 CPUs: one for data and sound and the other for
	video. Communication between the two CPUs is done using a 4K RAM
	space at $8000 (for ZooKeeper the data cpu maps it at $0000 and the
	video cpu at $8000) which both CPUs have direct access. FIRQs (fast
	interrupts) are generated by each CPU to interrupt the other at
	specific times.

	A third CPU, a 6802, is used for sample playback. It drives an 8-bit
	DAC and according to the schematics a TMS5220 speech chip, which is
	never accessed. ROM u27 is the only code needed. A sound command from
	the data CPU causes an IRQ to fire on the 6802 and the sound playback
	is started.

	The coin door switches and player controls are connected to the CPUs
	by Mototola 6821 PIAs. These devices are memory mapped as shown below.

	The screen is 256x256 with eight bit pixels (64K). The screen is
	divided into two halves each half mapped by the video CPU at
	$0000-$7FFF. The high order bit of the address latch at $9402
	specifies which half of the screen is being accessed.

	Timing is critical in the hardware. Communications between the data,
	video, and sound CPUs are all very sensitive.

	The address latch works as follows. When the video CPU accesses $9400,
	the screen address is computed by using the values at $9402 (high byte)
	and $9403 (low byte) to get a value between $0000-$FFFF. The value at
	that location is either returned or written.

	The scan line at $9800 on the video CPU records where the scan line is
	on the display (0-255). Several places in the ROM code wait until the
	scan line reaches zero before continuing.

****************************************************************************

	QIX NONVOLATILE CMOS MEMORY MAP (CPU #2 -- Video) $8400-$87ff
		$86A9 - $86AA:	When CMOS is valid, these bytes are $55AA
		$86AC - $86C3:	AUDIT TOTALS -- 4 4-bit BCD digits per setting
						(All totals default to: 0000)
						$86AC: TOTAL PAID CREDITS
						$86AE: LEFT COINS
						$86B0: CENTER COINS
						$86B2: RIGHT COINS
						$86B4: PAID CREDITS
						$86B6: AWARDED CREDITS
						$86B8: % FREE PLAYS
						$86BA: MINUTES PLAYED
						$86BC: MINUTES AWARDED
						$86BE: % FREE TIME
						$86C0: AVG. GAME [SEC]
						$86C2: HIGH SCORES
		$86C4 - $86FF:	High scores -- 10 scores/names, consecutive in memory
						Six 4-bit BCD digits followed by 3 ascii bytes
						(Default: 030000 QIX)
		$8700		 :	LANGUAGE SELECT (Default: $32)
						ENGLISH = $32  FRANCAIS = $33  ESPANOL = $34  DEUTSCH = $35
		$87D9 - $87DF:	COIN SLOT PROGRAMMING -- 2 4-bit BCD digits per setting
						$87D9: STANDARD COINAGE SETTING  (Default: 01)
						$87DA: COIN MULTIPLIERS LEFT (Default: 01)
						$87DB: COIN MULTIPLIERS CENTER (Default: 04)
						$87DC: COIN MULTIPLIERS RIGHT (Default: 01)
						$87DD: COIN UNITS FOR CREDIT (Default: 01)
						$87DE: COIN UNITS FOR BONUS (Default: 00)
						$87DF: MINIMUM COINS (Default: 00)
		$87E0 - $87EA:	LOCATION PROGRAMMING -- 2 4-bit BCD digits per setting
						$87E0: BACKUP HSTD [0000] (Default: 03)
						$87E1: MAXIMUM CREDITS (Default: 10)
						$87E2: NUMBER OF TURNS (Default: 03)
						$87E3: THRESHOLD (Default: 75)
						$87E4: TIME LINE (Default: 37)
						$87E5: DIFFICULTY 1 (Default: 01)
						$87E6: DIFFICULTY 2 (Default: 01)
						$87E7: DIFFICULTY 3 (Default: 01)
						$87E8: DIFFICULTY 4 (Default: 01)
						$87E9: ATTRACT SOUND (Default: 01)
						$87EA: TABLE MODE (Default: 00)

***************************************************************************/

#include "driver.h"
#include "qix.h"
#include "machine/6821pia.h"
#include "vidhrdw/generic.h"


/* Constants */
#define MAIN_CLOCK_OSC		20000000	/* 20 MHz */
#define SLITHER_CLOCK_OSC	21300000	/* 21.3 MHz */
#define SOUND_CLOCK_OSC		7372800		/* 7.3728 MHz */
#define COIN_CLOCK_OSC		4000000		/* 4 MHz */



/*************************************
 *
 *	Data CPU memory handlers
 *
 *************************************/

static ADDRESS_MAP_START( readmem_data, ADDRESS_SPACE_PROGRAM, 8 )
	AM_RANGE(0x8000, 0x83ff) AM_READ(qix_sharedram_r)
	AM_RANGE(0x8400, 0x87ff) AM_READ(MRA8_RAM)
	AM_RANGE(0x8800, 0x8800) AM_READ(MRA8_NOP)   /* ACIA */
	AM_RANGE(0x8c00, 0x8c00) AM_READ(qix_video_firq_r)
	AM_RANGE(0x8c01, 0x8c01) AM_READ(qix_data_firq_ack_r)
	AM_RANGE(0x9000, 0x93ff) AM_READ(pia_3_r)
	AM_RANGE(0x9400, 0x97ff) AM_READ(pia_0_r)
	AM_RANGE(0x9800, 0x9bff) AM_READ(pia_1_r)
	AM_RANGE(0x9c00, 0x9fff) AM_READ(pia_2_r)
	AM_RANGE(0xa000, 0xffff) AM_READ(MRA8_ROM)
ADDRESS_MAP_END


static ADDRESS_MAP_START( writemem_data, ADDRESS_SPACE_PROGRAM, 8 )
	AM_RANGE(0x8000, 0x83ff) AM_WRITE(qix_sharedram_w) AM_BASE(&qix_sharedram)
	AM_RANGE(0x8400, 0x87ff) AM_WRITE(MWA8_RAM)
	AM_RANGE(0x8c00, 0x8c00) AM_WRITE(qix_video_firq_w)
	AM_RANGE(0x8c01, 0x8c01) AM_WRITE(qix_data_firq_ack_w)
	AM_RANGE(0x9000, 0x93ff) AM_WRITE(pia_3_w)
	AM_RANGE(0x9400, 0x97ff) AM_WRITE(qix_pia_0_w)
	AM_RANGE(0x9800, 0x9bff) AM_WRITE(pia_1_w)
	AM_RANGE(0x9c00, 0x9fff) AM_WRITE(pia_2_w)
	AM_RANGE(0xa000, 0xffff) AM_WRITE(MWA8_ROM)
ADDRESS_MAP_END


static ADDRESS_MAP_START( zoo_readmem_data, ADDRESS_SPACE_PROGRAM, 8 )
	AM_RANGE(0x0000, 0x03ff) AM_READ(qix_sharedram_r)
	AM_RANGE(0x0400, 0x07ff) AM_READ(MRA8_RAM)
	AM_RANGE(0x0800, 0x0800) AM_READ(MRA8_NOP)   /* ACIA */
	AM_RANGE(0x0c00, 0x0c00) AM_READ(qix_video_firq_r)
	AM_RANGE(0x0c01, 0x0c01) AM_READ(qix_data_firq_ack_r)
	AM_RANGE(0x1000, 0x13ff) AM_READ(pia_3_r)
	AM_RANGE(0x1400, 0x17ff) AM_READ(pia_0_r)
	AM_RANGE(0x1900, 0x1bff) AM_READ(pia_1_r)
	AM_RANGE(0x1c00, 0x1fff) AM_READ(pia_2_r)
	AM_RANGE(0x8000, 0xffff) AM_READ(MRA8_ROM)
ADDRESS_MAP_END


static ADDRESS_MAP_START( zoo_writemem_data, ADDRESS_SPACE_PROGRAM, 8 )
	AM_RANGE(0x0000, 0x03ff) AM_WRITE(qix_sharedram_w) AM_BASE(&qix_sharedram)
	AM_RANGE(0x0400, 0x07ff) AM_WRITE(MWA8_RAM)
	AM_RANGE(0x0c00, 0x0c00) AM_WRITE(qix_video_firq_w)
	AM_RANGE(0x0c01, 0x0c01) AM_WRITE(qix_data_firq_ack_w)
	AM_RANGE(0x1000, 0x13ff) AM_WRITE(pia_3_w)
	AM_RANGE(0x1400, 0x17ff) AM_WRITE(qix_pia_0_w)
	AM_RANGE(0x1900, 0x1bff) AM_WRITE(pia_1_w)
	AM_RANGE(0x1c00, 0x1fff) AM_WRITE(pia_2_w)
	AM_RANGE(0x8000, 0xffff) AM_WRITE(MWA8_ROM)
ADDRESS_MAP_END



/*************************************
 *
 *	Video CPU memory handlers
 *
 *************************************/

static ADDRESS_MAP_START( readmem_video, ADDRESS_SPACE_PROGRAM, 8 )
	AM_RANGE(0x0000, 0x7fff) AM_READ(qix_videoram_r)
	AM_RANGE(0x8000, 0x83ff) AM_READ(qix_sharedram_r)
	AM_RANGE(0x8400, 0x87ff) AM_READ(MRA8_RAM)
	AM_RANGE(0x8c00, 0x8c00) AM_READ(qix_data_firq_r)
	AM_RANGE(0x8c01, 0x8c01) AM_READ(qix_video_firq_ack_r)
	AM_RANGE(0x9000, 0x93ff) AM_READ(MRA8_RAM)
	AM_RANGE(0x9400, 0x9400) AM_READ(qix_addresslatch_r)
	AM_RANGE(0x9800, 0x9800) AM_READ(qix_scanline_r)
	AM_RANGE(0xa000, 0xffff) AM_READ(MRA8_ROM)
ADDRESS_MAP_END


static ADDRESS_MAP_START( writemem_video, ADDRESS_SPACE_PROGRAM, 8 )
	AM_RANGE(0x0000, 0x7fff) AM_WRITE(qix_videoram_w)
	AM_RANGE(0x8000, 0x83ff) AM_WRITE(qix_sharedram_w)
	AM_RANGE(0x8400, 0x87ff) AM_WRITE(MWA8_RAM) AM_BASE(&generic_nvram) AM_SIZE(&generic_nvram_size)
	AM_RANGE(0x8800, 0x8800) AM_WRITE(qix_palettebank_w)
	AM_RANGE(0x8c00, 0x8c00) AM_WRITE(qix_data_firq_w)
	AM_RANGE(0x8c01, 0x8c01) AM_WRITE(qix_video_firq_ack_w)
	AM_RANGE(0x9000, 0x93ff) AM_WRITE(qix_paletteram_w) AM_BASE(&paletteram)
	AM_RANGE(0x9400, 0x9400) AM_WRITE(qix_addresslatch_w)
	AM_RANGE(0x9402, 0x9403) AM_WRITE(MWA8_RAM) AM_BASE(&qix_videoaddress)
	AM_RANGE(0x9c00, 0x9fff) AM_WRITE(MWA8_RAM) /* Video controller */
	AM_RANGE(0xa000, 0xffff) AM_WRITE(MWA8_ROM)
ADDRESS_MAP_END


static ADDRESS_MAP_START( zoo_readmem_video, ADDRESS_SPACE_PROGRAM, 8 )
	AM_RANGE(0x0000, 0x7fff) AM_READ(qix_videoram_r)
	AM_RANGE(0x8000, 0x83ff) AM_READ(qix_sharedram_r)
	AM_RANGE(0x8400, 0x87ff) AM_READ(MRA8_RAM)
	AM_RANGE(0x8c00, 0x8c00) AM_READ(qix_data_firq_r)
	AM_RANGE(0x8c01, 0x8c01) AM_READ(qix_video_firq_ack_r)
	AM_RANGE(0x9000, 0x93ff) AM_READ(MRA8_RAM)
	AM_RANGE(0x9400, 0x9400) AM_READ(qix_addresslatch_r)
	AM_RANGE(0x9800, 0x9800) AM_READ(qix_scanline_r)
	AM_RANGE(0xa000, 0xbfff) AM_READ(MRA8_BANK1)
	AM_RANGE(0xc000, 0xffff) AM_READ(MRA8_ROM)
ADDRESS_MAP_END


static ADDRESS_MAP_START( zoo_writemem_video, ADDRESS_SPACE_PROGRAM, 8 )
	AM_RANGE(0x0000, 0x7fff) AM_WRITE(qix_videoram_w)
	AM_RANGE(0x8000, 0x83ff) AM_WRITE(qix_sharedram_w)
	AM_RANGE(0x8400, 0x87ff) AM_WRITE(MWA8_RAM) AM_BASE(&generic_nvram) AM_SIZE(&generic_nvram_size)
	AM_RANGE(0x8800, 0x8800) AM_WRITE(qix_palettebank_w)
	AM_RANGE(0x8801, 0x8801) AM_WRITE(zoo_bankswitch_w)
	AM_RANGE(0x8c00, 0x8c00) AM_WRITE(qix_data_firq_w)
	AM_RANGE(0x8c01, 0x8c01) AM_WRITE(qix_video_firq_ack_w)
	AM_RANGE(0x9000, 0x93ff) AM_WRITE(qix_paletteram_w) AM_BASE(&paletteram)
	AM_RANGE(0x9400, 0x9400) AM_WRITE(qix_addresslatch_w)
	AM_RANGE(0x9402, 0x9403) AM_WRITE(MWA8_RAM) AM_BASE(&qix_videoaddress)
	AM_RANGE(0x9c00, 0x9fff) AM_WRITE(MWA8_RAM) /* Video controller */
	AM_RANGE(0xa000, 0xffff) AM_WRITE(MWA8_ROM)
ADDRESS_MAP_END



/*************************************
 *
 *	Sound CPU memory handlers
 *
 *************************************/

static ADDRESS_MAP_START( readmem_sound, ADDRESS_SPACE_PROGRAM, 8 )
	AM_RANGE(0x0000, 0x007f) AM_READ(MRA8_RAM)
	AM_RANGE(0x2000, 0x2003) AM_READ(pia_5_r)
	AM_RANGE(0x4000, 0x4003) AM_READ(pia_4_r)
	AM_RANGE(0xd000, 0xffff) AM_READ(MRA8_ROM)
ADDRESS_MAP_END


static ADDRESS_MAP_START( writemem_sound, ADDRESS_SPACE_PROGRAM, 8 )
	AM_RANGE(0x0000, 0x007f) AM_WRITE(MWA8_RAM)
	AM_RANGE(0x2000, 0x2003) AM_WRITE(pia_5_w)
	AM_RANGE(0x4000, 0x4003) AM_WRITE(pia_4_w)
	AM_RANGE(0xd000, 0xffff) AM_WRITE(MWA8_ROM)
ADDRESS_MAP_END



/*************************************
 *
 *	Coin CPU memory handlers
 *
 *************************************/

static ADDRESS_MAP_START( mcu_readmem, ADDRESS_SPACE_PROGRAM, 8 )
	ADDRESS_MAP_FLAGS( AMEF_ABITS(11) )
	AM_RANGE(0x0000, 0x0000) AM_READ(qix_68705_portA_r)
	AM_RANGE(0x0001, 0x0001) AM_READ(qix_68705_portB_r)
	AM_RANGE(0x0002, 0x0002) AM_READ(qix_68705_portC_r)
	AM_RANGE(0x0010, 0x007f) AM_READ(MRA8_RAM)
	AM_RANGE(0x0080, 0x07ff) AM_READ(MRA8_ROM)
ADDRESS_MAP_END


static ADDRESS_MAP_START( mcu_writemem, ADDRESS_SPACE_PROGRAM, 8 )
	ADDRESS_MAP_FLAGS( AMEF_ABITS(11) )
	AM_RANGE(0x0000, 0x0000) AM_WRITE(qix_68705_portA_w) AM_BASE(&qix_68705_port_out)
	AM_RANGE(0x0001, 0x0001) AM_WRITE(qix_68705_portB_w)
	AM_RANGE(0x0002, 0x0002) AM_WRITE(qix_68705_portC_w)
	AM_RANGE(0x0004, 0x0007) AM_WRITE(MWA8_RAM) AM_BASE(&qix_68705_ddr)
	AM_RANGE(0x0010, 0x007f) AM_WRITE(MWA8_RAM)
	AM_RANGE(0x0080, 0x07ff) AM_WRITE(MWA8_ROM)
ADDRESS_MAP_END



/*************************************
 *
 *	Input port definitions
 *
 *************************************/

#define COIN_PORT \
	PORT_BIT(0x01, IP_ACTIVE_LOW, IPT_SERVICE ) PORT_NAME("Test Advance") PORT_CODE(KEYCODE_F1) \
	PORT_BIT(0x02, IP_ACTIVE_LOW, IPT_SERVICE ) PORT_NAME("Test Next line") PORT_CODE(KEYCODE_F2) \
	PORT_BIT(0x04, IP_ACTIVE_LOW, IPT_SERVICE ) PORT_NAME("Test Slew Up") PORT_CODE(KEYCODE_F5) \
	PORT_BIT(0x08, IP_ACTIVE_LOW, IPT_SERVICE ) PORT_NAME("Test Slew Down") PORT_CODE(KEYCODE_F6) \
	PORT_BIT( 0x10, IP_ACTIVE_LOW, IPT_COIN1 ) \
	PORT_BIT( 0x20, IP_ACTIVE_LOW, IPT_COIN2 ) \
	PORT_BIT( 0x40, IP_ACTIVE_LOW, IPT_COIN3 ) \
	PORT_BIT( 0x80, IP_ACTIVE_LOW, IPT_TILT )


INPUT_PORTS_START( qix )
	PORT_START	/* PIA 0 Port A (PLAYER 1) */
    PORT_BIT( 0x01, IP_ACTIVE_LOW, IPT_JOYSTICK_UP ) PORT_4WAY
    PORT_BIT( 0x02, IP_ACTIVE_LOW, IPT_JOYSTICK_RIGHT ) PORT_4WAY
    PORT_BIT( 0x04, IP_ACTIVE_LOW, IPT_JOYSTICK_DOWN ) PORT_4WAY
    PORT_BIT( 0x08, IP_ACTIVE_LOW, IPT_JOYSTICK_LEFT ) PORT_4WAY
	PORT_BIT( 0x10, IP_ACTIVE_LOW, IPT_BUTTON2 )
	PORT_BIT( 0x20, IP_ACTIVE_LOW, IPT_START2 )
	PORT_BIT( 0x40, IP_ACTIVE_LOW, IPT_START1 )
	PORT_BIT( 0x80, IP_ACTIVE_LOW, IPT_BUTTON1 )

	PORT_START	/* PIA 0 Port B (COIN) */
	COIN_PORT

	PORT_START	/* PIA 1 Port A (SPARE) */
	PORT_BIT( 0xff, IP_ACTIVE_LOW, IPT_UNKNOWN )

    PORT_START	/* PIA 1 Port B (PLAYER 1/2) */
	PORT_BIT( 0xff, IP_ACTIVE_LOW, IPT_UNKNOWN )

	PORT_START	/* PIA 2 Port A (PLAYER 2) */
    PORT_BIT( 0x01, IP_ACTIVE_LOW, IPT_JOYSTICK_UP ) PORT_4WAY PORT_COCKTAIL
    PORT_BIT( 0x02, IP_ACTIVE_LOW, IPT_JOYSTICK_RIGHT ) PORT_4WAY PORT_COCKTAIL
    PORT_BIT( 0x04, IP_ACTIVE_LOW, IPT_JOYSTICK_DOWN ) PORT_4WAY PORT_COCKTAIL
    PORT_BIT( 0x08, IP_ACTIVE_LOW, IPT_JOYSTICK_LEFT ) PORT_4WAY PORT_COCKTAIL
    PORT_BIT( 0x10, IP_ACTIVE_LOW, IPT_BUTTON2 ) PORT_COCKTAIL
    PORT_BIT( 0x60, IP_ACTIVE_LOW, IPT_UNKNOWN )
    PORT_BIT( 0x80, IP_ACTIVE_LOW, IPT_BUTTON1 ) PORT_COCKTAIL
INPUT_PORTS_END


INPUT_PORTS_START( sdungeon )
	PORT_START	/* PIA 0 Port A (PLAYER 1) */
    PORT_BIT( 0x01, IP_ACTIVE_LOW, IPT_JOYSTICKLEFT_UP ) PORT_8WAY
    PORT_BIT( 0x02, IP_ACTIVE_LOW, IPT_JOYSTICKLEFT_RIGHT ) PORT_8WAY
    PORT_BIT( 0x04, IP_ACTIVE_LOW, IPT_JOYSTICKLEFT_DOWN ) PORT_8WAY
    PORT_BIT( 0x08, IP_ACTIVE_LOW, IPT_JOYSTICKLEFT_LEFT ) PORT_8WAY
    PORT_BIT( 0x10, IP_ACTIVE_LOW, IPT_JOYSTICKRIGHT_UP ) PORT_8WAY
    PORT_BIT( 0x20, IP_ACTIVE_LOW, IPT_JOYSTICKRIGHT_RIGHT ) PORT_8WAY
    PORT_BIT( 0x40, IP_ACTIVE_LOW, IPT_JOYSTICKRIGHT_DOWN ) PORT_8WAY
    PORT_BIT( 0x80, IP_ACTIVE_LOW, IPT_JOYSTICKRIGHT_LEFT ) PORT_8WAY

	PORT_START	/* PIA 0 Port B (COIN) */
	COIN_PORT

	PORT_START	/* PIA 1 Port A (SPARE) */
	PORT_BIT( 0xff, IP_ACTIVE_LOW, IPT_UNKNOWN )

    PORT_START	/* PIA 1 Port B (PLAYER 1/2) */
    PORT_BIT( 0x01, IP_ACTIVE_LOW, IPT_START1 )
    PORT_BIT( 0x02, IP_ACTIVE_LOW, IPT_START2 )
    PORT_BIT( 0xfc, IP_ACTIVE_LOW, IPT_UNKNOWN )

	PORT_START	/* PIA 2 Port A (PLAYER 2) */
    PORT_BIT( 0x01, IP_ACTIVE_LOW, IPT_JOYSTICKLEFT_UP ) PORT_8WAY PORT_COCKTAIL
    PORT_BIT( 0x02, IP_ACTIVE_LOW, IPT_JOYSTICKLEFT_RIGHT ) PORT_8WAY PORT_COCKTAIL
    PORT_BIT( 0x04, IP_ACTIVE_LOW, IPT_JOYSTICKLEFT_DOWN ) PORT_8WAY PORT_COCKTAIL
    PORT_BIT( 0x08, IP_ACTIVE_LOW, IPT_JOYSTICKLEFT_LEFT ) PORT_8WAY PORT_COCKTAIL
    PORT_BIT( 0x10, IP_ACTIVE_LOW, IPT_JOYSTICKRIGHT_UP ) PORT_8WAY PORT_COCKTAIL
    PORT_BIT( 0x20, IP_ACTIVE_LOW, IPT_JOYSTICKRIGHT_RIGHT ) PORT_8WAY PORT_COCKTAIL
    PORT_BIT( 0x40, IP_ACTIVE_LOW, IPT_JOYSTICKRIGHT_DOWN ) PORT_8WAY PORT_COCKTAIL
    PORT_BIT( 0x80, IP_ACTIVE_LOW, IPT_JOYSTICKRIGHT_LEFT ) PORT_8WAY PORT_COCKTAIL
INPUT_PORTS_END


INPUT_PORTS_START( elecyoyo )
	PORT_START	/* PIA 0 Port A (PLAYER 1) */
    PORT_BIT( 0x01, IP_ACTIVE_LOW, IPT_JOYSTICK_UP ) PORT_4WAY
    PORT_BIT( 0x02, IP_ACTIVE_LOW, IPT_JOYSTICK_RIGHT ) PORT_4WAY
    PORT_BIT( 0x04, IP_ACTIVE_LOW, IPT_JOYSTICK_DOWN ) PORT_4WAY
    PORT_BIT( 0x08, IP_ACTIVE_LOW, IPT_JOYSTICK_LEFT ) PORT_4WAY
	PORT_BIT( 0x10, IP_ACTIVE_LOW, IPT_UNKNOWN )
	PORT_BIT( 0x20, IP_ACTIVE_LOW, IPT_START2 )
	PORT_BIT( 0x40, IP_ACTIVE_LOW, IPT_START1 )
	PORT_BIT( 0x80, IP_ACTIVE_LOW, IPT_UNKNOWN )

	PORT_START	/* PIA 0 Port B (COIN) */
	COIN_PORT

	PORT_START	/* PIA 1 Port A (SPARE) */
	PORT_BIT( 0xff, IP_ACTIVE_LOW, IPT_UNKNOWN )

    PORT_START	/* PIA 1 Port B (PLAYER 1/2) */
	PORT_BIT( 0xff, IP_ACTIVE_LOW, IPT_UNKNOWN )

	PORT_START	/* PIA 2 Port A (PLAYER 2) */
    PORT_BIT( 0x01, IP_ACTIVE_LOW, IPT_JOYSTICK_UP ) PORT_4WAY PORT_COCKTAIL
    PORT_BIT( 0x02, IP_ACTIVE_LOW, IPT_JOYSTICK_RIGHT ) PORT_4WAY PORT_COCKTAIL
    PORT_BIT( 0x04, IP_ACTIVE_LOW, IPT_JOYSTICK_DOWN ) PORT_4WAY PORT_COCKTAIL
    PORT_BIT( 0x08, IP_ACTIVE_LOW, IPT_JOYSTICK_LEFT ) PORT_4WAY PORT_COCKTAIL
    PORT_BIT( 0xf0, IP_ACTIVE_LOW, IPT_UNKNOWN )
INPUT_PORTS_END


INPUT_PORTS_START( kram )
	PORT_START	/* PIA 0 Port A (PLAYER 1) */
    PORT_BIT( 0x01, IP_ACTIVE_LOW, IPT_JOYSTICK_UP ) PORT_8WAY
    PORT_BIT( 0x02, IP_ACTIVE_LOW, IPT_JOYSTICK_RIGHT ) PORT_8WAY
    PORT_BIT( 0x04, IP_ACTIVE_LOW, IPT_JOYSTICK_DOWN ) PORT_8WAY
    PORT_BIT( 0x08, IP_ACTIVE_LOW, IPT_JOYSTICK_LEFT ) PORT_8WAY
	PORT_BIT( 0x10, IP_ACTIVE_LOW, IPT_BUTTON2 )
	PORT_BIT( 0x20, IP_ACTIVE_LOW, IPT_START2 )
	PORT_BIT( 0x40, IP_ACTIVE_LOW, IPT_START1 )
	PORT_BIT( 0x80, IP_ACTIVE_LOW, IPT_BUTTON1 )

	PORT_START	/* PIA 0 Port B (COIN) */
	COIN_PORT

	PORT_START	/* PIA 1 Port A (SPARE) */
	PORT_BIT( 0xff, IP_ACTIVE_LOW, IPT_UNKNOWN )

    PORT_START	/* PIA 1 Port B (PLAYER 1/2) */
	PORT_BIT( 0xff, IP_ACTIVE_LOW, IPT_UNKNOWN )

	PORT_START	/* PIA 2 Port A (PLAYER 2) */
    PORT_BIT( 0x01, IP_ACTIVE_LOW, IPT_JOYSTICK_UP ) PORT_8WAY PORT_COCKTAIL
    PORT_BIT( 0x02, IP_ACTIVE_LOW, IPT_JOYSTICK_RIGHT ) PORT_8WAY PORT_COCKTAIL
    PORT_BIT( 0x04, IP_ACTIVE_LOW, IPT_JOYSTICK_DOWN ) PORT_8WAY PORT_COCKTAIL
    PORT_BIT( 0x08, IP_ACTIVE_LOW, IPT_JOYSTICK_LEFT ) PORT_8WAY PORT_COCKTAIL
	PORT_BIT( 0x10, IP_ACTIVE_LOW, IPT_BUTTON2 ) PORT_COCKTAIL
    PORT_BIT( 0x60, IP_ACTIVE_LOW, IPT_UNKNOWN )
	PORT_BIT( 0x80, IP_ACTIVE_LOW, IPT_BUTTON1 ) PORT_COCKTAIL
INPUT_PORTS_END


INPUT_PORTS_START( zookeep )
	PORT_START	/* PIA 0 Port A (PLAYER 1) */
	PORT_BIT( 0x01, IP_ACTIVE_LOW, IPT_JOYSTICK_UP ) PORT_4WAY
	PORT_BIT( 0x02, IP_ACTIVE_LOW, IPT_JOYSTICK_RIGHT ) PORT_4WAY
	PORT_BIT( 0x04, IP_ACTIVE_LOW, IPT_JOYSTICK_DOWN ) PORT_4WAY
	PORT_BIT( 0x08, IP_ACTIVE_LOW, IPT_JOYSTICK_LEFT ) PORT_4WAY
	PORT_BIT( 0x10, IP_ACTIVE_LOW, IPT_START1 )
	PORT_BIT( 0x20, IP_ACTIVE_LOW, IPT_START2 )
	PORT_BIT( 0x40, IP_ACTIVE_LOW, IPT_BUTTON1 )
	PORT_BIT( 0x80, IP_ACTIVE_LOW, IPT_UNKNOWN )

	PORT_START	/* PIA 0 Port B (COIN) */
	COIN_PORT

	PORT_START	/* PIA 1 Port A (SPARE) */
	PORT_BIT( 0xff, IP_ACTIVE_LOW, IPT_UNKNOWN )

    PORT_START	/* PIA 1 Port B (PLAYER 1/2) */
	PORT_BIT( 0xff, IP_ACTIVE_LOW, IPT_UNKNOWN )

	PORT_START	/* PIA 2 Port A (PLAYER 2) */
	PORT_BIT( 0x01, IP_ACTIVE_LOW, IPT_JOYSTICK_UP ) PORT_4WAY PORT_COCKTAIL
	PORT_BIT( 0x02, IP_ACTIVE_LOW, IPT_JOYSTICK_RIGHT ) PORT_4WAY PORT_COCKTAIL
	PORT_BIT( 0x04, IP_ACTIVE_LOW, IPT_JOYSTICK_DOWN ) PORT_4WAY PORT_COCKTAIL
	PORT_BIT( 0x08, IP_ACTIVE_LOW, IPT_JOYSTICK_LEFT ) PORT_4WAY PORT_COCKTAIL
	PORT_BIT( 0x30, IP_ACTIVE_LOW, IPT_UNKNOWN )
	PORT_BIT( 0x40, IP_ACTIVE_LOW, IPT_BUTTON1 ) PORT_COCKTAIL
	PORT_BIT( 0x80, IP_ACTIVE_LOW, IPT_UNKNOWN )
INPUT_PORTS_END


INPUT_PORTS_START( slither )
	PORT_START	/* PIA 0 Port A (PLAYER 1) */
    PORT_BIT( 0x0f, IP_ACTIVE_LOW, IPT_UNUSED )
	PORT_BIT( 0x10, IP_ACTIVE_LOW, IPT_BUTTON2 )
	PORT_BIT( 0x20, IP_ACTIVE_LOW, IPT_START2 )
	PORT_BIT( 0x40, IP_ACTIVE_LOW, IPT_START1 )
	PORT_BIT( 0x80, IP_ACTIVE_LOW, IPT_BUTTON1 )

	PORT_START	/* PIA 0 Port B (COIN) */
	PORT_SERVICE(0x01, IP_ACTIVE_LOW )
    PORT_BIT( 0x0e, IP_ACTIVE_LOW, IPT_UNUSED )
	PORT_BIT( 0x10, IP_ACTIVE_LOW, IPT_COIN1 )
	PORT_BIT( 0x20, IP_ACTIVE_LOW, IPT_COIN2 )
	PORT_BIT( 0x40, IP_ACTIVE_LOW, IPT_COIN3 )
	PORT_BIT( 0x80, IP_ACTIVE_LOW, IPT_TILT )

	PORT_START	/* PIA 3 Port A (PLAYER 2) */
    PORT_BIT( 0x0f, IP_ACTIVE_LOW, IPT_UNUSED )
	PORT_BIT( 0x10, IP_ACTIVE_LOW, IPT_BUTTON2 ) PORT_COCKTAIL
	PORT_BIT( 0x20, IP_ACTIVE_LOW, IPT_START2 ) PORT_COCKTAIL
	PORT_BIT( 0x40, IP_ACTIVE_LOW, IPT_START1 ) PORT_COCKTAIL
	PORT_BIT( 0x80, IP_ACTIVE_LOW, IPT_BUTTON1 ) PORT_COCKTAIL

	PORT_START	/* PIA 1 Port A (TRACKBALL L/R) */
    PORT_BIT( 0xff, 0, IPT_TRACKBALL_X ) PORT_MINMAX(0x00,0xff) PORT_SENSITIVITY(50) PORT_KEYDELTA(10)

	PORT_START	/* PIA 2 Port A (TRACKBALL U/D) */
    PORT_BIT( 0xff, 0, IPT_TRACKBALL_Y ) PORT_MINMAX(0x00,0xff) PORT_SENSITIVITY(50) PORT_KEYDELTA(10) PORT_REVERSE

	PORT_START	/* PIA 1 Port A (TRACKBALL L/R) */
    PORT_BIT( 0xff, 0, IPT_TRACKBALL_X ) PORT_MINMAX(0x00,0xff) PORT_SENSITIVITY(50) PORT_KEYDELTA(10) PORT_COCKTAIL

	PORT_START	/* PIA 2 Port A (TRACKBALL U/D) */
    PORT_BIT( 0xff, 0, IPT_TRACKBALL_Y ) PORT_MINMAX(0x00,0xff) PORT_SENSITIVITY(50) PORT_KEYDELTA(10) PORT_REVERSE PORT_COCKTAIL
INPUT_PORTS_END


INPUT_PORTS_START( complexx )
	PORT_START	/* PIA 0 Port A (Left Stick) */
    PORT_BIT( 0x01, IP_ACTIVE_LOW, IPT_JOYSTICKLEFT_UP ) PORT_8WAY
    PORT_BIT( 0x02, IP_ACTIVE_LOW, IPT_JOYSTICKLEFT_RIGHT ) PORT_8WAY
    PORT_BIT( 0x04, IP_ACTIVE_LOW, IPT_JOYSTICKLEFT_DOWN ) PORT_8WAY
    PORT_BIT( 0x08, IP_ACTIVE_LOW, IPT_JOYSTICKLEFT_LEFT ) PORT_8WAY
	PORT_BIT( 0x10, IP_ACTIVE_LOW, IPT_BUTTON1 )
	PORT_BIT( 0x20, IP_ACTIVE_LOW, IPT_START2 )
	PORT_BIT( 0x40, IP_ACTIVE_LOW, IPT_START1 )
	PORT_BIT( 0x80, IP_ACTIVE_LOW, IPT_UNUSED )

/* 0x80 isn't scanned unless the difficulty in the game is below 4. I
decided not to map this button, so if you set the difficulty that low,
you can't fire. If I do map that button, and you set difficulty below 4
and hit this fire button without pressing the second stick in any
direction it fires up. If you're pressing left it fires right, press
right it fires left and press down it fires down. That's just too wacky.
I gotta think it's not supposed to be hooked up at all and if the
difficulty is that low, you just can't shoot, so I think it should stay
as-is. Tim Lindquist 1-17-03 */

	PORT_START	/* PIA 0 Port B (COIN) */
	COIN_PORT

	PORT_START	/* PIA 1 Port A (SPARE) */
	PORT_BIT( 0xff, IP_ACTIVE_LOW, IPT_UNKNOWN )

    PORT_START	/* PIA 1 Port B (PLAYER 1/2) */
	PORT_BIT( 0xff, IP_ACTIVE_LOW, IPT_UNKNOWN )

	PORT_START	/* PIA 2 Port A (Right Stick) */
    PORT_BIT( 0x01, IP_ACTIVE_LOW, IPT_UNUSED )
    PORT_BIT( 0x02, IP_ACTIVE_LOW, IPT_UNUSED )
    PORT_BIT( 0x04, IP_ACTIVE_LOW, IPT_UNUSED )
    PORT_BIT( 0x08, IP_ACTIVE_LOW, IPT_UNUSED )
    PORT_BIT( 0x10, IP_ACTIVE_LOW, IPT_JOYSTICKRIGHT_UP ) PORT_8WAY
    PORT_BIT( 0x20, IP_ACTIVE_LOW, IPT_JOYSTICKRIGHT_RIGHT ) PORT_8WAY
    PORT_BIT( 0x40, IP_ACTIVE_LOW, IPT_JOYSTICKRIGHT_DOWN ) PORT_8WAY
    PORT_BIT( 0x80, IP_ACTIVE_LOW, IPT_JOYSTICKRIGHT_LEFT ) PORT_8WAY
INPUT_PORTS_END



/*************************************
 *
 *	Sound interfaces
 *
 *************************************/

static struct DACinterface dac_interface =
{
	1,
	{ 100 }
};


static struct SN76496interface sn76496_interface =
{
	2,
	{ SLITHER_CLOCK_OSC/4/4, SLITHER_CLOCK_OSC/4/4 },
	{ 50, 50 }
};



/*************************************
 *
 *	Machine drivers
 *
 *************************************/

static MACHINE_DRIVER_START( qix )

	/* basic machine hardware */
	MDRV_CPU_ADD_TAG("main", M6809, MAIN_CLOCK_OSC/4/4)	/* 1.25 MHz */
	MDRV_CPU_PROGRAM_MAP(readmem_data,writemem_data)
	MDRV_CPU_VBLANK_INT(qix_vblank_start,1)

	MDRV_CPU_ADD_TAG("video", M6809, MAIN_CLOCK_OSC/4/4)	/* 1.25 MHz */
	MDRV_CPU_PROGRAM_MAP(readmem_video,writemem_video)

	MDRV_CPU_ADD_TAG("sound", M6802, SOUND_CLOCK_OSC/2/4)	/* 0.92 MHz */
	MDRV_CPU_FLAGS(CPU_AUDIO_CPU)
	MDRV_CPU_PROGRAM_MAP(readmem_sound,writemem_sound)

	MDRV_FRAMES_PER_SECOND(60)
	MDRV_VBLANK_DURATION(DEFAULT_REAL_60HZ_VBLANK_DURATION)

	MDRV_MACHINE_INIT(qix)
	MDRV_NVRAM_HANDLER(generic_0fill)

	/* video hardware */
	MDRV_VIDEO_ATTRIBUTES(VIDEO_TYPE_RASTER)
	MDRV_SCREEN_SIZE(256, 256)
	MDRV_VISIBLE_AREA(0, 255, 8, 247)
	MDRV_PALETTE_LENGTH(1024)

	MDRV_VIDEO_START(qix)
	MDRV_VIDEO_UPDATE(qix)

	/* sound hardware */
	MDRV_SOUND_ADD_TAG("dac", DAC, dac_interface)
MACHINE_DRIVER_END


static MACHINE_DRIVER_START( mcu )

	/* basic machine hardware */
	MDRV_IMPORT_FROM(qix)

	MDRV_CPU_ADD(M68705, COIN_CLOCK_OSC/4)	/* 1.00 MHz */
	MDRV_CPU_PROGRAM_MAP(mcu_readmem,mcu_writemem)

	MDRV_MACHINE_INIT(qixmcu)
MACHINE_DRIVER_END


static MACHINE_DRIVER_START( zookeep )

	/* basic machine hardware */
	MDRV_IMPORT_FROM(mcu)

	MDRV_CPU_MODIFY("main")
	MDRV_CPU_PROGRAM_MAP(zoo_readmem_data,zoo_writemem_data)

	MDRV_CPU_MODIFY("video")
	MDRV_CPU_PROGRAM_MAP(zoo_readmem_video,zoo_writemem_video)
MACHINE_DRIVER_END


static MACHINE_DRIVER_START( slither )

	/* basic machine hardware */
	MDRV_IMPORT_FROM(qix)

	MDRV_CPU_REPLACE("main", M6809, SLITHER_CLOCK_OSC/4/4)	/* 1.34 MHz */
	MDRV_CPU_REPLACE("video", M6809, SLITHER_CLOCK_OSC/4/4)	/* 1.34 MHz */
	MDRV_CPU_REMOVE("sound")

	MDRV_MACHINE_INIT(slither)

	/* video hardware */
	MDRV_VISIBLE_AREA(0, 255, 0, 255)

	/* sound hardware */
	MDRV_SOUND_REPLACE("dac", SN76496, sn76496_interface)
MACHINE_DRIVER_END



/*************************************
 *
 *	ROM definitions
 *
 *************************************/

ROM_START( qix )
	ROM_REGION( 0x10000, REGION_CPU1, 0 )
	ROM_LOAD( "u12", 0xc000, 0x0800, CRC(aad35508) SHA1(5fa72e00b4373de21e27a86b49a44a9769f769f4) )
	ROM_LOAD( "u13", 0xc800, 0x0800, CRC(46c13504) SHA1(19c084c38b75f14bf5094b317afeecaca6870f7a) )
	ROM_LOAD( "u14", 0xd000, 0x0800, CRC(5115e896) SHA1(8359a1700fff7a38e8ea4f92a4f18bc628cf1cb1) )
	ROM_LOAD( "u15", 0xd800, 0x0800, CRC(ccd52a1b) SHA1(86d134cd769ef12820638b96a4ffedd8b15dffd2) )
	ROM_LOAD( "u16", 0xe000, 0x0800, CRC(cd1c36ee) SHA1(b379b1fe3109947a12c9683cd0c2400c2ee845b3) )
	ROM_LOAD( "u17", 0xe800, 0x0800, CRC(1acb682d) SHA1(a2c60964e8d838d09662f8a670c6da41ba850df9) )
	ROM_LOAD( "u18", 0xf000, 0x0800, CRC(de77728b) SHA1(8e183bb27858aad9a996e4a2e5a95f0145d1f5b4) )
	ROM_LOAD( "u19", 0xf800, 0x0800, CRC(c0994776) SHA1(9452a98c78a038679c4e58f4a9983adb28ea5e78) )

	ROM_REGION( 0x10000, REGION_CPU2, 0 )
	ROM_LOAD( "u4",  0xc800, 0x0800, CRC(5b906a09) SHA1(84a2e817d6718e0276fcea702811a91bc054a670) )
	ROM_LOAD( "u5",  0xd000, 0x0800, CRC(254a3587) SHA1(66045c71cc1d04d4e03c728e578f570fbf7c650d) )
	ROM_LOAD( "u6",  0xd800, 0x0800, CRC(ace30389) SHA1(50c6275d13cfbca7750d5a3e725faedba7574e04) )
	ROM_LOAD( "u7",  0xe000, 0x0800, CRC(8ebcfa7c) SHA1(21ccf5e74424ab5470473d1059ee6a43d144f685) )
	ROM_LOAD( "u8",  0xe800, 0x0800, CRC(b8a3c8f9) SHA1(32ba771913ef44b1133ecfaedaae7f96dcc84343) )
	ROM_LOAD( "u9",  0xf000, 0x0800, CRC(26cbcd55) SHA1(2e55e222f850548cd1d461ab5337e98dd817b567) )
	ROM_LOAD( "u10", 0xf800, 0x0800, CRC(568be942) SHA1(8b6a01d983d355a64372fa76af810ab53e09d5df) )

	ROM_REGION( 0x10000, REGION_CPU3, 0 )
	ROM_LOAD( "u27", 0xf800, 0x0800, CRC(f3782bd0) SHA1(bfc6d29f9668e02857453e96c005c81568ae931d) )
ROM_END


ROM_START( qixa )
	ROM_REGION( 0x10000, REGION_CPU1, 0 )
	ROM_LOAD( "qu12", 0xc000, 0x0800, CRC(1c55b44d) SHA1(6385e5e484e24cf396c14de86344170639c3cc65) )
	ROM_LOAD( "qu13", 0xc800, 0x0800, CRC(20279e8c) SHA1(722da239636de3fe40318768ddbe687b19afcdb6) )
	ROM_LOAD( "qu14", 0xd000, 0x0800, CRC(bafe3ce3) SHA1(648a54545a1b545c82c0ace5eb1ce17af5ea7391) )
	/* d800-dfff empty */
	ROM_LOAD( "qu16", 0xe000, 0x0800, CRC(db560753) SHA1(4acbe17f1e555f45606ddec197c5ab691ff46d39) )
	ROM_LOAD( "qu17", 0xe800, 0x0800, CRC(8c7aeed8) SHA1(b7da2b0f34f72f9853cdf6ce55e604b09fcf4728) )
	ROM_LOAD( "qu18", 0xf000, 0x0800, CRC(353be980) SHA1(a50e02fcc69771a13b238aa0e8dc3c56b01a58d5) )
	ROM_LOAD( "qu19", 0xf800, 0x0800, CRC(f46a69ca) SHA1(dacb53c0318445da3fbb86f9a45914c5b7a4c4a1) )

	ROM_REGION( 0x10000, REGION_CPU2, 0 )
	ROM_LOAD( "qu3",  0xc000, 0x0800, CRC(8b4c0ef0) SHA1(6d18d1052f342e3b3313f2174b20f2a179e2c6bd) )
	ROM_LOAD( "qu4",  0xc800, 0x0800, CRC(66a5c260) SHA1(8cce71bcd3a432650f0d0c94f3a2151ba8154220) )
	ROM_LOAD( "qu5",  0xd000, 0x0800, CRC(70160ea3) SHA1(a411130c5c669a181564369a8921b26e0f0b5450) )
	/* d800-dfff empty */
	ROM_LOAD( "qu7",  0xe000, 0x0800, CRC(d6733019) SHA1(89e9e63c91e044fe1c6ce883e3ec18eec0cb39d3) )
	ROM_LOAD( "qu8",  0xe800, 0x0800, CRC(66870dcc) SHA1(9f926390f5ce86d7c1bf55b75dbfb34119425c46) )
	ROM_LOAD( "qu9",  0xf000, 0x0800, CRC(c99bf94d) SHA1(7b6fa6e1cf0f131909d44694c261b1cc2de65003) )
	ROM_LOAD( "qu10", 0xf800, 0x0800, CRC(88b45037) SHA1(e2e5fefe377def3f784026b921527898af8b83a9) )

	ROM_REGION( 0x10000, REGION_CPU3, 0 )
	ROM_LOAD( "u27",  0xf800, 0x0800, CRC(f3782bd0) SHA1(bfc6d29f9668e02857453e96c005c81568ae931d) )
ROM_END


ROM_START( qixb )
	ROM_REGION( 0x10000, REGION_CPU1, 0 )
	ROM_LOAD( "lk14.bin", 0xc000, 0x1000, CRC(6d164986) SHA1(c805abe1a441e10080ceca8ba547835bafb61bcc) )
	ROM_LOAD( "lk15.bin", 0xd000, 0x1000, CRC(16c6ce0f) SHA1(b8091d2db476d2acb4b3f0789e1f155336be9b39) )
	ROM_LOAD( "lk16.bin", 0xe000, 0x1000, CRC(698b1f9c) SHA1(7e7637ca5985f072e821e16f8b65aedb87df136b) )
	ROM_LOAD( "lk17.bin", 0xf000, 0x1000, CRC(7e3adde6) SHA1(dfe66317f87e10919f1ea4b4d565703e73039821) )

	ROM_REGION( 0x10000, REGION_CPU2, 0 )
	ROM_LOAD( "lk10.bin", 0xc000, 0x1000, CRC(7eac67d0) SHA1(ca5938422aaa1e380af0afa505876d4682ac69b9) )
	ROM_LOAD( "lk11.bin", 0xd000, 0x1000, CRC(90ccbb6a) SHA1(b65592384597dc2aafc02f49b6b6f477c9112580) )
	ROM_LOAD( "lk12.bin", 0xe000, 0x1000, CRC(be9b9f7d) SHA1(e681bdb9aa8b8c31af1c14e23d0f420577d6db63) )
	ROM_LOAD( "lk13.bin", 0xf000, 0x1000, CRC(51c9853b) SHA1(29a5221f2af866d2ee73110409ecddc2c96404fd) )

	ROM_REGION( 0x10000, REGION_CPU3, 0 )
	ROM_LOAD( "u27",      0xf800, 0x0800, CRC(f3782bd0) SHA1(bfc6d29f9668e02857453e96c005c81568ae931d) )
ROM_END


ROM_START( qix2 )
	ROM_REGION( 0x10000, REGION_CPU1, 0 )
	ROM_LOAD( "u12.rmb", 0xc000, 0x0800, CRC(484280fd) SHA1(a60c1a278e519721294b2486dc817d248d19c3be) )
	ROM_LOAD( "u13.rmb", 0xc800, 0x0800, CRC(3d089fcb) SHA1(f4f31134c9c15160d2d15cb41296dfec6f2dfe37) )
	ROM_LOAD( "u14.rmb", 0xd000, 0x0800, CRC(362123a9) SHA1(3e2a853f6960f2d5fdcdef8dec8ccf5aad449548) )
	ROM_LOAD( "u15.rmb", 0xd800, 0x0800, CRC(60f3913d) SHA1(a97b658fe2c58b00c2749072828b2e0032894915) )
	ROM_LOAD( "u16.rmb", 0xe000, 0x0800, CRC(cc139e34) SHA1(0ed3e7179b0cbaa31fa91e1ed862b86f5032919a) )
	ROM_LOAD( "u17.rmb", 0xe800, 0x0800, CRC(cf31dc49) SHA1(71c089d827ab61ba69e5e95b7e53220763786df9) )
	ROM_LOAD( "u18.rmb", 0xf000, 0x0800, CRC(1f91ed7a) SHA1(85bb5370a244719663a4f859f66860613aa2b86e) )
	ROM_LOAD( "u19.rmb", 0xf800, 0x0800, CRC(68e8d5a6) SHA1(d09252c393be2fdaf3b9b9f477c79f721d15943f) )

	ROM_REGION( 0x10000, REGION_CPU2, 0 )
	ROM_LOAD( "u3.rmb",  0xc000, 0x0800, CRC(19cebaca) SHA1(7d7e79ab0920952cf7618567c9c65397535b6d4f) )
	ROM_LOAD( "u4.rmb",  0xc800, 0x0800, CRC(6cfb4185) SHA1(6545dece8eaeb716877aa6e7b24c21f6e5991451) )
	ROM_LOAD( "u5.rmb",  0xd000, 0x0800, CRC(948f53f3) SHA1(db6eddec8ba41335316d80b6f97e932bf91139af) )
	ROM_LOAD( "u6.rmb",  0xd800, 0x0800, CRC(8630120e) SHA1(14a020fd1bff4acbb034883e33130adda85884e5) )
	ROM_LOAD( "u7.rmb",  0xe000, 0x0800, CRC(bad037c9) SHA1(17218c31895b1547b71d2d9d2b6a93d2e5d73bdd) )
	ROM_LOAD( "u8.rmb",  0xe800, 0x0800, CRC(3159bc00) SHA1(479a69bfe5af48d5ce63978265ce59f79c25749f) )
	ROM_LOAD( "u9.rmb",  0xf000, 0x0800, CRC(e80e9b1d) SHA1(66ef22a26df3f766ae813213473b9ac4b35b01f6) )
	ROM_LOAD( "u10.rmb", 0xf800, 0x0800, CRC(9a55d360) SHA1(fc5f8c853dcc573f6b36dbdd63e5d1edba88bce1) )

	ROM_REGION( 0x10000, REGION_CPU3, 0 )
	ROM_LOAD( "u27",     0xf800, 0x0800, CRC(f3782bd0) SHA1(bfc6d29f9668e02857453e96c005c81568ae931d) )
ROM_END


ROM_START( sdungeon )
	ROM_REGION( 0x10000, REGION_CPU1, 0 )
	ROM_LOAD( "sd14.u14", 0xa000, 0x1000, CRC(7024b55a) SHA1(1e21cb2a9cba8c0a3684e137ff78d4b331d86922) )
	ROM_LOAD( "sd15.u15", 0xb000, 0x1000, CRC(a3ac9040) SHA1(f033c21983e87688884180c2336d766a0fa49765) )
	ROM_LOAD( "sd16.u16", 0xc000, 0x1000, CRC(cc20b580) SHA1(53e34405f1f39bce305f199d09d3a32c10c0c616) )
	ROM_LOAD( "sd17.u17", 0xd000, 0x1000, CRC(4663e4b8) SHA1(519b0e730db2047ebe2cce0953eda53e851f8fac) )
	ROM_LOAD( "sd18.u18", 0xe000, 0x1000, CRC(7ef1ffc0) SHA1(eca49a916b6b51b91ed45ff89bb37a67fee7db0e) )
	ROM_LOAD( "sd19.u19", 0xf000, 0x1000, CRC(7b20b7ac) SHA1(554e29adc75cc465ea603a628f9d60d6d903e7ab) )

	ROM_REGION( 0x10000, REGION_CPU2, 0 )
	ROM_LOAD( "sd05.u5",  0xa000, 0x1000, CRC(0b2bf48e) SHA1(51bfb35521864f09a20b38aeeb98ab1399d139a5) )
	ROM_LOAD( "sd06.u6",  0xb000, 0x1000, CRC(f86db512) SHA1(7e7ae64db7821f18a5eefdcc7a2e1abc37abd3ac) )
	ROM_LOAD( "sd07.u7",  0xc000, 0x1000, CRC(7b796831) SHA1(cd3d8975e99886f51b27530f0e261f749aadee73) )
	ROM_LOAD( "sd08.u8",  0xd000, 0x1000, CRC(5fbe7068) SHA1(2884909156179b81181b908431de15a7c586f619) )
	ROM_LOAD( "sd09.u9",  0xe000, 0x1000, CRC(89bc51ea) SHA1(57ad4806f0b39af3b3cde91cef16234fc82f21d0) )
	ROM_LOAD( "sd10.u10", 0xf000, 0x1000, CRC(754de734) SHA1(a37b8362a592f5d1c0aeaa525fc003f6060fc12b) )

	ROM_REGION( 0x10000, REGION_CPU3, 0 )
	ROM_LOAD( "sd26.u26", 0xf000, 0x0800, CRC(3df8630d) SHA1(5f87c4d49799d424ce36469bf2b36b14c782fcd8) )
	ROM_LOAD( "sd27.u27", 0xf800, 0x0800, CRC(0386f351) SHA1(24ba6aba7c62c313397d743d18093c646f4a6526) )

	ROM_REGION( 0x0800, REGION_CPU4, 0 )
	ROM_LOAD( "sd101",    0x0000, 0x0800, CRC(e255af9a) SHA1(2410d3b7dec8e72a93d71c824c9403a6d96b9e8c) )
ROM_END


ROM_START( elecyoyo )
	ROM_REGION( 0x10000, REGION_CPU1, 0 )
    ROM_LOAD( "yy14",   0xa000, 0x1000, CRC(0d2edcb9) SHA1(36e1a1aa81111f38e1c06a8174e7de406478cc67) )
    ROM_LOAD( "yy15",   0xb000, 0x1000, CRC(a91f01e3) SHA1(7818299d25a0816b856e83fae02d8019e5e8b4a3) )
    ROM_LOAD( "yy16-1", 0xc000, 0x1000, CRC(2710f360) SHA1(4a6210b07618fba261c38b7bf9a779598dd6bb3c) )
    ROM_LOAD( "yy17",   0xd000, 0x1000, CRC(25fd489d) SHA1(ca7b8d1bcbc223fc0706c8f2e9f02821519b75e4) )
    ROM_LOAD( "yy18",   0xe000, 0x1000, CRC(0b6661c0) SHA1(3e24acbfea1a3b83223d780ea34c83759a751175) )
    ROM_LOAD( "yy19-1", 0xf000, 0x1000, CRC(95b8b244) SHA1(2852dcfc6a638118ee7fe60b957d8aa9a5a8984c) )

	ROM_REGION( 0x10000, REGION_CPU2, 0 )
    ROM_LOAD( "yy5",    0xa000, 0x1000, CRC(3793fec5) SHA1(d31f3ba6364755c98beb814b5e7d5541e8b4e1a0) )
    ROM_LOAD( "yy6",    0xb000, 0x1000, CRC(2e8b1265) SHA1(6fb67b6a5b627bcbd10c72ac8c9e4d9f4dd7860f) )
    ROM_LOAD( "yy7",    0xc000, 0x1000, CRC(20f93411) SHA1(126c27442a4e35d054a236c41930603241d08ccf) )
    ROM_LOAD( "yy8",    0xd000, 0x1000, CRC(926f90c8) SHA1(dbdfc0e9184a4d9d44d04fdb8fed19b35a8edadc) )
    ROM_LOAD( "yy9",    0xe000, 0x1000, CRC(2f999480) SHA1(582baf285cadf8431ff80f5b63a02fbbefb62e45) )
    ROM_LOAD( "yy10",   0xf000, 0x1000, CRC(b31d20e2) SHA1(e2ba4e6b81a02c3c02353774eb63a72cdfe5f2c3) )

	ROM_REGION( 0x10000, REGION_CPU3, 0 )
    ROM_LOAD( "yy27",   0xf800, 0x0800, CRC(5a2aa0f3) SHA1(16c7c2db39f33ea3506e07312352ccbfe5528fbd) )

	ROM_REGION( 0x0800, REGION_CPU4, 0 )
	ROM_LOAD( "yy101",  0x0000, 0x0800, CRC(3cf13038) SHA1(29c1b309d9046087a50ca78a5f7cba694271af26) )
ROM_END


ROM_START( elecyoy2 )
	ROM_REGION( 0x10000, REGION_CPU1, 0 )
    ROM_LOAD( "yy14",  0xa000, 0x1000, CRC(0d2edcb9) SHA1(36e1a1aa81111f38e1c06a8174e7de406478cc67) )
    ROM_LOAD( "yy15",  0xb000, 0x1000, CRC(a91f01e3) SHA1(7818299d25a0816b856e83fae02d8019e5e8b4a3) )
    ROM_LOAD( "yy16",  0xc000, 0x1000, CRC(cab19f3a) SHA1(df41649a800b77f046edb6623f65d100f6a8ef5f) )
    ROM_LOAD( "yy17",  0xd000, 0x1000, CRC(25fd489d) SHA1(ca7b8d1bcbc223fc0706c8f2e9f02821519b75e4) )
    ROM_LOAD( "yy18",  0xe000, 0x1000, CRC(0b6661c0) SHA1(3e24acbfea1a3b83223d780ea34c83759a751175) )
    ROM_LOAD( "yy19",  0xf000, 0x1000, CRC(d0215d2e) SHA1(816131bc272252df6cea1b84a42750a71ce4f427) )

	ROM_REGION( 0x10000, REGION_CPU2, 0 )
    ROM_LOAD( "yy5",   0xa000, 0x1000, CRC(3793fec5) SHA1(d31f3ba6364755c98beb814b5e7d5541e8b4e1a0) )
    ROM_LOAD( "yy6",   0xb000, 0x1000, CRC(2e8b1265) SHA1(6fb67b6a5b627bcbd10c72ac8c9e4d9f4dd7860f) )
    ROM_LOAD( "yy7",   0xc000, 0x1000, CRC(20f93411) SHA1(126c27442a4e35d054a236c41930603241d08ccf) )
    ROM_LOAD( "yy8",   0xd000, 0x1000, CRC(926f90c8) SHA1(dbdfc0e9184a4d9d44d04fdb8fed19b35a8edadc) )
    ROM_LOAD( "yy9",   0xe000, 0x1000, CRC(2f999480) SHA1(582baf285cadf8431ff80f5b63a02fbbefb62e45) )
    ROM_LOAD( "yy10",  0xf000, 0x1000, CRC(b31d20e2) SHA1(e2ba4e6b81a02c3c02353774eb63a72cdfe5f2c3) )

	ROM_REGION( 0x10000, REGION_CPU3, 0 )
    ROM_LOAD( "yy27",  0xf800, 0x0800, CRC(5a2aa0f3) SHA1(16c7c2db39f33ea3506e07312352ccbfe5528fbd) )

	ROM_REGION( 0x0800, REGION_CPU4, 0 )
	ROM_LOAD( "yy101", 0x0000, 0x0800, CRC(3cf13038) SHA1(29c1b309d9046087a50ca78a5f7cba694271af26) )
ROM_END


ROM_START( kram )
	ROM_REGION( 0x10000, REGION_CPU1, 0 )
    ROM_LOAD( "ks14-1", 0xa000, 0x1000, CRC(fe69ac79) SHA1(6df0f98e6c0901c058123988bf22a6dd9f0a1fac) )
    ROM_LOAD( "ks15",   0xb000, 0x1000, CRC(4b2c175e) SHA1(4f9d4dcc78a12e994d499b182c8229d5fa63b805) )
    ROM_LOAD( "ks16",   0xc000, 0x1000, CRC(9500a05d) SHA1(18e0107111f79ba5dc6d568e3a6e7e7778955d0b) )
    ROM_LOAD( "ks17",   0xd000, 0x1000, CRC(c752a3a1) SHA1(1d03ea97b9ca6fa3d4c43ac867ab737439d987af) )
    ROM_LOAD( "ks18",   0xe000, 0x1000, CRC(79158b03) SHA1(0d4873471b5b7ace0de8ec421ff3d74650790f7e) )
    ROM_LOAD( "ks19-1", 0xf000, 0x1000, CRC(759ea6ce) SHA1(7962f713dd93c73475fa1f64635d8e965336484b) )

	ROM_REGION( 0x10000, REGION_CPU2, 0 )
    ROM_LOAD( "ks5",    0xa000, 0x1000, CRC(1c472080) SHA1(a85400be562ef6b817f8a654f29d966d3a198ab4) )
    ROM_LOAD( "ks6",    0xb000, 0x1000, CRC(b8926622) SHA1(e25a8b2ff192f6ab0328fd7b3c58d638342f79e2) )
    ROM_LOAD( "ks7",    0xc000, 0x1000, CRC(c98a7485) SHA1(e310d53ae65d456e12a2475e9ac578592b0e82ba) )
    ROM_LOAD( "ks8",    0xd000, 0x1000, CRC(1127c4e4) SHA1(d78a8a964aac29fb71a55acf7956355724a234eb) )
    ROM_LOAD( "ks9",    0xe000, 0x1000, CRC(d3bc8b5e) SHA1(2c5b882c54bdb48f9a76abfe734b99390b89d76c) )
    ROM_LOAD( "ks10",   0xf000, 0x1000, CRC(e0426444) SHA1(3c4ea66f8ee907452e67aeb7cfaa15bee02b004b) )

	ROM_REGION( 0x10000, REGION_CPU3, 0 )
    ROM_LOAD( "ks27",   0xf800, 0x0800, CRC(c46530c8) SHA1(d2df3f2228a5cff7d7b04b5bbbc4820d2fe84d8d) )

	ROM_REGION( 0x0800, REGION_CPU4, 0 )
	ROM_LOAD( "ks101.dat", 0x0000, 0x0800, CRC(e53d97b7) SHA1(acfc3a5c5e73bd2d37e04ac357043b708d1982de) )
ROM_END


ROM_START( kram2 )
	ROM_REGION( 0x10000, REGION_CPU1, 0 )
    ROM_LOAD( "ks14", 0xa000, 0x1000, CRC(a2eac1ff) SHA1(128f83b1760492cbb272828ad8c67ea9a5db862a) )
    ROM_LOAD( "ks15", 0xb000, 0x1000, CRC(4b2c175e) SHA1(4f9d4dcc78a12e994d499b182c8229d5fa63b805) )
    ROM_LOAD( "ks16", 0xc000, 0x1000, CRC(9500a05d) SHA1(18e0107111f79ba5dc6d568e3a6e7e7778955d0b) )
    ROM_LOAD( "ks17", 0xd000, 0x1000, CRC(c752a3a1) SHA1(1d03ea97b9ca6fa3d4c43ac867ab737439d987af) )
    ROM_LOAD( "ks18", 0xe000, 0x1000, CRC(79158b03) SHA1(0d4873471b5b7ace0de8ec421ff3d74650790f7e) )
    ROM_LOAD( "ks19", 0xf000, 0x1000, CRC(053c5e09) SHA1(cd6e5b54abf73c1ccf318ca18fceb56b51a3847f) )

	ROM_REGION( 0x10000, REGION_CPU2, 0 )
    ROM_LOAD( "ks5",  0xa000, 0x1000, CRC(1c472080) SHA1(a85400be562ef6b817f8a654f29d966d3a198ab4) )
    ROM_LOAD( "ks6",  0xb000, 0x1000, CRC(b8926622) SHA1(e25a8b2ff192f6ab0328fd7b3c58d638342f79e2) )
    ROM_LOAD( "ks7",  0xc000, 0x1000, CRC(c98a7485) SHA1(e310d53ae65d456e12a2475e9ac578592b0e82ba) )
    ROM_LOAD( "ks8",  0xd000, 0x1000, CRC(1127c4e4) SHA1(d78a8a964aac29fb71a55acf7956355724a234eb) )
    ROM_LOAD( "ks9",  0xe000, 0x1000, CRC(d3bc8b5e) SHA1(2c5b882c54bdb48f9a76abfe734b99390b89d76c) )
    ROM_LOAD( "ks10", 0xf000, 0x1000, CRC(e0426444) SHA1(3c4ea66f8ee907452e67aeb7cfaa15bee02b004b) )

	ROM_REGION( 0x10000, REGION_CPU3, 0 )
    ROM_LOAD( "ks27", 0xf800, 0x0800, CRC(c46530c8) SHA1(d2df3f2228a5cff7d7b04b5bbbc4820d2fe84d8d) )

	ROM_REGION( 0x0800, REGION_CPU4, 0 )
	ROM_LOAD( "ks101.dat", 0x0000, 0x0800, CRC(e53d97b7) SHA1(acfc3a5c5e73bd2d37e04ac357043b708d1982de) )
ROM_END


ROM_START( kram3 )
	ROM_REGION( 0x10000, REGION_CPU1, 0 )
    ROM_LOAD( "kr-u14", 0xa000, 0x1000, CRC(02c1bd1e) SHA1(5f13f32ca2da0e93ed43b052c8c33af9ac67cb6c) )
    ROM_LOAD( "kr-u15", 0xb000, 0x1000, CRC(46b3ff33) SHA1(7db45971972df144a21fee4cc015b0190b502e12) )
    ROM_LOAD( "kr-u16", 0xc000, 0x1000, CRC(f202b9cf) SHA1(baf27507611c3029e2dfb1a4ff86e6fe17171246) )
    ROM_LOAD( "kr-u17", 0xd000, 0x1000, CRC(257cea23) SHA1(f9503c4a0f94d35a55033c02dda4d03737eedb90) )
    ROM_LOAD( "kr-u18", 0xe000, 0x1000, CRC(da3aed8c) SHA1(0107d58fa006a39b47513381aead760190abef35) )
    ROM_LOAD( "kr-u19", 0xf000, 0x1000, CRC(496ab571) SHA1(30e12b31ffd70a8a1ce23c845e89170ca3cabaa5) )

	ROM_REGION( 0x10000, REGION_CPU2, 0 )
    ROM_LOAD( "kr-u5",  0xa000, 0x1000, CRC(9e63c2bc) SHA1(f61a2b93ed322b62818f31fddb324c666380eff7) )
    ROM_LOAD( "kr-u6",  0xb000, 0x1000, CRC(a0ff1244) SHA1(5bc3f3f8caac0dfc8c1381d34e5b8ef5c8202982) )
    ROM_LOAD( "kr-u7",  0xc000, 0x1000, CRC(20a15024) SHA1(094951c4cd06e32af2cb2faec04c31d55ade6b7b) )
    ROM_LOAD( "ks8",    0xd000, 0x1000, CRC(1127c4e4) SHA1(d78a8a964aac29fb71a55acf7956355724a234eb) )
    ROM_LOAD( "ks9",    0xe000, 0x1000, CRC(d3bc8b5e) SHA1(2c5b882c54bdb48f9a76abfe734b99390b89d76c) )
    ROM_LOAD( "kr-u10", 0xf000, 0x1000, CRC(0a8adbd8) SHA1(8ab806108c68aa2740d9e157dd215b371e81c482) )

	ROM_REGION( 0x10000, REGION_CPU3, 0 )
    ROM_LOAD( "ks27",   0xf800, 0x0800, CRC(c46530c8) SHA1(d2df3f2228a5cff7d7b04b5bbbc4820d2fe84d8d) )
ROM_END


ROM_START( zookeep )
	ROM_REGION( 0x10000, REGION_CPU1, 0 )
	ROM_LOAD( "za12", 0x8000, 0x1000, CRC(4e40d8dc) SHA1(dd7923dcb55a2a1ae0f2029caf5a8904a9ebe8b1) )
	ROM_LOAD( "za13", 0x9000, 0x1000, CRC(eebd5248) SHA1(ebe7f8c436bfefa4236f603fbcbd38d2f4cfd2bd) )
	ROM_LOAD( "za14", 0xa000, 0x1000, CRC(fab43297) SHA1(daa5a780bc9f171da0f6db5319b1519caa09c6c9) )
	ROM_LOAD( "za15", 0xb000, 0x1000, CRC(ef8cd67c) SHA1(f3a2e12ccfa45eb77c2a6e0d9cc0601b99273fd3) )
	ROM_LOAD( "za16", 0xc000, 0x1000, CRC(ccfc15bc) SHA1(56ac6e89825fcde9cf7aeb5d0765a02f5474174a) )
	ROM_LOAD( "za17", 0xd000, 0x1000, CRC(358013f4) SHA1(dd3fae69bf460f89c71e7c78d229dd86605c5950) )
	ROM_LOAD( "za18", 0xe000, 0x1000, CRC(37886afe) SHA1(a0cc902b2d253466e21c4fbf9d3339069fe79ebe) )
	ROM_LOAD( "za19", 0xf000, 0x1000, CRC(bbfb30d9) SHA1(bc6bd5525b159bee7b08b6967cd3088b7bd10fee) )

	ROM_REGION( 0x12000, REGION_CPU2, 0 )
	ROM_LOAD( "za5",  0x0a000, 0x1000, CRC(dc0c3cbd) SHA1(8335cd91bbacc680a3a98a5242d4cb5a6f61b2b5) )
	ROM_LOAD( "za3",  0x10000, 0x1000, CRC(cc4d0aee) SHA1(05c0025e96b432088b46100051a2c780e46b7457) )
	ROM_LOAD( "za6",  0x0b000, 0x1000, CRC(27c787dd) SHA1(1142790d875573d0c39d846aba4b06946fd2bc88) )
	ROM_LOAD( "za4",  0x11000, 0x1000, CRC(ec3b10b1) SHA1(44303ab923d776052ca4d2d09a6a4315d67adc4b) )

	ROM_LOAD( "za7",  0x0c000, 0x1000, CRC(1479f480) SHA1(bd0e9eead0f1213eb3b9653c73257bc704346ab0) )
	ROM_LOAD( "za8",  0x0d000, 0x1000, CRC(4c96cdb2) SHA1(67e506462317d882c4e5c2e16318411d4958ac63) )
	ROM_LOAD( "za9",  0x0e000, 0x1000, CRC(a4f7d9e0) SHA1(a958b4d305bb397aa46d8fdab9dc7e472237ca11) )
	ROM_LOAD( "za10", 0x0f000, 0x1000, CRC(05df1a5a) SHA1(30797838c25cca038023c188cd9fa45277c4190d) )

	ROM_REGION( 0x10000, REGION_CPU3, 0 )
	ROM_LOAD( "za25", 0xd000, 0x1000, CRC(779b8558) SHA1(7312e63c23d92c9c52e93cc445a718bc8fe35a0a) )
	ROM_LOAD( "za26", 0xe000, 0x1000, CRC(60a810ce) SHA1(d97e5acea0ef1c208f8e5e95024c83dd6bc9b028) )
	ROM_LOAD( "za27", 0xf000, 0x1000, CRC(99ed424e) SHA1(e4e543dcb77f153aeb78904d11b95381d039299e) )

	ROM_REGION( 0x0800, REGION_CPU4, 0 )
	ROM_LOAD( "za_coin.bin", 0x0000, 0x0800, CRC(364d3557) SHA1(049d0759750c576187053306e07984b1e5877df7) )
ROM_END


ROM_START( zookeep2 )
	ROM_REGION( 0x10000, REGION_CPU1, 0 )
	ROM_LOAD( "za12",     0x8000, 0x1000, CRC(4e40d8dc) SHA1(dd7923dcb55a2a1ae0f2029caf5a8904a9ebe8b1) )
	ROM_LOAD( "za13",     0x9000, 0x1000, CRC(eebd5248) SHA1(ebe7f8c436bfefa4236f603fbcbd38d2f4cfd2bd) )
	ROM_LOAD( "za14",     0xa000, 0x1000, CRC(fab43297) SHA1(daa5a780bc9f171da0f6db5319b1519caa09c6c9) )
	ROM_LOAD( "za15",     0xb000, 0x1000, CRC(ef8cd67c) SHA1(f3a2e12ccfa45eb77c2a6e0d9cc0601b99273fd3) )
	ROM_LOAD( "za16",     0xc000, 0x1000, CRC(ccfc15bc) SHA1(56ac6e89825fcde9cf7aeb5d0765a02f5474174a) )
	ROM_LOAD( "za17",     0xd000, 0x1000, CRC(358013f4) SHA1(dd3fae69bf460f89c71e7c78d229dd86605c5950) )
	ROM_LOAD( "za18",     0xe000, 0x1000, CRC(37886afe) SHA1(a0cc902b2d253466e21c4fbf9d3339069fe79ebe) )
	ROM_LOAD( "za19.red", 0xf000, 0x1000, CRC(ec01760e) SHA1(169ab9d3a0abe325d960f9ed358258b3d6fcd4be) )

	ROM_REGION( 0x12000, REGION_CPU2, 0 )
	ROM_LOAD( "za5",      0x0a000, 0x1000, CRC(dc0c3cbd) SHA1(8335cd91bbacc680a3a98a5242d4cb5a6f61b2b5) )
	ROM_LOAD( "za3",      0x10000, 0x1000, CRC(cc4d0aee) SHA1(05c0025e96b432088b46100051a2c780e46b7457) )
	ROM_LOAD( "za6",      0x0b000, 0x1000, CRC(27c787dd) SHA1(1142790d875573d0c39d846aba4b06946fd2bc88) )
	ROM_LOAD( "za4",      0x11000, 0x1000, CRC(ec3b10b1) SHA1(44303ab923d776052ca4d2d09a6a4315d67adc4b) )

	ROM_LOAD( "za7",      0x0c000, 0x1000, CRC(1479f480) SHA1(bd0e9eead0f1213eb3b9653c73257bc704346ab0) )
	ROM_LOAD( "za8",      0x0d000, 0x1000, CRC(4c96cdb2) SHA1(67e506462317d882c4e5c2e16318411d4958ac63) )
	ROM_LOAD( "za9",      0x0e000, 0x1000, CRC(a4f7d9e0) SHA1(a958b4d305bb397aa46d8fdab9dc7e472237ca11) )
	ROM_LOAD( "za10",     0x0f000, 0x1000, CRC(05df1a5a) SHA1(30797838c25cca038023c188cd9fa45277c4190d) )

	ROM_REGION( 0x10000, REGION_CPU3, 0 )
	ROM_LOAD( "za25",     0xd000, 0x1000, CRC(779b8558) SHA1(7312e63c23d92c9c52e93cc445a718bc8fe35a0a) )
	ROM_LOAD( "za26",     0xe000, 0x1000, CRC(60a810ce) SHA1(d97e5acea0ef1c208f8e5e95024c83dd6bc9b028) )
	ROM_LOAD( "za27",     0xf000, 0x1000, CRC(99ed424e) SHA1(e4e543dcb77f153aeb78904d11b95381d039299e) )

	ROM_REGION( 0x0800, REGION_CPU4, 0 )
	ROM_LOAD( "za_coin.bin", 0x0000, 0x0800, CRC(364d3557) SHA1(049d0759750c576187053306e07984b1e5877df7) )
ROM_END


ROM_START( zookeep3 )
	ROM_REGION( 0x10000, REGION_CPU1, 0 )
	ROM_LOAD( "za12",    0x8000, 0x1000, CRC(4e40d8dc) SHA1(dd7923dcb55a2a1ae0f2029caf5a8904a9ebe8b1) )
	ROM_LOAD( "za13",    0x9000, 0x1000, CRC(eebd5248) SHA1(ebe7f8c436bfefa4236f603fbcbd38d2f4cfd2bd) )
	ROM_LOAD( "za14",    0xa000, 0x1000, CRC(fab43297) SHA1(daa5a780bc9f171da0f6db5319b1519caa09c6c9) )
	ROM_LOAD( "za15",    0xb000, 0x1000, CRC(ef8cd67c) SHA1(f3a2e12ccfa45eb77c2a6e0d9cc0601b99273fd3) )
	ROM_LOAD( "za16",    0xc000, 0x1000, CRC(ccfc15bc) SHA1(56ac6e89825fcde9cf7aeb5d0765a02f5474174a) )
	ROM_LOAD( "za17",    0xd000, 0x1000, CRC(358013f4) SHA1(dd3fae69bf460f89c71e7c78d229dd86605c5950) )
	ROM_LOAD( "za18",    0xe000, 0x1000, CRC(37886afe) SHA1(a0cc902b2d253466e21c4fbf9d3339069fe79ebe) )
	ROM_LOAD( "za19",    0xf000, 0x1000, CRC(bbfb30d9) SHA1(bc6bd5525b159bee7b08b6967cd3088b7bd10fee) )

	ROM_REGION( 0x12000, REGION_CPU2, 0 )
	ROM_LOAD( "za5",     0x0a000, 0x1000, CRC(dc0c3cbd) SHA1(8335cd91bbacc680a3a98a5242d4cb5a6f61b2b5) )
	ROM_LOAD( "za3",     0x10000, 0x1000, CRC(cc4d0aee) SHA1(05c0025e96b432088b46100051a2c780e46b7457) )
	ROM_LOAD( "za6",     0x0b000, 0x1000, CRC(27c787dd) SHA1(1142790d875573d0c39d846aba4b06946fd2bc88) )
	ROM_LOAD( "za4",     0x11000, 0x1000, CRC(ec3b10b1) SHA1(44303ab923d776052ca4d2d09a6a4315d67adc4b) )

	ROM_LOAD( "za7",     0x0c000, 0x1000, CRC(1479f480) SHA1(bd0e9eead0f1213eb3b9653c73257bc704346ab0) )
	ROM_LOAD( "za8",     0x0d000, 0x1000, CRC(4c96cdb2) SHA1(67e506462317d882c4e5c2e16318411d4958ac63) )
	ROM_LOAD( "zv35.9",  0x0e000, 0x1000, CRC(d14123b7) SHA1(5d35bffd2203225937bb83598ffdc31a46a1dbca) )
	ROM_LOAD( "zv36.10", 0x0f000, 0x1000, CRC(23705777) SHA1(952cd8d9ee00268bff2022b2428b1dbfab061254) )

	ROM_REGION( 0x10000, REGION_CPU3, 0 )
	ROM_LOAD( "za25",    0xd000, 0x1000, CRC(779b8558) SHA1(7312e63c23d92c9c52e93cc445a718bc8fe35a0a) )
	ROM_LOAD( "za26",    0xe000, 0x1000, CRC(60a810ce) SHA1(d97e5acea0ef1c208f8e5e95024c83dd6bc9b028) )
	ROM_LOAD( "za27",    0xf000, 0x1000, CRC(99ed424e) SHA1(e4e543dcb77f153aeb78904d11b95381d039299e) )

	ROM_REGION( 0x0800, REGION_CPU4, 0 )
	ROM_LOAD( "za_coin.bin", 0x0000, 0x0800, CRC(364d3557) SHA1(049d0759750c576187053306e07984b1e5877df7) )
ROM_END


ROM_START( slither )
	ROM_REGION( 0x10000, REGION_CPU1, 0 )
	ROM_LOAD( "u31.cpu", 0xd800, 0x0800, CRC(3dfff970) SHA1(ee50840e26aa7be226dbe9a32a8344bb75b8de07) )
	ROM_LOAD( "u30.cpu", 0xe000, 0x0800, CRC(8cbc5af8) SHA1(3d563d0bbbce007bd6db6d620e1b0996c67029f6) )
	ROM_LOAD( "u29.cpu", 0xe800, 0x0800, CRC(98c14510) SHA1(7a39b4b691883ad5d079a5c199b93986071c4a49) )
	ROM_LOAD( "u28.cpu", 0xf000, 0x0800, CRC(2762f612) SHA1(2f094832b199d8514ed04c517fca828c75ac7bfa) )
	ROM_LOAD( "u27.cpu", 0xf800, 0x0800, CRC(9306d5b1) SHA1(e5a2c613b1e083b70d63e24dd45472364930398a) )

	ROM_REGION( 0x10000, REGION_CPU2, 0 )
	ROM_LOAD( "u41.cpu", 0xd000, 0x0800, CRC(e4c60a57) SHA1(5ce9fe3d84b7a5ded372f5c3fc14a335a37ad472) )
	ROM_LOAD( "u40.cpu", 0xd800, 0x0800, CRC(5dcec622) SHA1(4c00b91106d1e505bdbd3aefb46bfb1a17f14fc1) )
	ROM_LOAD( "u39.cpu", 0xe000, 0x0800, CRC(69829c2a) SHA1(5ea1f7e6db2b2cebab8663f7a05496f2e13131f9) )
	ROM_LOAD( "u38.cpu", 0xe800, 0x0800, CRC(6adc64c6) SHA1(0502fa9f793a246ade30d506cce62c5d9b773952) )
	ROM_LOAD( "u37.cpu", 0xf000, 0x0800, CRC(55d31c96) SHA1(cf04b30369407dd9a8f0ef5850fdc13d13a7d56d) )
	ROM_LOAD( "u36.cpu", 0xf800, 0x0800, CRC(d5ffc013) SHA1(d278c0a6be86010cdf4b18afc5099dfd39f26523) )
ROM_END


ROM_START( slithera )
	ROM_REGION( 0x10000, REGION_CPU1, 0 )
	ROM_LOAD( "u31.cpu", 0xd800, 0x0800, CRC(3dfff970) SHA1(ee50840e26aa7be226dbe9a32a8344bb75b8de07) )
	ROM_LOAD( "u30.cpu", 0xe000, 0x0800, CRC(8cbc5af8) SHA1(3d563d0bbbce007bd6db6d620e1b0996c67029f6) )
	ROM_LOAD( "u29.cpu", 0xe800, 0x0800, CRC(98c14510) SHA1(7a39b4b691883ad5d079a5c199b93986071c4a49) )
	ROM_LOAD( "u28.cpu", 0xf000, 0x0800, CRC(2762f612) SHA1(2f094832b199d8514ed04c517fca828c75ac7bfa) )
	ROM_LOAD( "u27.cpu", 0xf800, 0x0800, CRC(9306d5b1) SHA1(e5a2c613b1e083b70d63e24dd45472364930398a) )

	ROM_REGION( 0x10000, REGION_CPU2, 0 )
	ROM_LOAD( "u41.cpu", 0xd000, 0x0800, CRC(e4c60a57) SHA1(5ce9fe3d84b7a5ded372f5c3fc14a335a37ad472) )
	ROM_LOAD( "u40.cpu", 0xd800, 0x0800, CRC(5dcec622) SHA1(4c00b91106d1e505bdbd3aefb46bfb1a17f14fc1) )
	ROM_LOAD( "u39.cpu", 0xe000, 0x0800, CRC(69829c2a) SHA1(5ea1f7e6db2b2cebab8663f7a05496f2e13131f9) )
	ROM_LOAD( "u38a.cpu",0xe800, 0x0800, CRC(423adfef) SHA1(f10ca9acf31e602a77b011b002bd53cebf5ba502) )
	ROM_LOAD( "u37.cpu", 0xf000, 0x0800, CRC(55d31c96) SHA1(cf04b30369407dd9a8f0ef5850fdc13d13a7d56d) )
	ROM_LOAD( "u36a.cpu",0xf800, 0x0800, CRC(5ac4e244) SHA1(077bf6f3cb98c3f7845393fb57a080d5bdfc2920) )
ROM_END

ROM_START( complexx )
	ROM_REGION( 0x10000, REGION_CPU1, 0 )
	ROM_LOAD( "cx14.bin", 0xa000, 0x1000, CRC(f123a0de) SHA1(fbb64c33d01031e9da78e725dbdaf87d6e33e23c) )
	ROM_LOAD( "cx15.bin", 0xb000, 0x1000, CRC(0fcb966f) SHA1(f7ea6f0ce356629b8133214c7b2e5ede41c54e6c) )
	ROM_LOAD( "cx16.bin", 0xc000, 0x1000, CRC(aa11e0e3) SHA1(4c0b4fc61c682d501ec3dffd7e324d4dc16425a7) )
	ROM_LOAD( "cx17.bin", 0xd000, 0x1000, CRC(f610856e) SHA1(a01edb705cf7c321800c2739beaee584bfc37270) )
	ROM_LOAD( "cx18.bin", 0xe000, 0x1000, CRC(8f8c3984) SHA1(4cfc83c7a972eeb6e386c9f663388b57c1ebfd00) )
	ROM_LOAD( "cx19.bin", 0xf000, 0x1000, CRC(13af3ba8) SHA1(79ce3dce960d89161db89821d9b211ffd1d399d7) )

	ROM_REGION( 0x10000, REGION_CPU2, 0 )
	ROM_LOAD( "cx5.bin",  0xa000, 0x1000, CRC(62a2b87b) SHA1(eeecdfd3eeba15cd93d1514132919fdc9254c1cb) )
	ROM_LOAD( "cx6.bin",  0xb000, 0x1000, CRC(dfa7c088) SHA1(626cae67db85ab8f87c59f5945032b4cb6683c8b) )
	ROM_LOAD( "cx7.bin",  0xc000, 0x1000, CRC(c8bd6759) SHA1(5e2debc2f5acf5c14da2d0c3daf49a0f63ade07b) )
	ROM_LOAD( "cx8.bin",  0xd000, 0x1000, CRC(14a57221) SHA1(21e0ac7db246cc0a23f0992e6568a9e737db725a) )
	ROM_LOAD( "cx9.bin",  0xe000, 0x1000, CRC(fc2d4a9f) SHA1(ce16cafe09e2a4411bfe3063136507e649ac7870) )
	ROM_LOAD( "cx10.bin", 0xf000, 0x1000, CRC(96e0c1ad) SHA1(4e67b46cc21b4b3e3259d34c618c6d4e4bb1ae78) )

	ROM_REGION( 0x10000, REGION_CPU3, 0 )
	ROM_LOAD( "cx26.bin",    0xe000, 0x1000, CRC(f4f19c9b) SHA1(d14ae6b59016c428a7e08862f1a4ec89f4eac4fb) )
	ROM_LOAD( "cx27.bin",    0xf000, 0x1000, CRC(7e177328) SHA1(bd3d361bb44341a01a8d118a682ad4efa19be8ff) )
ROM_END




/*************************************
 *
 *	Game-specific initialization
 *
 *************************************/

static DRIVER_INIT( kram )
{
	/* we need to override one PIA handler to prevent controls from getting disabled */
	memory_install_write8_handler(0, ADDRESS_SPACE_PROGRAM, 0x9400, 0x97ff, 0, 0, zookeep_pia_0_w);
}


static DRIVER_INIT( zookeep )
{
	/* we need to override two PIA handlers to prevent controls from getting disabled */
	memory_install_write8_handler(0, ADDRESS_SPACE_PROGRAM, 0x1400, 0x17ff, 0, 0, zookeep_pia_0_w);
	memory_install_write8_handler(0, ADDRESS_SPACE_PROGRAM, 0x1c00, 0x1fff, 0, 0, zookeep_pia_2_w);
}


static DRIVER_INIT( slither )
{
	memory_install_write8_handler(1, ADDRESS_SPACE_PROGRAM, 0x9401, 0x9401, 0, 0, slither_vram_mask_w);
}



/*************************************
 *
 *	Game drivers
 *
 *************************************/

GAME( 1981, qix,      0,        qix,     qix,      0,        ROT270, "Taito America Corporation", "Qix (set 1)" )
GAME( 1981, qixa,     qix,      qix,     qix,      0,        ROT270, "Taito America Corporation", "Qix (set 2)" )
GAME( 1981, qixb,     qix,      qix,     qix,      0,        ROT270, "Taito America Corporation", "Qix (set 3)" )
GAME( 1981, qix2,     qix,      qix,     qix,      0,        ROT270, "Taito America Corporation", "Qix II (Tournament)" )
GAME( 1981, sdungeon, 0,        mcu,     sdungeon, 0,        ROT270, "Taito America Corporation", "Space Dungeon" )
GAME( 1982, elecyoyo, 0,        mcu,     elecyoyo, 0,        ROT270, "Taito America Corporation", "The Electric Yo-Yo (set 1)" )
GAME( 1982, elecyoy2, elecyoyo, mcu,     elecyoyo, 0,        ROT270, "Taito America Corporation", "The Electric Yo-Yo (set 2)" )
GAME( 1982, kram,     0,        mcu,     kram,     kram,     ROT0,   "Taito America Corporation", "Kram (set 1)" )
GAME( 1982, kram2,    kram,     mcu,     kram,     kram,     ROT0,   "Taito America Corporation", "Kram (set 2)" )
GAMEX(1982, kram3,    kram,     qix,     kram,     kram,     ROT0,   "Taito America Corporation", "Kram (encrypted)", GAME_UNEMULATED_PROTECTION )
GAME( 1982, zookeep,  0,        zookeep, zookeep,  zookeep,  ROT0,   "Taito America Corporation", "Zoo Keeper (set 1)" )
GAME( 1982, zookeep2, zookeep,  zookeep, zookeep,  zookeep,  ROT0,   "Taito America Corporation", "Zoo Keeper (set 2)" )
GAME( 1982, zookeep3, zookeep,  zookeep, zookeep,  zookeep,  ROT0,   "Taito America Corporation", "Zoo Keeper (set 3)" )
GAME( 1982, slither,  0,        slither, slither,  slither,  ROT270, "Century II",                "Slither (set 1)" )
GAME( 1982, slithera, slither,  slither, slither,  slither,  ROT270, "Century II",                "Slither (set 2)" )
GAME( 1984, complexx, 0,        qix,     complexx, 0,        ROT270, "Taito America Corporation", "Complex X" )
